/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.manager.upgrade;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.accumulo.core.client.BatchDeleter;
import org.apache.accumulo.core.client.BatchWriter;
import org.apache.accumulo.core.client.MutationsRejectedException;
import org.apache.accumulo.core.client.Scanner;
import org.apache.accumulo.core.client.TableNotFoundException;
import org.apache.accumulo.core.data.InstanceId;
import org.apache.accumulo.core.data.Key;
import org.apache.accumulo.core.data.Mutation;
import org.apache.accumulo.core.data.Range;
import org.apache.accumulo.core.data.TableId;
import org.apache.accumulo.core.fate.zookeeper.ZooReaderWriter;
import org.apache.accumulo.core.fate.zookeeper.ZooUtil;
import org.apache.accumulo.core.manager.state.tables.TableState;
import org.apache.accumulo.core.metadata.MetadataTable;
import org.apache.accumulo.core.metadata.schema.MetadataSchema;
import org.apache.accumulo.core.security.Authorizations;
import org.apache.accumulo.manager.upgrade.Upgrader;
import org.apache.accumulo.server.ServerContext;
import org.apache.accumulo.server.conf.codec.VersionedProperties;
import org.apache.accumulo.server.conf.store.PropStore;
import org.apache.accumulo.server.conf.store.PropStoreKey;
import org.apache.accumulo.server.conf.store.TablePropKey;
import org.apache.accumulo.server.util.MetadataTableUtil;
import org.apache.hadoop.io.Text;
import org.apache.zookeeper.KeeperException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Upgrader10to11
implements Upgrader {
    private static final Logger log = LoggerFactory.getLogger(Upgrader10to11.class);
    private static final TableId REPLICATION_ID = TableId.of((String)"+rep");
    private static final Range REP_TABLE_RANGE = new Range((CharSequence)(REPLICATION_ID.canonical() + ";"), true, (CharSequence)(REPLICATION_ID.canonical() + "<"), true);
    private static final Range REP_WAL_RANGE = new Range((CharSequence)"~repl", true, (CharSequence)"~repm", false);

    @Override
    public void upgradeZookeeper(ServerContext context) {
        log.info("upgrade of ZooKeeper entries");
        ZooReaderWriter zrw = context.getZooReaderWriter();
        InstanceId iid = context.getInstanceID();
        if (!this.checkReplicationTableInZk(iid, zrw)) {
            log.debug("replication table root node does not exist in ZooKeeper - nothing to do");
            return;
        }
        if (!this.checkReplicationOffline(iid, zrw)) {
            throw new IllegalStateException("Replication table is not offline. Cannot continue with upgrade that will remove replication with replication active");
        }
        this.cleanMetaConfig(iid, context.getPropStore());
        this.deleteReplicationTableZkEntries(zrw, iid);
    }

    @Override
    public void upgradeRoot(ServerContext context) {
        log.info("upgrade root - skipping, nothing to do");
    }

    @Override
    public void upgradeMetadata(ServerContext context) {
        log.info("upgrade metadata entries");
        List<String> replTableFiles = this.readReplFilesFromMetadata(context);
        this.deleteReplMetadataEntries(context);
        this.deleteReplTableFiles(context, replTableFiles);
    }

    List<String> readReplFilesFromMetadata(ServerContext context) {
        ArrayList<String> results = new ArrayList<String>();
        try (Scanner scanner = context.createScanner(MetadataTable.NAME, Authorizations.EMPTY);){
            scanner.fetchColumnFamily(MetadataSchema.TabletsSection.DataFileColumnFamily.NAME);
            scanner.setRange(REP_TABLE_RANGE);
            for (Map.Entry entry : scanner) {
                String f = ((Key)entry.getKey()).getColumnQualifier(MetadataSchema.TabletsSection.DataFileColumnFamily.NAME).toString();
                results.add(f);
            }
        }
        catch (TableNotFoundException ex) {
            throw new IllegalStateException("failed to read replication files from metadata", ex);
        }
        return results;
    }

    void deleteReplTableFiles(ServerContext context, List<String> replTableFiles) {
        if (replTableFiles.isEmpty()) {
            return;
        }
        boolean haveFailures = false;
        try (BatchWriter writer = context.createBatchWriter(MetadataTable.NAME);){
            for (String filename : replTableFiles) {
                Mutation m = this.createDelMutation(filename);
                log.debug("Adding delete marker for file: {}", (Object)filename);
                writer.addMutation(m);
            }
        }
        catch (MutationsRejectedException ex) {
            log.debug("Failed to write delete marker {}", (Object)ex.getMessage());
            haveFailures = true;
        }
        catch (TableNotFoundException ex) {
            throw new IllegalStateException("failed to read replication files from metadata", ex);
        }
        if (haveFailures) {
            throw new IllegalStateException("deletes rejected adding deletion marker for replication file entries, check log");
        }
    }

    private Mutation createDelMutation(String path) {
        Mutation delFlag = new Mutation(new Text(MetadataSchema.DeletesSection.encodeRow((String)path)));
        delFlag.put(MetadataTableUtil.EMPTY_TEXT, MetadataTableUtil.EMPTY_TEXT, MetadataSchema.DeletesSection.SkewedKeyValue.NAME);
        return delFlag;
    }

    private void deleteReplMetadataEntries(ServerContext context) {
        try (BatchDeleter deleter = context.createBatchDeleter(MetadataTable.NAME, Authorizations.EMPTY, 10);){
            deleter.setRanges(List.of(REP_TABLE_RANGE, REP_WAL_RANGE));
            deleter.delete();
        }
        catch (MutationsRejectedException | TableNotFoundException ex) {
            throw new IllegalStateException("failed to remove replication info from metadata table", ex);
        }
    }

    private boolean checkReplicationTableInZk(InstanceId iid, ZooReaderWriter zrw) {
        try {
            String path = Upgrader10to11.buildRepTablePath(iid);
            return zrw.exists(path);
        }
        catch (KeeperException ex) {
            throw new IllegalStateException("ZooKeeper error - cannot determine replication table status", ex);
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new IllegalStateException("interrupted reading replication state from ZooKeeper", ex);
        }
    }

    private boolean checkReplicationOffline(InstanceId iid, ZooReaderWriter zrw) {
        try {
            String path = Upgrader10to11.buildRepTablePath(iid) + "/state";
            byte[] bytes = zrw.getData(path);
            if (bytes != null && bytes.length > 0) {
                String status = new String(bytes, StandardCharsets.UTF_8);
                return TableState.OFFLINE.name().equals(status);
            }
            return false;
        }
        catch (KeeperException ex) {
            throw new IllegalStateException("ZooKeeper error - cannot determine replication table status", ex);
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new IllegalStateException("interrupted reading replication state from ZooKeeper", ex);
        }
    }

    static String buildRepTablePath(InstanceId iid) {
        return ZooUtil.getRoot((InstanceId)iid) + "/tables/" + REPLICATION_ID.canonical();
    }

    private void deleteReplicationTableZkEntries(ZooReaderWriter zrw, InstanceId iid) {
        String repTablePath = Upgrader10to11.buildRepTablePath(iid);
        try {
            zrw.recursiveDelete(repTablePath, ZooUtil.NodeMissingPolicy.SKIP);
        }
        catch (KeeperException ex) {
            throw new IllegalStateException("ZooKeeper error - failed recursive deletion on " + repTablePath, ex);
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new IllegalStateException("interrupted deleting " + repTablePath + " from ZooKeeper", ex);
        }
    }

    private void cleanMetaConfig(InstanceId iid, PropStore propStore) {
        TablePropKey metaKey = TablePropKey.of((InstanceId)iid, (TableId)MetadataTable.ID);
        VersionedProperties p = propStore.get((PropStoreKey)metaKey);
        Map props = p.asMap();
        List<String> filtered = this.filterReplConfigKeys(props.keySet());
        String v = (String)props.get("table.formatter");
        if (v != null && v.compareTo("org.apache.accumulo.server.replication.StatusFormatter") == 0) {
            filtered.add("table.formatter");
        }
        if (filtered.size() > 0) {
            log.trace("Upgrade filtering replication iterators for id: {}", (Object)metaKey);
            propStore.removeProperties((PropStoreKey)metaKey, filtered);
        }
    }

    private List<String> filterReplConfigKeys(Set<String> keys) {
        String REPL_ITERATOR_PATTERN = "^table\\.iterator\\.(majc|minc|scan)\\.replcombiner$";
        String REPL_COLUMN_PATTERN = "^table\\.iterator\\.(majc|minc|scan)\\.replcombiner\\.opt\\.columns$";
        Pattern p = Pattern.compile("(" + REPL_ITERATOR_PATTERN + "|" + REPL_COLUMN_PATTERN + ")");
        return keys.stream().filter(e -> p.matcher((CharSequence)e).find()).collect(Collectors.toList());
    }
}

