/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.ambari.server.audit.request.eventcreator;

import java.util.Set;

import org.apache.ambari.server.api.services.Request;
import org.apache.ambari.server.api.services.Result;
import org.apache.ambari.server.api.services.ResultStatus;
import org.apache.ambari.server.audit.event.AuditEvent;
import org.apache.ambari.server.audit.event.request.AddRequestRequestAuditEvent;
import org.apache.ambari.server.controller.internal.RequestOperationLevel;
import org.apache.ambari.server.controller.spi.Resource;
import java.util.Map;
import org.apache.ambari.server.controller.internal.RequestResourceProvider;
import com.google.common.collect.ImmutableSet;

/**
 * This creator handles request type requests
 * For resource type {@link Resource.Type#Request}
 * and request types {@link Request.Type#POST}
 */
public class RequestEventCreator implements RequestAuditEventCreator {

  /**
   * Set of {@link Request.Type}s that are handled by this plugin
   */
  private Set<Request.Type> requestTypes = ImmutableSet.<Request.Type>builder().add(Request.Type.POST).build();

  /**
   * Set of {@link Resource.Type}s that are handled by this plugin
   */
  private Set<Resource.Type> resourceTypes = ImmutableSet.<Resource.Type>builder().add(Resource.Type.Request).build();

  /**
   * {@inheritDoc}
   */
  @Override
  public Set<Request.Type> getRequestTypes() {
    return requestTypes;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Set<Resource.Type> getResourceTypes() {
    return resourceTypes;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Set<ResultStatus.STATUS> getResultStatuses() {
    // null makes this default
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public AuditEvent createAuditEvent(Request request, Result result) {

    switch (request.getRequestType()) {
      case POST:
        return AddRequestRequestAuditEvent.builder()
          .withTimestamp(System.currentTimeMillis())
          .withRequestType(request.getRequestType())
          .withResultStatus(result.getStatus())
          .withUrl(request.getURI())
          .withRemoteIp(request.getRemoteAddress())
          .withCommand(request.getBody().getRequestInfoProperties().get("command"))
          .withClusterName(getClusterName(request, RequestOperationLevel.OPERATION_CLUSTER_ID))
	  .build();
      default:
        return null;
    }
  }
  /**
   *Returns clusterName from the request based on the propertyName parameter
   *@param request
   *@param propertyName
   *@return
   */
  private String getClusterName(Request request ,String propertyName) {
    Map<String, String> requestInfoProps = request.getBody().getRequestInfoProperties();
    return requestInfoProps.containsKey(propertyName)?requestInfoProps.get(propertyName):getProperty(request, RequestResourceProvider.REQUEST_CLUSTER_NAME_PROPERTY_ID);
  }

  /**
   *Returns property from the request based on the propertyName parameter
   *@param request
   *@param propertyName
   *@return
   */
  private String getProperty(Request request, String propertyName) {
    if (!request.getBody().getPropertySets().isEmpty()) {
      return String.valueOf(request.getBody().getPropertySets().iterator().next().get(propertyName));
    }
    return null;
  }
}
