////////////////////////////////////////////////////////////////////////////////
//
//  Licensed to the Apache Software Foundation (ASF) under one or more
//  contributor license agreements.  See the NOTICE file distributed with
//  this work for additional information regarding copyright ownership.
//  The ASF licenses this file to You under the Apache License, Version 2.0
//  (the "License"); you may not use this file except in compliance with
//  the License.  You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.
//
////////////////////////////////////////////////////////////////////////////////

package mx.olap
{

/**
 *  The OLAPTrace class controls the generation of trace information 
 *  generated by the OLAP classes and written to the log file. 
 *  This class uses the <code>trace()</code> method internally to write out the log information.
 *
 *  <p>You must set <code>TraceOutputFileEnable=1</code> in mm.cfg, and use the 
 *  Debug Flash Player or AIR Debug Launcher to generate trace output.</p> 
 *
 *  @see trace()
 *  
 *  @langversion 3.0
 *  @playerversion Flash 9
 *  @playerversion AIR 1.1
 *  @productversion Flex 3
 */
public class OLAPTrace
{
    include "../core/Version.as";
    
    //--------------------------------------------------------------------------
    //
    //  Class constants
    //
    //--------------------------------------------------------------------------
    
    /**
     *  Set to <code>true</code> to enable trace output.
     *
     *  @default true
     *  
     *  @langversion 3.0
     *  @playerversion Flash 9
     *  @playerversion AIR 1.1
     *  @productversion Flex 3
     */
    public static var traceOn:Boolean = true;
    
    /**
     *  The current trace level, which determines the amount of trace information
     *  written to the log file, <code>TRACE_LEVEL_1</code> writes the least amount of 
     *  information to the log file, and <code>TRACE_LEVEL_3</code> writes the most.
     *
     *  <p>All trace information output by a call to the <code>traceMsg()</code> method 
     *  with a <code>level</code> argument less than <code>traceLevel</code> 
     *  is sent to the log file.</p>
     *
     *  @default TRACE_LEVEL_1
     *  
     *  @langversion 3.0
     *  @playerversion Flash 9
     *  @playerversion AIR 1.1
     *  @productversion Flex 3
     */
    public static var traceLevel:int = 1;

    /**
     *  Specifies to write minimal trace information to the log file. 
     *  
     *  @langversion 3.0
     *  @playerversion Flash 9
     *  @playerversion AIR 1.1
     *  @productversion Flex 3
     */
    public static const TRACE_LEVEL_1:int = 1;

    /**
     *  Specifies to write more trace information to the log file than <code>TRACE_LEVEL_1</code>. 
     *  
     *  @langversion 3.0
     *  @playerversion Flash 9
     *  @playerversion AIR 1.1
     *  @productversion Flex 3
     */
    public static const TRACE_LEVEL_2:int = 2;

    /**
     *  Specifies to write the most trace information to the log file.
     *  
     *  @langversion 3.0
     *  @playerversion Flash 9
     *  @playerversion AIR 1.1
     *  @productversion Flex 3
     */
    public static const TRACE_LEVEL_3:int = 3;
    
    //--------------------------------------------------------------------------
    //
    //  Class methods
    //
    //--------------------------------------------------------------------------
    
    /**
     *  Writes trace information to the log file
     *.
     *  <p>You must set <code>TraceOutputFileEnable=1</code> in mm.cfg, and use the 
     *  Debug Flash Player or AIR Debug Launcher to generate 
     *  trace output by calling this method.</p> 
     *
     *  @param msg The trace message.
     *
     *  @param level The trace level of the message.
     *  Only trace messages with a <code>level</code> argument less than <code>traceLevel</code> 
     *  are sent to the log file.
     *  
     *  @langversion 3.0
     *  @playerversion Flash 9
     *  @playerversion AIR 1.1
     *  @productversion Flex 3
     */
    public static function traceMsg(msg:String, level:int=1):void
    {
        if (traceOn && level <= traceLevel)
            trace(msg);
    }
}

}