////////////////////////////////////////////////////////////////////////////////
//
//  Licensed to the Apache Software Foundation (ASF) under one or more
//  contributor license agreements.  See the NOTICE file distributed with
//  this work for additional information regarding copyright ownership.
//  The ASF licenses this file to You under the Apache License, Version 2.0
//  (the "License"); you may not use this file except in compliance with
//  the License.  You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.
//
////////////////////////////////////////////////////////////////////////////////

package mx.messaging.events
{

import flash.events.Event;
import mx.messaging.messages.IMessage;

[ExcludeClass]

/**
 *  Represents events that are dispatched as a result of invoking operations
 *  on a MessagePersister.
 *
 *  @see mx.messaging.MessagePersister
 * @private
 */
public class MessagePersisterEvent extends Event
{
    /**
     *  Normally called internally and not used in application code. This event is
     *  generated to report a successful result for a load() invocation or succes or fault
     *  for invoking operations on a <code>MessagePersister</code>.
     *
     *  @see mx.messaging.MessagePersister
     *
     *  @param The event type of the event.
     * 
     *  @param bubbles Specifies whether the event can bubble up the display 
     *  list hierarchy.
     * 
     *  @param cancelable Indicates whether the behavior associated with the 
     *  event can be prevented.
     * 
     *  @param The Id for the message agent that invoked the operation.
     *
     *  @param The operation that the message agent invoked.
     *  
     *  @langversion 3.0
     *  @playerversion Flash 9
     *  @playerversion AIR 1.1
     *  @productversion BlazeDS 4
     *  @productversion LCDS 3 
     */
    public function MessagePersisterEvent(type:String, bubbles:Boolean = false, cancelable:Boolean = false, 
            id:String = null, operation:String = null)
    {
        super(type, bubbles, cancelable);

        _id = id;
        _op = operation;
    }
    
    /**
     *  The Id for the message agent that invoked the operation.
     *  
     *  @langversion 3.0
     *  @playerversion Flash 9
     *  @playerversion AIR 1.1
     *  @productversion BlazeDS 4
     *  @productversion LCDS 3 
     */
    public function get id():String
    {
        return _id;
    }
    
    /**
     *  The operation that the message agent invoked.
     *  
     *  @langversion 3.0
     *  @playerversion Flash 9
     *  @playerversion AIR 1.1
     *  @productversion BlazeDS 4
     *  @productversion LCDS 3 
     */
    public function get operation():String
    {
        return _op;
    }

    /**
     *  The messages associated with an event having status <code>RESULT</code>.
     *  
     *  @langversion 3.0
     *  @playerversion Flash 9
     *  @playerversion AIR 1.1
     *  @productversion BlazeDS 4
     *  @productversion LCDS 3 
     */
    public function set messages(value:Array):void
    {
        _messages = value;
    }

    public function get messages():Array
    {
        return _messages;
    }
    
    /**
     *  The source message that was passed to <code>save</code>
     *  that was successfully saved or faulted.
     *  
     *  @langversion 3.0
     *  @playerversion Flash 9
     *  @playerversion AIR 1.1
     *  @productversion BlazeDS 4
     *  @productversion LCDS 3 
     */
    public function set message(value:IMessage):void
    {
        _message = message;
    }
    
    public function get message():IMessage
    {
        return _message;
    }   

    /**
     * @private
     */
    public function get messageId():String
    {
        if (_message != null)
        {
            return _message.messageId;
        }
        return null;
    }

    /**
     * @private
     */
    public function get messageCount():int
    {
        if (_messages != null)
        {
            return _messages.count;
        }
        return 0;
    }
    
    public static function createEvent(type:String, id:String, operation:String):MessagePersisterEvent
    {
        return new MessagePersisterEvent(type, false, false, id, operation);
    }

    /**
     *  @private
     */
    override public function clone():Event
    {
        var cloneEvent:MessagePersisterEvent = new MessagePersisterEvent(type, bubbles, cancelable, _id, _op);
        if (_messages)
        {
            cloneEvent.messages = _messages;
        }
        return cloneEvent;
    }

    override public function toString():String
    {
        return formatToString("MessagePersisterEvent", "id", "operation", "messageId", "messageCount", "type", "bubbles", "cancelable", "eventPhase");
    }

    // Status constants.
    /**
     *  The <code>SUCCESS</code> status indicates that an operation invoked on a
     *  message persister completed without errors.
     *  
     *  @langversion 3.0
     *  @playerversion Flash 9
     *  @playerversion AIR 1.1
     *  @productversion BlazeDS 4
     *  @productversion LCDS 3 
     */
    public static const SUCCESS:String = "success";

    /**
     *  The <code>RESULT</code> status indicates that a <code>load</code> operation 
     *  invoked on a message persister is returning an array of stored messages.
     *  
     *  @langversion 3.0
     *  @playerversion Flash 9
     *  @playerversion AIR 1.1
     *  @productversion BlazeDS 4
     *  @productversion LCDS 3 
     */
    public static const RESULT:String = "result";
    
    /**
     *  The <code>FAULT</code> status indicates that an operation invoked on a 
     *  message persister failed.
     *  
     *  @langversion 3.0
     *  @playerversion Flash 9
     *  @playerversion AIR 1.1
     *  @productversion BlazeDS 4
     *  @productversion LCDS 3 
     */
    public static const FAULT:String = "fault";

    // private members
    private var _id:String;
    private var _op:String;
    private var _messages:Array;
    private var _message:IMessage;
}

}
