/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.connector.jdbc.source.enumerator;

import org.apache.flink.annotation.PublicEvolving;
import org.apache.flink.connector.jdbc.source.split.JdbcSourceSplit;

import javax.annotation.Nullable;

import java.io.IOException;
import java.io.Serializable;
import java.util.List;

/**
 * Base class for jdbc sql split enumerator.
 *
 * @param <SplitT> JDBC split type.
 */
@PublicEvolving
public abstract class JdbcSqlSplitEnumeratorBase<SplitT> implements AutoCloseable, Serializable {
    private final char[] currentId = "0000000000".toCharArray();

    protected @Nullable Serializable optionalSqlSplitEnumeratorState;

    public JdbcSqlSplitEnumeratorBase(@Nullable Serializable optionalSqlSplitEnumeratorState) {
        this.optionalSqlSplitEnumeratorState = optionalSqlSplitEnumeratorState;
    }

    protected final String getNextId() {
        // because we just increment numbers, we increment the char representation directly,
        // rather than incrementing an integer and converting it to a string representation
        // every time again (requires quite some expensive conversion logic).
        incrementCharArrayByOne(currentId, currentId.length - 1);
        return new String(currentId);
    }

    private static void incrementCharArrayByOne(char[] array, int pos) {
        char c = array[pos];
        c++;

        if (c > '9') {
            c = '0';
            incrementCharArrayByOne(array, pos - 1);
        }
        array[pos] = c;
    }

    /**
     * Enumerate the JDBC splits.
     *
     * @return The result splits generated by the split enumerator.
     * @throws IOException IO exception.
     */
    public abstract List<JdbcSourceSplit> enumerateSplits() throws IOException;

    /**
     * A provider to create or restore a JDBC sql splits enumerator.
     *
     * @param <SplitT> Split type.
     */
    @PublicEvolving
    public interface Provider<SplitT> extends Serializable {

        /**
         * Called when init the provider without state.
         *
         * @return An instance of {@link JdbcSqlSplitEnumeratorBase}.
         */
        JdbcSqlSplitEnumeratorBase<SplitT> create();

        /**
         * Called when restore the provider without state.
         *
         * @param optionalSqlSplitEnumeratorState The state defined by users.
         * @return An instance of {@link JdbcSqlSplitEnumeratorBase}.
         */
        JdbcSqlSplitEnumeratorBase<SplitT> restore(
                @Nullable Serializable optionalSqlSplitEnumeratorState);
    }

    public void open() {}

    public void close() {}
}
