/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.flume.channel.file;

import com.google.common.collect.Maps;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import static org.apache.flume.channel.file.TestUtils.compareInputAndOut;
import static org.apache.flume.channel.file.TestUtils.takeEvents;

public class TestFileChannelFormatRegression extends TestFileChannelBase {
  protected static final Logger LOG = LoggerFactory
      .getLogger(TestFileChannelFormatRegression.class);

  @Before
  public void setup() throws Exception {
    super.setup();
  }

  @After
  public void teardown() {
    super.teardown();
  }
  /**
   * This is regression test with files generated by a file channel
   * with the FLUME-1432 patch.
   */
  @Test
  public void testFileFormatV2postFLUME1432()
          throws Exception {
    TestUtils.copyDecompressed("fileformat-v2-checkpoint.gz",
            new File(checkpointDir, "checkpoint"));
    for (int i = 0; i < dataDirs.length; i++) {
      int fileIndex = i + 1;
      TestUtils.copyDecompressed("fileformat-v2-log-" + fileIndex + ".gz",
                                 new File(dataDirs[i], "log-" + fileIndex));
    }
    Map<String, String> overrides = Maps.newHashMap();
    overrides.put(FileChannelConfiguration.CAPACITY, String.valueOf(10));
    overrides.put(FileChannelConfiguration.TRANSACTION_CAPACITY,
        String.valueOf(10));
    channel = createFileChannel(overrides);
    channel.start();
    Assert.assertTrue(channel.isOpen());
    Set<String> events = takeEvents(channel, 1);
    Set<String> expected = new HashSet<String>();
    expected.addAll(Arrays.asList(
            (new String[]{
              "2684", "2685", "2686", "2687", "2688", "2689", "2690", "2691"
            })));
    compareInputAndOut(expected, events);

  }
  /**
   * This is a regression test with files generated by a file channel
   * without the FLUME-1432 patch.
   */
  @Test
  public void testFileFormatV2PreFLUME1432LogReplayV1()
          throws Exception {
    doTestFileFormatV2PreFLUME1432(true);
  }
  @Test
  public void testFileFormatV2PreFLUME1432LogReplayV2()
          throws Exception {
    doTestFileFormatV2PreFLUME1432(false);
  }
  public void doTestFileFormatV2PreFLUME1432(boolean useLogReplayV1)
          throws Exception {
    TestUtils.copyDecompressed("fileformat-v2-pre-FLUME-1432-checkpoint.gz",
            new File(checkpointDir, "checkpoint"));
    for (int i = 0; i < dataDirs.length; i++) {
      int fileIndex = i + 1;
      TestUtils.copyDecompressed("fileformat-v2-pre-FLUME-1432-log-" + fileIndex
          + ".gz", new File(dataDirs[i], "log-" + fileIndex));
    }
    Map<String, String> overrides = Maps.newHashMap();
    overrides.put(FileChannelConfiguration.CAPACITY, String.valueOf(10000));
    overrides.put(FileChannelConfiguration.USE_LOG_REPLAY_V1,
            String.valueOf(useLogReplayV1));
    channel = createFileChannel(overrides);
    channel.start();
    Assert.assertTrue(channel.isOpen());
    Set<String> events = takeEvents(channel, 1);
    Assert.assertEquals(50, events.size());
  }
}
