/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.datanode;

import com.google.common.annotations.VisibleForTesting;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.commons.lang3.time.FastDateFormat;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.StorageType;
import org.apache.hadoop.hdfs.server.datanode.DataNode;
import org.apache.hadoop.hdfs.server.datanode.ReplicaInfo;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.FsDatasetSpi;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.FsVolumeSpi;
import org.apache.hadoop.util.Daemon;
import org.apache.hadoop.util.StopWatch;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class DirectoryScanner
implements Runnable {
    private static final Logger LOG = LoggerFactory.getLogger(DirectoryScanner.class);
    private static final int MILLIS_PER_SECOND = 1000;
    private static final String START_MESSAGE = "Periodic Directory Tree Verification scan starting at %s with interval of %dms";
    private static final String START_MESSAGE_WITH_THROTTLE = "Periodic Directory Tree Verification scan starting at %s with interval of %dms and throttle limit of %dms/s";
    private static final int RECONCILE_BLOCKS_BATCH_SIZE = 1000;
    private final FsDatasetSpi<?> dataset;
    private final ExecutorService reportCompileThreadPool;
    private final ScheduledExecutorService masterThread;
    private final long scanPeriodMsecs;
    private final int throttleLimitMsPerSec;
    private volatile boolean shouldRun = false;
    private boolean retainDiffs = false;
    private final DataNode datanode;
    @VisibleForTesting
    final AtomicLong timeRunningMs = new AtomicLong(0L);
    @VisibleForTesting
    final AtomicLong timeWaitingMs = new AtomicLong(0L);
    @VisibleForTesting
    final ScanInfoPerBlockPool diffs = new ScanInfoPerBlockPool();
    @VisibleForTesting
    final Map<String, Stats> stats = new HashMap<String, Stats>();

    @VisibleForTesting
    public void setRetainDiffs(boolean b) {
        this.retainDiffs = b;
    }

    public DirectoryScanner(DataNode datanode, FsDatasetSpi<?> dataset, Configuration conf) {
        this.datanode = datanode;
        this.dataset = dataset;
        int interval = (int)conf.getTimeDuration("dfs.datanode.directoryscan.interval", 21600L, TimeUnit.SECONDS);
        this.scanPeriodMsecs = interval * 1000;
        int throttle = conf.getInt("dfs.datanode.directoryscan.throttle.limit.ms.per.sec", 1000);
        if (throttle > 1000 || throttle <= 0) {
            if (throttle > 1000) {
                LOG.error("dfs.datanode.directoryscan.throttle.limit.ms.per.sec set to value above 1000 ms/sec. Assuming default value of 1000");
            } else {
                LOG.error("dfs.datanode.directoryscan.throttle.limit.ms.per.sec set to value below 1 ms/sec. Assuming default value of 1000");
            }
            this.throttleLimitMsPerSec = 1000;
        } else {
            this.throttleLimitMsPerSec = throttle;
        }
        int threads = conf.getInt("dfs.datanode.directoryscan.threads", 1);
        this.reportCompileThreadPool = Executors.newFixedThreadPool(threads, (ThreadFactory)new Daemon.DaemonFactory());
        this.masterThread = new ScheduledThreadPoolExecutor(1, (ThreadFactory)new Daemon.DaemonFactory());
    }

    void start() {
        this.shouldRun = true;
        long offset = ThreadLocalRandom.current().nextInt((int)(this.scanPeriodMsecs / 1000L)) * 1000;
        long firstScanTime = Time.now() + offset;
        String logMsg = this.throttleLimitMsPerSec < 1000 ? String.format(START_MESSAGE_WITH_THROTTLE, FastDateFormat.getInstance().format(firstScanTime), this.scanPeriodMsecs, this.throttleLimitMsPerSec) : String.format(START_MESSAGE, FastDateFormat.getInstance().format(firstScanTime), this.scanPeriodMsecs);
        LOG.info(logMsg);
        this.masterThread.scheduleAtFixedRate(this, offset, this.scanPeriodMsecs, TimeUnit.MILLISECONDS);
    }

    @VisibleForTesting
    boolean getRunStatus() {
        return this.shouldRun;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void clear() {
        ScanInfoPerBlockPool scanInfoPerBlockPool = this.diffs;
        synchronized (scanInfoPerBlockPool) {
            this.diffs.clear();
        }
        this.stats.clear();
    }

    @Override
    public void run() {
        try {
            if (!this.shouldRun) {
                LOG.warn("this cycle terminating immediately because 'shouldRun' has been deactivated");
                return;
            }
            this.reconcile();
        }
        catch (Exception e) {
            LOG.error("Exception during DirectoryScanner execution - will continue next cycle", (Throwable)e);
        }
        catch (Error er) {
            LOG.error("System Error during DirectoryScanner execution - permanently terminating periodic scanner", (Throwable)er);
            throw er;
        }
    }

    void shutdown() {
        if (!this.shouldRun) {
            LOG.warn("DirectoryScanner: shutdown has been called, but periodic scanner not started");
        } else {
            LOG.warn("DirectoryScanner: shutdown has been called");
        }
        this.shouldRun = false;
        if (this.masterThread != null) {
            this.masterThread.shutdown();
        }
        if (this.reportCompileThreadPool != null) {
            this.reportCompileThreadPool.shutdownNow();
        }
        if (this.masterThread != null) {
            try {
                this.masterThread.awaitTermination(1L, TimeUnit.MINUTES);
            }
            catch (InterruptedException e) {
                LOG.error("interrupted while waiting for masterThread to terminate", (Throwable)e);
            }
        }
        if (this.reportCompileThreadPool != null) {
            try {
                this.reportCompileThreadPool.awaitTermination(1L, TimeUnit.MINUTES);
            }
            catch (InterruptedException e) {
                LOG.error("interrupted while waiting for reportCompileThreadPool to terminate", (Throwable)e);
            }
        }
        if (!this.retainDiffs) {
            this.clear();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    public void reconcile() throws IOException {
        LOG.debug("reconcile start DirectoryScanning");
        this.scan();
        int loopCount = 0;
        ScanInfoPerBlockPool scanInfoPerBlockPool = this.diffs;
        synchronized (scanInfoPerBlockPool) {
            for (Map.Entry entry : this.diffs.entrySet()) {
                String bpid = (String)entry.getKey();
                LinkedList diff = (LinkedList)entry.getValue();
                for (FsVolumeSpi.ScanInfo info : diff) {
                    this.dataset.checkAndUpdate(bpid, info);
                    if (loopCount % 1000 == 0) {
                        try {
                            Thread.sleep(2000L);
                        }
                        catch (InterruptedException interruptedException) {
                            // empty catch block
                        }
                    }
                    ++loopCount;
                }
            }
        }
        if (!this.retainDiffs) {
            this.clear();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void scan() {
        this.clear();
        Map<String, FsVolumeSpi.ScanInfo[]> diskReport = this.getDiskReport();
        for (Map.Entry<String, FsVolumeSpi.ScanInfo[]> entry : diskReport.entrySet()) {
            String bpid = entry.getKey();
            FsVolumeSpi.ScanInfo[] blockpoolReport = entry.getValue();
            Stats statsRecord = new Stats(bpid);
            this.stats.put(bpid, statsRecord);
            LinkedList<FsVolumeSpi.ScanInfo> diffRecord = new LinkedList<FsVolumeSpi.ScanInfo>();
            ScanInfoPerBlockPool scanInfoPerBlockPool = this.diffs;
            synchronized (scanInfoPerBlockPool) {
                this.diffs.put(bpid, diffRecord);
            }
            statsRecord.totalBlocks = blockpoolReport.length;
            List<ReplicaInfo> bl = this.dataset.getFinalizedBlocks(bpid);
            Collections.sort(bl);
            int d = 0;
            int m = 0;
            while (m < bl.size() && d < blockpoolReport.length) {
                ReplicaInfo memBlock = bl.get(m);
                FsVolumeSpi.ScanInfo info = blockpoolReport[d];
                if (info.getBlockId() < memBlock.getBlockId()) {
                    if (!this.dataset.isDeletingBlock(bpid, info.getBlockId())) {
                        ++statsRecord.missingMemoryBlocks;
                        this.addDifference(diffRecord, statsRecord, info);
                    }
                    ++d;
                    continue;
                }
                if (info.getBlockId() > memBlock.getBlockId()) {
                    this.addDifference(diffRecord, statsRecord, memBlock.getBlockId(), info.getVolume());
                    ++m;
                    continue;
                }
                if (info.getBlockFile() == null) {
                    this.addDifference(diffRecord, statsRecord, info);
                } else if (info.getGenStamp() != memBlock.getGenerationStamp() || info.getBlockLength() != memBlock.getNumBytes()) {
                    ++statsRecord.mismatchBlocks;
                    this.addDifference(diffRecord, statsRecord, info);
                } else if (memBlock.compareWith(info) != 0) {
                    ++statsRecord.duplicateBlocks;
                    this.addDifference(diffRecord, statsRecord, info);
                }
                if (++d < blockpoolReport.length) {
                    FsVolumeSpi.ScanInfo nextInfo = blockpoolReport[d];
                    if (nextInfo.getBlockId() == info.getBlockId()) continue;
                    ++m;
                    continue;
                }
                ++m;
            }
            while (m < bl.size()) {
                ReplicaInfo current = bl.get(m++);
                this.addDifference(diffRecord, statsRecord, current.getBlockId(), current.getVolume());
            }
            while (d < blockpoolReport.length) {
                if (!this.dataset.isDeletingBlock(bpid, blockpoolReport[d].getBlockId())) {
                    ++statsRecord.missingMemoryBlocks;
                    this.addDifference(diffRecord, statsRecord, blockpoolReport[d]);
                }
                ++d;
            }
            LOG.info(statsRecord.toString());
        }
    }

    private void addDifference(LinkedList<FsVolumeSpi.ScanInfo> diffRecord, Stats statsRecord, FsVolumeSpi.ScanInfo info) {
        statsRecord.missingMetaFile = statsRecord.missingMetaFile + (info.getMetaFile() == null ? 1L : 0L);
        statsRecord.missingBlockFile = statsRecord.missingBlockFile + (info.getBlockFile() == null ? 1L : 0L);
        diffRecord.add(info);
    }

    private void addDifference(LinkedList<FsVolumeSpi.ScanInfo> diffRecord, Stats statsRecord, long blockId, FsVolumeSpi vol) {
        ++statsRecord.missingBlockFile;
        ++statsRecord.missingMetaFile;
        diffRecord.add(new FsVolumeSpi.ScanInfo(blockId, null, null, null, vol));
    }

    @VisibleForTesting
    public Map<String, FsVolumeSpi.ScanInfo[]> getDiskReport() {
        ScanInfoPerBlockPool list = new ScanInfoPerBlockPool();
        ScanInfoPerBlockPool[] dirReports = null;
        try (FsDatasetSpi.FsVolumeReferences volumes = this.dataset.getFsVolumeReferences();){
            dirReports = new ScanInfoPerBlockPool[volumes.size()];
            HashMap<Integer, Future<ScanInfoPerBlockPool>> compilersInProgress = new HashMap<Integer, Future<ScanInfoPerBlockPool>>();
            for (int i = 0; i < volumes.size(); ++i) {
                if (volumes.get(i).getStorageType() == StorageType.PROVIDED) continue;
                ReportCompiler reportCompiler = new ReportCompiler(this.datanode, volumes.get(i));
                Future<ScanInfoPerBlockPool> result = this.reportCompileThreadPool.submit(reportCompiler);
                compilersInProgress.put(i, result);
            }
            for (Map.Entry report : compilersInProgress.entrySet()) {
                Integer index = (Integer)report.getKey();
                try {
                    dirReports[index.intValue()] = (ScanInfoPerBlockPool)((Future)report.getValue()).get();
                    if (dirReports[index] != null) continue;
                    dirReports = null;
                    break;
                }
                catch (Exception ex) {
                    FsVolumeSpi fsVolumeSpi = volumes.get(index);
                    LOG.error("Error compiling report for the volume, StorageId: " + fsVolumeSpi.getStorageID(), (Throwable)ex);
                }
            }
        }
        catch (IOException e) {
            LOG.error("Unexpected IOException by closing FsVolumeReference", (Throwable)e);
        }
        if (dirReports != null) {
            for (ScanInfoPerBlockPool report : dirReports) {
                if (report == null) continue;
                list.addAll(report);
            }
        }
        return list.toSortedArrays();
    }

    public static enum BlockDirFilter implements FilenameFilter
    {
        INSTANCE;


        @Override
        public boolean accept(File dir, String name) {
            return name.startsWith("subdir") || name.startsWith("finalized") || name.startsWith("blk_");
        }
    }

    public class ReportCompiler
    implements Callable<ScanInfoPerBlockPool> {
        private final FsVolumeSpi volume;
        private final DataNode datanode;
        private final StopWatch throttleTimer = new StopWatch();
        private final StopWatch perfTimer = new StopWatch();

        public ReportCompiler(DataNode datanode, FsVolumeSpi volume) {
            this.datanode = datanode;
            this.volume = volume;
        }

        @Override
        public ScanInfoPerBlockPool call() throws IOException {
            String[] bpList = this.volume.getBlockPoolList();
            ScanInfoPerBlockPool result = new ScanInfoPerBlockPool(bpList.length);
            this.perfTimer.start();
            this.throttleTimer.start();
            for (String bpid : bpList) {
                LinkedList<FsVolumeSpi.ScanInfo> report = new LinkedList<FsVolumeSpi.ScanInfo>();
                this.perfTimer.reset().start();
                this.throttleTimer.reset().start();
                try {
                    result.put(bpid, this.volume.compileReport(bpid, report, this));
                }
                catch (InterruptedException ex) {
                    result = null;
                    break;
                }
            }
            return result;
        }

        public void throttle() throws InterruptedException {
            this.accumulateTimeRunning();
            if (DirectoryScanner.this.throttleLimitMsPerSec < 1000 && this.throttleTimer.now(TimeUnit.MILLISECONDS) > (long)DirectoryScanner.this.throttleLimitMsPerSec) {
                Thread.sleep(1000 - DirectoryScanner.this.throttleLimitMsPerSec);
                this.throttleTimer.reset().start();
            }
            this.accumulateTimeWaiting();
        }

        private void accumulateTimeRunning() {
            DirectoryScanner.this.timeRunningMs.getAndAdd(this.perfTimer.now(TimeUnit.MILLISECONDS));
            this.perfTimer.reset().start();
        }

        private void accumulateTimeWaiting() {
            DirectoryScanner.this.timeWaitingMs.getAndAdd(this.perfTimer.now(TimeUnit.MILLISECONDS));
            this.perfTimer.reset().start();
        }
    }

    static class ScanInfoPerBlockPool
    extends HashMap<String, LinkedList<FsVolumeSpi.ScanInfo>> {
        private static final long serialVersionUID = 1L;

        ScanInfoPerBlockPool() {
        }

        ScanInfoPerBlockPool(int sz) {
            super(sz);
        }

        public void addAll(ScanInfoPerBlockPool that) {
            if (that == null) {
                return;
            }
            for (Map.Entry entry : that.entrySet()) {
                String bpid = (String)entry.getKey();
                LinkedList list = (LinkedList)entry.getValue();
                if (this.containsKey(bpid)) {
                    ((LinkedList)this.get(bpid)).addAll(list);
                    continue;
                }
                this.put(bpid, list);
            }
        }

        public Map<String, FsVolumeSpi.ScanInfo[]> toSortedArrays() {
            HashMap<String, FsVolumeSpi.ScanInfo[]> result = new HashMap<String, FsVolumeSpi.ScanInfo[]>(this.size());
            for (Map.Entry entry : this.entrySet()) {
                String bpid = (String)entry.getKey();
                LinkedList list = (LinkedList)entry.getValue();
                Object[] record = list.toArray(new FsVolumeSpi.ScanInfo[list.size()]);
                Arrays.sort(record);
                result.put(bpid, (FsVolumeSpi.ScanInfo[])record);
            }
            return result;
        }
    }

    @VisibleForTesting
    static class Stats {
        final String bpid;
        long totalBlocks = 0L;
        long missingMetaFile = 0L;
        long missingBlockFile = 0L;
        long missingMemoryBlocks = 0L;
        long mismatchBlocks = 0L;
        long duplicateBlocks = 0L;

        public Stats(String bpid) {
            this.bpid = bpid;
        }

        public String toString() {
            return "BlockPool " + this.bpid + " Total blocks: " + this.totalBlocks + ", missing metadata files:" + this.missingMetaFile + ", missing block files:" + this.missingBlockFile + ", missing blocks in memory:" + this.missingMemoryBlocks + ", mismatched blocks:" + this.mismatchBlocks;
        }
    }
}

