/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.fluss.lake.paimon.source;

import org.apache.fluss.config.Configuration;
import org.apache.fluss.lake.paimon.PaimonLakeStorage;
import org.apache.fluss.metadata.TablePath;
import org.apache.fluss.utils.CloseableIterator;

import org.apache.flink.streaming.api.environment.StreamExecutionEnvironment;
import org.apache.flink.table.api.EnvironmentSettings;
import org.apache.flink.table.api.bridge.java.StreamTableEnvironment;
import org.apache.flink.types.Row;
import org.apache.paimon.catalog.Catalog;
import org.apache.paimon.catalog.CatalogContext;
import org.apache.paimon.catalog.CatalogFactory;
import org.apache.paimon.data.InternalRow;
import org.apache.paimon.options.Options;
import org.apache.paimon.schema.Schema;
import org.apache.paimon.table.Table;
import org.apache.paimon.table.sink.BatchTableCommit;
import org.apache.paimon.table.sink.BatchTableWrite;
import org.apache.paimon.table.sink.BatchWriteBuilder;
import org.apache.paimon.table.sink.CommitMessage;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.io.TempDir;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;

import static org.apache.fluss.lake.paimon.utils.PaimonConversions.toPaimon;

/** Base class for paimon lakehouse test. */
class PaimonSourceTestBase {
    protected static final String DEFAULT_DB = "fluss_lakehouse";
    protected static final String DEFAULT_TABLE = "test_lakehouse_table";
    protected static final int DEFAULT_BUCKET_NUM = 1;

    private static @TempDir File tempWarehouseDir;
    protected static PaimonLakeStorage lakeStorage;
    protected static Catalog paimonCatalog;

    @BeforeAll
    protected static void beforeAll() {
        Configuration configuration = new Configuration();
        configuration.setString("type", "paimon");
        configuration.setString("warehouse", tempWarehouseDir.toString());
        configuration.setString("user", "root");
        configuration.setString("password", "root-password");
        lakeStorage = new PaimonLakeStorage(configuration);
        paimonCatalog =
                CatalogFactory.createCatalog(
                        CatalogContext.create(Options.fromMap(configuration.toMap())));
        initPaimonPrivilege();
    }

    // Test for paimon privilege table
    public static void initPaimonPrivilege() {
        StreamTableEnvironment streamTEnv =
                StreamTableEnvironment.create(
                        StreamExecutionEnvironment.getExecutionEnvironment(),
                        EnvironmentSettings.inStreamingMode());
        streamTEnv.executeSql(
                String.format(
                        "create catalog %s with ('type'='paimon', 'warehouse' = '%s')",
                        "paimon_catalog", tempWarehouseDir));
        streamTEnv.executeSql(
                "CALL paimon_catalog.sys.init_file_based_privilege('root-password');");
    }

    public void createTable(TablePath tablePath, Schema schema) throws Exception {
        paimonCatalog.createDatabase(tablePath.getDatabaseName(), true);
        paimonCatalog.createTable(toPaimon(tablePath), schema, true);
    }

    public void writeRecord(TablePath tablePath, List<InternalRow> records) throws Exception {
        Table table = getTable(tablePath);
        BatchWriteBuilder writeBuilder = table.newBatchWriteBuilder();
        try (BatchTableWrite writer = writeBuilder.newWrite()) {
            for (InternalRow record : records) {
                writer.write(record);
            }
            List<CommitMessage> messages = writer.prepareCommit();
            try (BatchTableCommit commit = writeBuilder.newCommit()) {
                commit.commit(messages);
            }
        }
    }

    public Table getTable(TablePath tablePath) throws Exception {
        return paimonCatalog.getTable(toPaimon(tablePath));
    }

    public static List<Row> convertToFlinkRow(
            org.apache.fluss.row.InternalRow.FieldGetter[] fieldGetters,
            CloseableIterator<org.apache.fluss.row.InternalRow> flussRowIterator) {
        List<Row> rows = new ArrayList<>();
        while (flussRowIterator.hasNext()) {
            org.apache.fluss.row.InternalRow row = flussRowIterator.next();
            Row flinkRow = new Row(fieldGetters.length);
            for (int i = 0; i < fieldGetters.length; i++) {
                flinkRow.setField(i, fieldGetters[i].getFieldOrNull(row));
            }
            rows.add(flinkRow);
        }
        return rows;
    }

    /** Adapter for transforming closeable iterator. */
    public static class TransformingCloseableIterator<T, U> implements CloseableIterator<U> {
        private final CloseableIterator<T> source;
        private final Function<? super T, ? extends U> transformer;

        public TransformingCloseableIterator(
                CloseableIterator<T> source, Function<? super T, ? extends U> transformer) {
            this.source = source;
            this.transformer = transformer;
        }

        @Override
        public boolean hasNext() {
            return source.hasNext();
        }

        @Override
        public U next() {
            return transformer.apply(source.next());
        }

        @Override
        public void close() {
            source.close();
        }

        public static <T, U> CloseableIterator<U> transform(
                CloseableIterator<T> source, Function<? super T, ? extends U> transformer) {
            return new TransformingCloseableIterator<>(source, transformer);
        }
    }
}
