/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.myfaces.orchestra.frameworkAdapter.springBasic;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.myfaces.orchestra.frameworkAdapter.basic.BasicFrameworkAdapter;
import org.apache.myfaces.orchestra.frameworkAdapter.basic.BasicFrameworkAdapterFilter;
import org.apache.myfaces.orchestra.lib._UrlMatcher;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import java.io.IOException;

/**
 * Configures the SpringBasicFrameworkAdapter.
 * <p>
 * Orchestra accesses information about the request, response, session, etc via a
 * FrameworkAdapter so that it can be used with multiple web tier frameworks. This
 * class selects and configures a version of this adapter which looks up variables
 * via Spring.
 * <p>
 * If filter config parameter "conversationMessagerClass" is set, then this is
 * passed to the SpringBasicFrameworkAdapter, meaning that this can be either a
 * beanname defined in the dependency-injection framework, or an absolute
 * classname of a type implementing ConversationMessager.
 * <p>
 * This filter access the Spring framework to lookup the bean in its
 *  {@link org.apache.myfaces.orchestra.frameworkAdapter.FrameworkAdapter#getBean} method
 */
public class SpringBasicFrameworkAdapterFilter implements Filter
{
    private final static String INIT_CONVERSATION_MESSAGER = "conversationMessagerClass";

    private final Log log = LogFactory.getLog(BasicFrameworkAdapterFilter.class);
    private BasicFrameworkAdapter adapter;
    private _UrlMatcher urlMatcher;

    public void init(FilterConfig filterConfig) throws ServletException
    {
        String conversationMessager = filterConfig.getInitParameter(INIT_CONVERSATION_MESSAGER);

         adapter = new SpringBasicFrameworkAdapter(filterConfig.getServletContext(), conversationMessager);
         urlMatcher = new _UrlMatcher(filterConfig);
    }

    public void doFilter(ServletRequest req, ServletResponse rsp, FilterChain filterChain)
    throws IOException, ServletException
    {
        if (!urlMatcher.accept(req))
        {
            filterChain.doFilter(req, rsp);
            return;
        }

        log.debug("doFilter");
        try
        {
            adapter.beginRequest(req, rsp);
            filterChain.doFilter(req, rsp);
        }
        finally
        {
            adapter.endRequest();
        }
    }

    public void destroy()
    {
    }
}