/*
 * Decompiled with CFR 0.152.
 */
package io.airlift.discovery.store;

import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.base.Supplier;
import com.google.common.collect.Iterables;
import io.airlift.concurrent.Threads;
import io.airlift.discovery.store.Entry;
import io.airlift.discovery.store.LocalStore;
import io.airlift.discovery.store.RemoteStore;
import io.airlift.discovery.store.StoreConfig;
import io.airlift.discovery.store.Version;
import io.airlift.units.Duration;
import java.util.Arrays;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.inject.Inject;
import org.joda.time.DateTime;
import org.weakref.jmx.Managed;

public class DistributedStore {
    private final String name;
    private final LocalStore localStore;
    private final RemoteStore remoteStore;
    private final Supplier<DateTime> timeSupplier;
    private final Duration tombstoneMaxAge;
    private final Duration garbageCollectionInterval;
    private final ScheduledExecutorService garbageCollector;
    private final AtomicLong lastGcTimestamp = new AtomicLong();

    @Inject
    public DistributedStore(String name, LocalStore localStore, RemoteStore remoteStore, StoreConfig config, Supplier<DateTime> timeSupplier) {
        this.name = (String)Preconditions.checkNotNull((Object)name, (Object)"name is null");
        this.localStore = (LocalStore)Preconditions.checkNotNull((Object)localStore, (Object)"localStore is null");
        this.remoteStore = (RemoteStore)Preconditions.checkNotNull((Object)remoteStore, (Object)"remoteStore is null");
        this.timeSupplier = (Supplier)Preconditions.checkNotNull(timeSupplier, (Object)"timeSupplier is null");
        Preconditions.checkNotNull((Object)config, (Object)"config is null");
        this.tombstoneMaxAge = config.getTombstoneMaxAge();
        this.garbageCollectionInterval = config.getGarbageCollectionInterval();
        this.garbageCollector = Executors.newSingleThreadScheduledExecutor(Threads.daemonThreadsNamed((String)("distributed-store-gc-" + name)));
    }

    @PostConstruct
    public void start() {
        this.garbageCollector.scheduleAtFixedRate(new Runnable(){

            @Override
            public void run() {
                DistributedStore.this.removeExpiredEntries();
            }
        }, 0L, this.garbageCollectionInterval.toMillis(), TimeUnit.MILLISECONDS);
    }

    @Managed
    public String getName() {
        return this.name;
    }

    @Managed
    public long getLastGcTimestamp() {
        return this.lastGcTimestamp.get();
    }

    @Managed
    public void removeExpiredEntries() {
        for (Entry entry : this.localStore.getAll()) {
            if (!this.isExpired(entry)) continue;
            this.localStore.delete(entry.getKey(), entry.getVersion());
        }
        this.lastGcTimestamp.set(System.currentTimeMillis());
    }

    private boolean isExpired(Entry entry) {
        long ageInMs = ((DateTime)this.timeSupplier.get()).getMillis() - entry.getTimestamp();
        return entry.getValue() == null && ageInMs > this.tombstoneMaxAge.toMillis() || entry.getMaxAgeInMs() != null && ageInMs > entry.getMaxAgeInMs();
    }

    @PreDestroy
    public void shutdown() {
        this.garbageCollector.shutdownNow();
    }

    public void put(byte[] key, byte[] value) {
        Preconditions.checkNotNull((Object)key, (Object)"key is null");
        Preconditions.checkNotNull((Object)value, (Object)"value is null");
        long now = ((DateTime)this.timeSupplier.get()).getMillis();
        Entry entry = new Entry(key, value, new Version(now), now, null);
        this.localStore.put(entry);
        this.remoteStore.put(entry);
    }

    public void put(byte[] key, byte[] value, Duration maxAge) {
        Preconditions.checkNotNull((Object)key, (Object)"key is null");
        Preconditions.checkNotNull((Object)value, (Object)"value is null");
        Preconditions.checkNotNull((Object)maxAge, (Object)"maxAge is null");
        long now = ((DateTime)this.timeSupplier.get()).getMillis();
        Entry entry = new Entry(key, value, new Version(now), now, maxAge.toMillis());
        this.localStore.put(entry);
        this.remoteStore.put(entry);
    }

    public byte[] get(byte[] key) {
        Preconditions.checkNotNull((Object)key, (Object)"key is null");
        Entry entry = this.localStore.get(key);
        byte[] result = null;
        if (entry != null && entry.getValue() != null && !this.isExpired(entry)) {
            result = Arrays.copyOf(entry.getValue(), entry.getValue().length);
        }
        return result;
    }

    public void delete(byte[] key) {
        Preconditions.checkNotNull((Object)key, (Object)"key is null");
        long now = ((DateTime)this.timeSupplier.get()).getMillis();
        Entry entry = new Entry(key, null, new Version(now), now, null);
        this.localStore.put(entry);
        this.remoteStore.put(entry);
    }

    public Iterable<Entry> getAll() {
        return Iterables.filter(this.localStore.getAll(), (Predicate)Predicates.and((Predicate)Predicates.not(this.expired()), (Predicate)Predicates.not(DistributedStore.tombstone())));
    }

    private Predicate<? super Entry> expired() {
        return new Predicate<Entry>(){

            public boolean apply(Entry entry) {
                return DistributedStore.this.isExpired(entry);
            }
        };
    }

    private static Predicate<? super Entry> tombstone() {
        return new Predicate<Entry>(){

            public boolean apply(Entry entry) {
                return entry.getValue() == null;
            }
        };
    }
}

