/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.jackrabbit.oak.spi.commit;

import org.apache.jackrabbit.oak.api.CommitFailedException;
import org.apache.jackrabbit.oak.api.PropertyState;
import org.apache.jackrabbit.oak.spi.state.NodeState;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.List;

/**
 * Aggregation of a list of editors into a single editor.
 */
public class CompositeEditor implements Editor {

    @Nullable
    public static Editor compose(@NotNull List<? extends Editor> editors) {
        switch (editors.size()) {
            case 0:
                return null;
            case 1:
                return editors.get(0);
            default:
                return new CompositeEditor(editors);
        }
    }

    private final List<? extends Editor> editors;

    public CompositeEditor(List<? extends Editor> editors) {
        this.editors = editors;
    }

    public CompositeEditor(Editor... editors) {
        this(List.of(editors));
    }

    @Override
    public void enter(NodeState before, NodeState after)
            throws CommitFailedException {
        for (Editor editor : editors) {
            editor.enter(before, after);
        }
    }

    @Override
    public void leave(NodeState before, NodeState after)
            throws CommitFailedException {
        for (Editor editor : editors) {
            editor.leave(before, after);
        }
    }

    @Override
    public void propertyAdded(PropertyState after)
            throws CommitFailedException {
        // Performance critical code, avoid creating an iterator object
        for (int i = 0; i < editors.size(); i++) {
            editors.get(i).propertyAdded(after);
        }
    }

    @Override
    public void propertyChanged(PropertyState before, PropertyState after)
            throws CommitFailedException {
        // Performance critical code, avoid creating an iterator object
        for (int i = 0; i < editors.size(); i++) {
            editors.get(i).propertyChanged(before, after);
        }
    }

    @Override
    public void propertyDeleted(PropertyState before)
            throws CommitFailedException {
        // Performance critical code, avoid creating an iterator object
        for (int i = 0; i < editors.size(); i++) {
            editors.get(i).propertyDeleted(before);
        }
    }

    @Override
    public Editor childNodeAdded(String name, NodeState after)
            throws CommitFailedException {
        List<Editor> list = new ArrayList<>(editors.size());
        // Performance critical code, avoid creating an iterator object
        for (int i = 0; i < editors.size(); i++) {
            Editor child = editors.get(i).childNodeAdded(name, after);
            if (child != null) {
                list.add(child);
            }
        }
        return compose(list);
    }

    @Override
    public Editor childNodeChanged(String name, NodeState before, NodeState after)
            throws CommitFailedException {
        List<Editor> list = new ArrayList<>(editors.size());
        // Performance critical code, avoid creating an iterator object
        for (int i = 0; i < editors.size(); i++) {
            Editor child = editors.get(i).childNodeChanged(name, before, after);
            if (child != null) {
                list.add(child);
            }
        }
        return compose(list);
    }

    @Override
    public Editor childNodeDeleted(String name, NodeState before)
            throws CommitFailedException {
        List<Editor> list = new ArrayList<>(editors.size());
        // Performance critical code, avoid creating an iterator object
        for (int i = 0; i < editors.size(); i++) {
            Editor child = editors.get(i).childNodeDeleted(name, before);
            if (child != null) {
                list.add(child);
            }
        }
        return compose(list);
    }
}
