# Copyright (C) 2024 Intel Corporation
# SPDX-License-Identifier: MIT

import subprocess
import sys
from typing import Union


class CommandException(Exception):
    def __init__(self, cmd, exit_code, out, err):
        self.__cmd = cmd if type(cmd) is list else [cmd]
        self.__exit_code = exit_code
        self.__out = out
        self.__err = err

    def __str__(self):
        return f'Command {" ".join(self.__cmd)} returned with exit code {self.__exit_code}\n' \
               f'Output stream: {self.__out}\n' \
               f'Error stream: {self.__err}'

    @property
    def exit_code(self):
        return self.__exit_code

    @property
    def stderr(self):
        return self.__err

    @property
    def stdout(self):
        return self.__out

class Command:

    @staticmethod
    def run(cmd: list, never_echo: bool  = False, use_platform_extension: bool = False, **kv) -> str:
        if use_platform_extension and sys.platform == 'win32':
            cmd = [cmd[0] + '.cmd'] + cmd[1:]
        try:
            return Command.check_output(cmd, never_echo, **kv)
        except CommandException:
            raise

    @staticmethod
    def run_with_real_time_output(cmd: list, use_platform_extension: bool = False, **kv) -> str:
        if use_platform_extension and sys.platform == 'win32':
            cmd = [cmd[0] + '.cmd'] + cmd[1:]
        try:
            return Command.check_output_real_time(cmd, **kv)
        except CommandException:
            raise

    @staticmethod
    def call(cmd: list, **kv):
        return subprocess.call(cmd, **kv)

    @staticmethod
    def check_output(cmd: Union[list, str], never_echo: bool = False, **kv) -> str:
        startup_info = Command.hide_subprocess_window()
        subproc = subprocess.Popen(cmd, startupinfo=startup_info, stdout=subprocess.PIPE, stderr=subprocess.PIPE, **kv)
        out, err = subproc.communicate()
        out_str = out.decode('utf-8', errors='replace') if isinstance(out, bytes) else out
        err_str = err.decode('utf-8', errors='replace') if isinstance(err, bytes) else err
        if not never_echo:
            print(f'{out_str}\n{err_str}')
        if subproc.returncode != 0:
            raise CommandException(cmd, subproc.returncode, out_str, err_str)
        return out_str

    @staticmethod
    def hide_subprocess_window():
        startup_info = None
        if sys.platform == 'win32':
            startup_info = subprocess.STARTUPINFO()
            startup_info.dwFlags |= subprocess.STARTF_USESHOWWINDOW
        return startup_info

    @staticmethod
    def check_output_real_time(cmd: Union[list, str], **kv) -> str:
        startup_info = Command.hide_subprocess_window()
        with subprocess.Popen(cmd, startupinfo=startup_info, stdout=subprocess.PIPE, stderr=subprocess.PIPE, text=True, **kv) as subproc:
            out = Command.handle_real_time_output(subproc)
            err = Command.handle_real_time_error_output(subproc)
            if subproc.returncode != 0:
                raise CommandException(cmd, subproc.returncode, ''.join(out), ''.join(err))
            return ''.join(out)

    @staticmethod
    def handle_real_time_error_output(subproc):
        result = []
        err_output = subproc.stderr.read()
        Command.conditional_echo(err_output, result)
        return result

    @staticmethod
    def handle_real_time_output(subproc):
        result = []
        while True:
            output = subproc.stdout.readline()
            if output == '' and subproc.poll() is not None:
                break
            Command.conditional_echo(output, result)
        return result

    @staticmethod
    def conditional_echo(stream_data, result, never_echo=False):
        if stream_data:
            result.append(stream_data)
            if not never_echo:
                print(stream_data.rstrip())

