﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/dataexchange/DataExchangeRequest.h>
#include <aws/dataexchange/DataExchange_EXPORTS.h>

#include <utility>

namespace Aws {
namespace DataExchange {
namespace Model {

/**
 */
class CreateRevisionRequest : public DataExchangeRequest {
 public:
  AWS_DATAEXCHANGE_API CreateRevisionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateRevision"; }

  AWS_DATAEXCHANGE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>An optional comment about the revision.</p>
   */
  inline const Aws::String& GetComment() const { return m_comment; }
  inline bool CommentHasBeenSet() const { return m_commentHasBeenSet; }
  template <typename CommentT = Aws::String>
  void SetComment(CommentT&& value) {
    m_commentHasBeenSet = true;
    m_comment = std::forward<CommentT>(value);
  }
  template <typename CommentT = Aws::String>
  CreateRevisionRequest& WithComment(CommentT&& value) {
    SetComment(std::forward<CommentT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique identifier for a data set.</p>
   */
  inline const Aws::String& GetDataSetId() const { return m_dataSetId; }
  inline bool DataSetIdHasBeenSet() const { return m_dataSetIdHasBeenSet; }
  template <typename DataSetIdT = Aws::String>
  void SetDataSetId(DataSetIdT&& value) {
    m_dataSetIdHasBeenSet = true;
    m_dataSetId = std::forward<DataSetIdT>(value);
  }
  template <typename DataSetIdT = Aws::String>
  CreateRevisionRequest& WithDataSetId(DataSetIdT&& value) {
    SetDataSetId(std::forward<DataSetIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A revision tag is an optional label that you can assign to a revision when
   * you create it. Each tag consists of a key and an optional value, both of which
   * you define. When you use tagging, you can also use tag-based access control in
   * IAM policies to control access to these data sets and revisions.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateRevisionRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateRevisionRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_comment;
  bool m_commentHasBeenSet = false;

  Aws::String m_dataSetId;
  bool m_dataSetIdHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace DataExchange
}  // namespace Aws
