﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/qbusiness/QBusinessRequest.h>
#include <aws/qbusiness/QBusiness_EXPORTS.h>
#include <aws/qbusiness/model/UserAlias.h>

#include <utility>

namespace Aws {
namespace QBusiness {
namespace Model {

/**
 */
class CreateUserRequest : public QBusinessRequest {
 public:
  AWS_QBUSINESS_API CreateUserRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateUser"; }

  AWS_QBUSINESS_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The identifier of the application for which the user mapping will be
   * created.</p>
   */
  inline const Aws::String& GetApplicationId() const { return m_applicationId; }
  inline bool ApplicationIdHasBeenSet() const { return m_applicationIdHasBeenSet; }
  template <typename ApplicationIdT = Aws::String>
  void SetApplicationId(ApplicationIdT&& value) {
    m_applicationIdHasBeenSet = true;
    m_applicationId = std::forward<ApplicationIdT>(value);
  }
  template <typename ApplicationIdT = Aws::String>
  CreateUserRequest& WithApplicationId(ApplicationIdT&& value) {
    SetApplicationId(std::forward<ApplicationIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The user emails attached to a user mapping.</p>
   */
  inline const Aws::String& GetUserId() const { return m_userId; }
  inline bool UserIdHasBeenSet() const { return m_userIdHasBeenSet; }
  template <typename UserIdT = Aws::String>
  void SetUserId(UserIdT&& value) {
    m_userIdHasBeenSet = true;
    m_userId = std::forward<UserIdT>(value);
  }
  template <typename UserIdT = Aws::String>
  CreateUserRequest& WithUserId(UserIdT&& value) {
    SetUserId(std::forward<UserIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The list of user aliases in the mapping.</p>
   */
  inline const Aws::Vector<UserAlias>& GetUserAliases() const { return m_userAliases; }
  inline bool UserAliasesHasBeenSet() const { return m_userAliasesHasBeenSet; }
  template <typename UserAliasesT = Aws::Vector<UserAlias>>
  void SetUserAliases(UserAliasesT&& value) {
    m_userAliasesHasBeenSet = true;
    m_userAliases = std::forward<UserAliasesT>(value);
  }
  template <typename UserAliasesT = Aws::Vector<UserAlias>>
  CreateUserRequest& WithUserAliases(UserAliasesT&& value) {
    SetUserAliases(std::forward<UserAliasesT>(value));
    return *this;
  }
  template <typename UserAliasesT = UserAlias>
  CreateUserRequest& AddUserAliases(UserAliasesT&& value) {
    m_userAliasesHasBeenSet = true;
    m_userAliases.emplace_back(std::forward<UserAliasesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A token that you provide to identify the request to create your Amazon Q
   * Business user mapping.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateUserRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_applicationId;
  bool m_applicationIdHasBeenSet = false;

  Aws::String m_userId;
  bool m_userIdHasBeenSet = false;

  Aws::Vector<UserAlias> m_userAliases;
  bool m_userAliasesHasBeenSet = false;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace QBusiness
}  // namespace Aws
