﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/elasticfilesystem/EFSRequest.h>
#include <aws/elasticfilesystem/EFS_EXPORTS.h>
#include <aws/elasticfilesystem/model/IpAddressType.h>

#include <utility>

namespace Aws {
namespace EFS {
namespace Model {

/**
 * <p/><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/CreateMountTargetRequest">AWS
 * API Reference</a></p>
 */
class CreateMountTargetRequest : public EFSRequest {
 public:
  AWS_EFS_API CreateMountTargetRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateMountTarget"; }

  AWS_EFS_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The ID of the file system for which to create the mount target.</p>
   */
  inline const Aws::String& GetFileSystemId() const { return m_fileSystemId; }
  inline bool FileSystemIdHasBeenSet() const { return m_fileSystemIdHasBeenSet; }
  template <typename FileSystemIdT = Aws::String>
  void SetFileSystemId(FileSystemIdT&& value) {
    m_fileSystemIdHasBeenSet = true;
    m_fileSystemId = std::forward<FileSystemIdT>(value);
  }
  template <typename FileSystemIdT = Aws::String>
  CreateMountTargetRequest& WithFileSystemId(FileSystemIdT&& value) {
    SetFileSystemId(std::forward<FileSystemIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the subnet to add the mount target in. For One Zone file systems,
   * use the subnet that is associated with the file system's Availability Zone.</p>
   */
  inline const Aws::String& GetSubnetId() const { return m_subnetId; }
  inline bool SubnetIdHasBeenSet() const { return m_subnetIdHasBeenSet; }
  template <typename SubnetIdT = Aws::String>
  void SetSubnetId(SubnetIdT&& value) {
    m_subnetIdHasBeenSet = true;
    m_subnetId = std::forward<SubnetIdT>(value);
  }
  template <typename SubnetIdT = Aws::String>
  CreateMountTargetRequest& WithSubnetId(SubnetIdT&& value) {
    SetSubnetId(std::forward<SubnetIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If the IP address type for the mount target is IPv4, then specify the IPv4
   * address within the address range of the specified subnet.</p>
   */
  inline const Aws::String& GetIpAddress() const { return m_ipAddress; }
  inline bool IpAddressHasBeenSet() const { return m_ipAddressHasBeenSet; }
  template <typename IpAddressT = Aws::String>
  void SetIpAddress(IpAddressT&& value) {
    m_ipAddressHasBeenSet = true;
    m_ipAddress = std::forward<IpAddressT>(value);
  }
  template <typename IpAddressT = Aws::String>
  CreateMountTargetRequest& WithIpAddress(IpAddressT&& value) {
    SetIpAddress(std::forward<IpAddressT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If the IP address type for the mount target is IPv6, then specify the IPv6
   * address within the address range of the specified subnet.</p>
   */
  inline const Aws::String& GetIpv6Address() const { return m_ipv6Address; }
  inline bool Ipv6AddressHasBeenSet() const { return m_ipv6AddressHasBeenSet; }
  template <typename Ipv6AddressT = Aws::String>
  void SetIpv6Address(Ipv6AddressT&& value) {
    m_ipv6AddressHasBeenSet = true;
    m_ipv6Address = std::forward<Ipv6AddressT>(value);
  }
  template <typename Ipv6AddressT = Aws::String>
  CreateMountTargetRequest& WithIpv6Address(Ipv6AddressT&& value) {
    SetIpv6Address(std::forward<Ipv6AddressT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specify the type of IP address of the mount target you are creating. Options
   * are IPv4, dual stack, or IPv6. If you don’t specify an IpAddressType, then IPv4
   * is used.</p> <ul> <li> <p>IPV4_ONLY – Create mount target with IPv4 only subnet
   * or dual-stack subnet.</p> </li> <li> <p>DUAL_STACK – Create mount target with
   * dual-stack subnet.</p> </li> <li> <p>IPV6_ONLY – Create mount target with IPv6
   * only subnet.</p> </li> </ul>  <p>Creating IPv6 mount target only ENI in
   * dual-stack subnet is not supported.</p>
   */
  inline IpAddressType GetIpAddressType() const { return m_ipAddressType; }
  inline bool IpAddressTypeHasBeenSet() const { return m_ipAddressTypeHasBeenSet; }
  inline void SetIpAddressType(IpAddressType value) {
    m_ipAddressTypeHasBeenSet = true;
    m_ipAddressType = value;
  }
  inline CreateMountTargetRequest& WithIpAddressType(IpAddressType value) {
    SetIpAddressType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>VPC security group IDs, of the form <code>sg-xxxxxxxx</code>. These must be
   * for the same VPC as the subnet specified. The maximum number of security groups
   * depends on account quota. For more information, see <a
   * href="https://docs.aws.amazon.com/vpc/latest/userguide/amazon-vpc-limits.html">Amazon
   * VPC Quotas</a> in the <i>Amazon VPC User Guide</i> (see the <b>Security
   * Groups</b> table). </p>
   */
  inline const Aws::Vector<Aws::String>& GetSecurityGroups() const { return m_securityGroups; }
  inline bool SecurityGroupsHasBeenSet() const { return m_securityGroupsHasBeenSet; }
  template <typename SecurityGroupsT = Aws::Vector<Aws::String>>
  void SetSecurityGroups(SecurityGroupsT&& value) {
    m_securityGroupsHasBeenSet = true;
    m_securityGroups = std::forward<SecurityGroupsT>(value);
  }
  template <typename SecurityGroupsT = Aws::Vector<Aws::String>>
  CreateMountTargetRequest& WithSecurityGroups(SecurityGroupsT&& value) {
    SetSecurityGroups(std::forward<SecurityGroupsT>(value));
    return *this;
  }
  template <typename SecurityGroupsT = Aws::String>
  CreateMountTargetRequest& AddSecurityGroups(SecurityGroupsT&& value) {
    m_securityGroupsHasBeenSet = true;
    m_securityGroups.emplace_back(std::forward<SecurityGroupsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_fileSystemId;

  Aws::String m_subnetId;

  Aws::String m_ipAddress;

  Aws::String m_ipv6Address;

  IpAddressType m_ipAddressType{IpAddressType::NOT_SET};

  Aws::Vector<Aws::String> m_securityGroups;
  bool m_fileSystemIdHasBeenSet = false;
  bool m_subnetIdHasBeenSet = false;
  bool m_ipAddressHasBeenSet = false;
  bool m_ipv6AddressHasBeenSet = false;
  bool m_ipAddressTypeHasBeenSet = false;
  bool m_securityGroupsHasBeenSet = false;
};

}  // namespace Model
}  // namespace EFS
}  // namespace Aws
