﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/rds/RDSRequest.h>
#include <aws/rds/RDS_EXPORTS.h>
#include <aws/rds/model/Filter.h>

#include <utility>

namespace Aws {
namespace RDS {
namespace Model {

/**
 */
class DescribeTenantDatabasesRequest : public RDSRequest {
 public:
  AWS_RDS_API DescribeTenantDatabasesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeTenantDatabases"; }

  AWS_RDS_API Aws::String SerializePayload() const override;

 protected:
  AWS_RDS_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The user-supplied DB instance identifier, which must match the identifier of
   * an existing instance owned by the Amazon Web Services account. This parameter
   * isn't case-sensitive.</p>
   */
  inline const Aws::String& GetDBInstanceIdentifier() const { return m_dBInstanceIdentifier; }
  inline bool DBInstanceIdentifierHasBeenSet() const { return m_dBInstanceIdentifierHasBeenSet; }
  template <typename DBInstanceIdentifierT = Aws::String>
  void SetDBInstanceIdentifier(DBInstanceIdentifierT&& value) {
    m_dBInstanceIdentifierHasBeenSet = true;
    m_dBInstanceIdentifier = std::forward<DBInstanceIdentifierT>(value);
  }
  template <typename DBInstanceIdentifierT = Aws::String>
  DescribeTenantDatabasesRequest& WithDBInstanceIdentifier(DBInstanceIdentifierT&& value) {
    SetDBInstanceIdentifier(std::forward<DBInstanceIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The user-supplied tenant database name, which must match the name of an
   * existing tenant database on the specified DB instance owned by your Amazon Web
   * Services account. This parameter isn’t case-sensitive.</p>
   */
  inline const Aws::String& GetTenantDBName() const { return m_tenantDBName; }
  inline bool TenantDBNameHasBeenSet() const { return m_tenantDBNameHasBeenSet; }
  template <typename TenantDBNameT = Aws::String>
  void SetTenantDBName(TenantDBNameT&& value) {
    m_tenantDBNameHasBeenSet = true;
    m_tenantDBName = std::forward<TenantDBNameT>(value);
  }
  template <typename TenantDBNameT = Aws::String>
  DescribeTenantDatabasesRequest& WithTenantDBName(TenantDBNameT&& value) {
    SetTenantDBName(std::forward<TenantDBNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A filter that specifies one or more database tenants to describe.</p>
   * <p>Supported filters:</p> <ul> <li> <p> <code>tenant-db-name</code> - Tenant
   * database names. The results list only includes information about the tenant
   * databases that match these tenant DB names.</p> </li> <li> <p>
   * <code>tenant-database-resource-id</code> - Tenant database resource
   * identifiers.</p> </li> <li> <p> <code>dbi-resource-id</code> - DB instance
   * resource identifiers. The results list only includes information about the
   * tenants contained within the DB instances identified by these resource
   * identifiers.</p> </li> </ul>
   */
  inline const Aws::Vector<Filter>& GetFilters() const { return m_filters; }
  inline bool FiltersHasBeenSet() const { return m_filtersHasBeenSet; }
  template <typename FiltersT = Aws::Vector<Filter>>
  void SetFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters = std::forward<FiltersT>(value);
  }
  template <typename FiltersT = Aws::Vector<Filter>>
  DescribeTenantDatabasesRequest& WithFilters(FiltersT&& value) {
    SetFilters(std::forward<FiltersT>(value));
    return *this;
  }
  template <typename FiltersT = Filter>
  DescribeTenantDatabasesRequest& AddFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters.emplace_back(std::forward<FiltersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional pagination token provided by a previous
   * <code>DescribeTenantDatabases</code> request. If this parameter is specified,
   * the response includes only records beyond the marker, up to the value specified
   * by <code>MaxRecords</code>.</p>
   */
  inline const Aws::String& GetMarker() const { return m_marker; }
  inline bool MarkerHasBeenSet() const { return m_markerHasBeenSet; }
  template <typename MarkerT = Aws::String>
  void SetMarker(MarkerT&& value) {
    m_markerHasBeenSet = true;
    m_marker = std::forward<MarkerT>(value);
  }
  template <typename MarkerT = Aws::String>
  DescribeTenantDatabasesRequest& WithMarker(MarkerT&& value) {
    SetMarker(std::forward<MarkerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of records to include in the response. If more records
   * exist than the specified <code>MaxRecords</code> value, a pagination token
   * called a marker is included in the response so that you can retrieve the
   * remaining results.</p>
   */
  inline int GetMaxRecords() const { return m_maxRecords; }
  inline bool MaxRecordsHasBeenSet() const { return m_maxRecordsHasBeenSet; }
  inline void SetMaxRecords(int value) {
    m_maxRecordsHasBeenSet = true;
    m_maxRecords = value;
  }
  inline DescribeTenantDatabasesRequest& WithMaxRecords(int value) {
    SetMaxRecords(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_dBInstanceIdentifier;

  Aws::String m_tenantDBName;

  Aws::Vector<Filter> m_filters;

  Aws::String m_marker;

  int m_maxRecords{0};
  bool m_dBInstanceIdentifierHasBeenSet = false;
  bool m_tenantDBNameHasBeenSet = false;
  bool m_filtersHasBeenSet = false;
  bool m_markerHasBeenSet = false;
  bool m_maxRecordsHasBeenSet = false;
};

}  // namespace Model
}  // namespace RDS
}  // namespace Aws
