=head1 NAME

Graph::Renderer - draw the graph onto a real plane

=cut


package Graph::Renderer;

use strict;
use Carp qw (croak);

use vars qw ($VERSION @ISA @EXPORT_OK);

# $Id: Renderer.pm,v 1.4 2006/02/11 17:11:39 pasky Exp $
$VERSION = 0.03;


=head1 SYNOPSIS

  my $graph = new Graph;
  ...

  use Graph::Layouter;
  Graph::Layouter::layout($graph);

  use Graph::Renderer;
  Graph::Renderer::render($graph, $img);

=cut


use base qw (Graph::Layouter);

require Exporter;
push @ISA, 'Exporter';

@EXPORT_OK = qw (render);


=head1 DESCRIPTION

This module provides an abstract class for drawing a given layouted graph
(created usually by C<Graph::Layouter>) through various image creation
interfaces (C<GD>, C<Imager>).

All the relevant layouting information used for nodes positioning is load from
the graph, as saved by C<Graph::Layouter> (see its documentation for the
relevant attributes description). In addition, relevant rendering information
can be load from the graph attributes as well, if necessary and available.

This module contains only the abstract class, you will probably want to get an
instance of some particular drawing tool interface instead;
C<Graph::Renderer::Imager> is bundled with this distribution. The general
interface for all the subclasses is described below, but be sure consult also
the particular class' documentation for remarks, special notes and specific
extensions.


=over 4

=cut


use Graph;


=item B<render()>

This subroutine is the only entry point of this module, taking a given graph
and rendering it accordingly to its attributes. The subroutine can be called in
several ways:

=over 4

=item I<Functional interface>

The subroutine can be called as a function (it is not automatically exported,
but you can import it on your own if you really want; see the synopsis above).
It takes two parameters, the C<Graph> class (or any descendant) instance and
instance of the appropriate image object (see subclasses documentation). It
will load the rendering information from graph's both global and per-vertex
attributes.

=item I<Class constructor interface>

TODO

=item I<Class method interface>

TODO

=back

The subroutine returns the image object if it all went well, undef otherwise.

The common rendering attributes not generated by the layouter are:

=over 4

=item I<Global attributes>

=over 4

=item B<renderer_vertex_font>

This should contain information about the font used for vertex titles; the
specific format depends on the subclass. If this information is not made
available by the caller, the given backend can try to figure it out but it
might well fail.

=back

=item I<Vertex attributes>

=over 4

=item B<renderer_vertex_font>

This is identical to the global B<renderer_vertex_font> attribute, but used
only for the single vertex.

=item B<renderer_vertex_title>

Title of the vertex, which is displayed nearby. The vertex number is displayed
if no title is attached. You can attach an empty string to the vertex if you
want no title shown at all.

=back

=back

=cut

sub render {
	my $graph = shift;

	croak "Graph::Renderer::render() called instead of something of a subclass!";
	$graph;
}


# Internal use only, for subclasses; get maximal weight in the graph.
sub _max_weight($) {
	my $graph = shift;
	my $max_weight = 0;

	my @edges = $graph->edges;
	foreach my $edge (@edges) {
		my $weight = $graph->get_edge_attribute(@$edge, 'weight');
		$weight ||= 1; # TODO : configurable
		$max_weight = $weight if $max_weight < $weight;
	}

	$max_weight;
}

# Internal use only, for subclasses; maps virtual coord to physical pane
sub _transpose_coord($$$$) {
	my ($virtual, $min, $max, $size) = @_;

	$max++ if $max == $min; # Division by zero protector

	$size * ($virtual - $min) / ($max - $min);
}



=back


=head1 SEE ALSO

C<Graph>, C<Graph::Layouter>


=head1 BUGS

The object-oriented interface is missing as well as some more universal render
calling interface (hash parameters). Also, some more rendering attributes
(ie. color settings, own dimensions) are missing.


=head1 COPYRIGHT

Copyright 2004 by Petr Baudis E<lt>pasky@ucw.czE<gt>.

This code is distributed under the same copyright terms as
Perl itself.


=head1 VERSION

Version 0.03

$Id: Renderer.pm,v 1.4 2006/02/11 17:11:39 pasky Exp $

=cut

1;
