#!/usr/bin/perl -w
#
#    DateFormat.t
#
#    Copyright (c) 2000 Brian Stell
#
#    This package is free software and is provided ``as is'' without
#    express or implied warranty. It may be used, redistributed and/or
#    modified under the terms of the Perl Artistic License
#    (see http://www.perl.com/perl/misc/Artistic.html)
#

use strict;
use Test;
BEGIN { plan tests => 16 }
use ExtUtils::testlib;

use utf8;
use ICU::DateFormat qw(:EStyle);
use ICU::Locale;

##
## Make sure we always use the same fallback locale
##
my $default_locale = new ICU::Locale('en_US');
ICU::Locale->setDefault($default_locale);

ok(check_getAvailableLocales(), 1);
ok(check_getCountry(), 1);
ok(check_getDisplayCountry(), 1);
ok(check_getDisplayLanguage(), 1);
ok(check_getDisplayName(), 1);
ok(check_getDisplayVariant(), 1);
ok(check_getISOCountries(), 1);
ok(check_getISOLanguages(), 1);
ok(check_getLanguage(), 1);
ok(check_getLCID(), 1);
ok(check_getName(), 1);
ok(check_getVariant(), 1);
ok(check_hashCode(), 1);
ok(check_new(), 1);
ok(check_setDebugLevel(), 1);
ok(check_setDefault(), 1);

########################################################################
#
# The subroutines and tests
#
# (in alphabetic order)
#
########################################################################

sub check_getAvailableLocales {
    my $result = 1;

    my @locales = ICU::Locale->getAvailableLocales();
    if (!scalar(@locales)) {
        $result = 0;
    }
    my $found_en_US = 0;
    foreach my $locale (@locales) {
        if ($locale->getName() eq 'en_US') {
	    $found_en_US = 1;
	    last;
	}
    }
    if (!$found_en_US) {
        $result = 0;
    }

    return($result);
}

sub check_getCountry {
    my $result = 1;
    if (   (!check_getCountry1('en_US', 'US'))
        || (!check_getCountry1('fr_FR', 'FR'))
        || (!check_getCountry1('de_DE', 'DE'))
        || (!check_getCountry1('nl_NL', 'NL'))
    ) {
        $result = 0;
    }
    return($result);
}

sub check_getCountry1 {
    my ($locale_str, $expected_result) = @_;

    my $result = 1;
    my $locale = new ICU::Locale($locale_str);
    if ($locale->getCountry() ne $expected_result) {
        print "$locale_str getCountry() = (".$locale->getCountry().") "
                                           . "expect ($expected_result)\n";
        $result = 0;
    }
    return($result);
}

sub check_getDisplayCountry {
    my $result = 1;
    if (   (!check_getDisplayCountry1('en_US', 'United States'))
        || (!check_getDisplayCountry1('fr_FR', 'France'))
        || (!check_getDisplayCountry1('de_DE', 'Germany'))
        || (!check_getDisplayCountry1('nl_NL', 'Netherlands'))
    ) {
        $result = 0;
    }
    return($result);
}

sub check_getDisplayCountry1 {
    my ($locale_str, $expected_result) = @_;

    my $result = 1;
    my $locale = new ICU::Locale($locale_str);
    if ($locale->getDisplayCountry() ne $expected_result) {
        print "$locale_str getDisplayCountry() = (".$locale->getDisplayCountry().") "
                                           . "expect ($expected_result)\n";
        $result = 0;
    }
    return($result);
}

sub check_getDisplayLanguage {
    my $result = 1;
    if (   (!check_getDisplayLanguage1('en_US', 'English'))
        || (!check_getDisplayLanguage1('fr_FR', 'French'))
        || (!check_getDisplayLanguage1('de_DE', 'German'))
        || (!check_getDisplayLanguage1('nl_NL', 'Dutch'))
    ) {
        $result = 0;
    }
    return($result);
}

sub check_getDisplayLanguage1 {
    my ($locale_str, $expected_result) = @_;

    my $result = 1;
    my $locale = new ICU::Locale($locale_str);
    if ($locale->getDisplayLanguage() ne $expected_result) {
        print "$locale_str getDisplayLanguage() = (".$locale->getDisplayLanguage().") "
                                           . "expect ($expected_result)\n";
        $result = 0;
    }
    return($result);
}

sub check_getDisplayName {
    my $result = 1;
    if (   (!check_getDisplayName1('en_US', 'English (United States)'))
        || (!check_getDisplayName1('fr_FR', 'French (France)'))
        || (!check_getDisplayName1('de_DE', 'German (Germany)'))
        || (!check_getDisplayName1('nl_NL', 'Dutch (Netherlands)'))
    ) {
        $result = 0;
    }
    return($result);
}

sub check_getDisplayName1 {
    my ($locale_str, $expected_result) = @_;

    my $result = 1;
    my $locale = new ICU::Locale($locale_str);
    if ($locale->getDisplayName() ne $expected_result) {
        print "$locale_str getDisplayName() = (".$locale->getDisplayName().") "
                                           . "expect ($expected_result)\n";
        $result = 0;
    }
    return($result);
}

sub check_getDisplayVariant {
    my $result = 1;
    if (   (!check_getDisplayVariant1('en_US', ''))
        || (!check_getDisplayVariant1('fr_FR', ''))
        || (!check_getDisplayVariant1('de_DE', ''))
        || (!check_getDisplayVariant1('nl_NL', ''))
    ) {
        $result = 0;
    }
    return($result);
}

sub check_getDisplayVariant1 {
    my ($locale_str, $expected_result) = @_;

    my $result = 1;
    my $locale = new ICU::Locale($locale_str);
    if ($locale->getDisplayVariant() ne $expected_result) {
        print "$locale_str getDisplayVariant() = (".$locale->getDisplayVariant().") "
                                           . "expect ($expected_result)\n";
        $result = 0;
    }
    return($result);
}

sub check_getISOCountries {
    my $result = 1;

    my @countries = ICU::Locale->getISOCountries();
    if (!scalar(@countries)) {
        $result = 0;
    }
    my $found_en_US = 0;
    foreach my $country (@countries) {
        if ($country eq 'US') {
	    $found_en_US = 1;
	    last;
	}
    }
    if (!$found_en_US) {
        $result = 0;
    }

    return($result);
}

sub check_getISOLanguages {
    my $result = 1;

    my @languages = ICU::Locale->getISOLanguages();
    if (!scalar(@languages)) {
        $result = 0;
    }
    my $found_en_US = 0;
    foreach my $language (@languages) {
        if ($language eq 'en') {
	    $found_en_US = 1;
	    last;
	}
    }
    if (!$found_en_US) {
        $result = 0;
    }

    return($result);
}

sub check_getLanguage {
    my $result = 1;
    if (   (!check_getLanguage1('en_US', 'en'))
        || (!check_getLanguage1('fr_FR', 'fr'))
        || (!check_getLanguage1('de_DE', 'de'))
        || (!check_getLanguage1('nl_NL', 'nl'))
    ) {
        $result = 0;
    }
    return($result);
}

sub check_getLanguage1 {
    my ($locale_str, $expected_result) = @_;

    my $result = 1;
    my $locale = new ICU::Locale($locale_str);
    if ($locale->getLanguage() ne $expected_result) {
        print "$locale_str getLanguage() = (".$locale->getLanguage().") "
                                           . "expect ($expected_result)\n";
        $result = 0;
    }
    return($result);
}

sub check_getLCID {
    my $result = 1;
    if (   (!check_getLCID1('en_US', '1033'))
        || (!check_getLCID1('fr_FR', '1036'))
        || (!check_getLCID1('de_DE', '1031'))
        || (!check_getLCID1('nl_NL', '1043'))
    ) {
        $result = 0;
    }
    return($result);
}

sub check_getLCID1 {
    my ($locale_str, $expected_result) = @_;

    my $result = 1;
    my $locale = new ICU::Locale($locale_str);
    if ($locale->getLCID() ne $expected_result) {
        print "$locale_str getLCID() = (".$locale->getLCID().") "
                                           . "expect ($expected_result)\n";
        $result = 0;
    }
    return($result);
}

sub check_getName {
    my $result = 1;
    if (   (!check_getName1('en_US', 'en_US'))
        || (!check_getName1('fr_FR', 'fr_FR'))
        || (!check_getName1('de_DE', 'de_DE'))
        || (!check_getName1('nl_NL', 'nl_NL'))
    ) {
        $result = 0;
    }
    return($result);
}

sub check_getName1 {
    my ($locale_str, $expected_result) = @_;

    my $result = 1;
    my $locale = new ICU::Locale($locale_str);
    if ($locale->getName() ne $expected_result) {
        print "$locale_str getName() = (".$locale->getName().") "
                                           . "expect ($expected_result)\n";
        $result = 0;
    }
    return($result);
}

sub check_getVariant {
    my $result = 1;
    if (   (!check_getVariant1('en_US', ''))
        || (!check_getVariant1('fr_FR', ''))
        || (!check_getVariant1('de_DE', ''))
        || (!check_getVariant1('nl_NL', ''))
    ) {
        $result = 0;
    }
    return($result);
}

sub check_getVariant1 {
    my ($locale_str, $expected_result) = @_;

    my $result = 1;
    my $locale = new ICU::Locale($locale_str);
    if ($locale->getVariant() ne $expected_result) {
        print "$locale_str getVariant() = (".$locale->getVariant().") "
                                           . "expect ($expected_result)\n";
        $result = 0;
    }
    return($result);
}

sub check_hashCode {
    my $result = 1;
    if (   (!check_hashCode1('en_US', '194995374'))
        || (!check_hashCode1('fr_FR', '197071591'))
        || (!check_hashCode1('de_DE', '192664693'))
        || (!check_hashCode1('nl_NL', '211761251'))
    ) {
        $result = 0;
    }
    return($result);
}

sub check_hashCode1 {
    my ($locale_str, $expected_result) = @_;

    my $result = 1;
    my $locale = new ICU::Locale($locale_str);
    if ($locale->hashCode() ne $expected_result) {
        print "$locale_str hashCode() = (".$locale->hashCode().") "
                                           . "expect ($expected_result)\n";
        $result = 0;
    }
    return($result);
}

sub check_new {
    my $result = 1;
    if (   (!check_new1('en', 'US'))
        || (!check_new1('fr', 'FR'))
        || (!check_new1('de', 'DE'))
        || (!check_new1('nl', 'NL'))
    ) {
        $result = 0;
    }
    return($result);
}

sub check_new1 {
    my ($lang, $country) = @_;

    my $result = 1;
    my $locale = new ICU::Locale($lang, $country);
    if (!defined($locale)) {
        print "failed to create locale for $lang $country\n";
        $result = 0;
    }
    return($result);
}

sub check_setDebugLevel {
    my $result = 1;
    my $old_debug_level1 = ICU::Locale::setDebugLevel();
    my $old_debug_level2 = ICU::Locale::setDebugLevel();
    if ($old_debug_level1 != $old_debug_level2) {
        $result = 0;
    }
    my $val1 = ICU::Locale::setDebugLevel(3);
    if ($val1 != $old_debug_level2) {
        $result = 0;
    }
    my $val2 = ICU::Locale::setDebugLevel();
    if ($val2 != 3) {
        $result = 0;
    }
    ICU::Locale::setDebugLevel($old_debug_level2);

    return($result);
}

sub check_setDefault {
    my $result = 1;
    if (   (!check_setDefault1('en_US', 'EEEE, MMMM d, yyyy h:mm:ss a'))
        || (!check_setDefault1('fr_FR', 'EEEE d MMMM yyyy HH:mm:ss'))
        || (!check_setDefault1('de_DE', 'EEEE, d. MMMM yyyy HH:mm:ss'))
        || (!check_setDefault1('nl_NL', 'EEEE d MMMM yyyy H:mm:ss'))
    ) {
        $result = 0;
    }
    return($result);
}

sub check_setDefault1 {
    my ($locale_str, $expected_result) = @_;

    my $result = 1;
    my $locale = new ICU::Locale($locale_str);
    ICU::Locale->setDefault($locale);
    my $df = ICU::DateFormat->createDateTimeInstance(kFull);
    if ($df->toPattern() ne $expected_result) {
        $result = 0;
	print "$locale_str setDefault(): toPattern = (".$df->toPattern()
	                           ."), expected ($expected_result)\n";
    }
    return($result);
}

exit(0);

