# -*- Perl -*-
# a module for managing decks of cards
package Game::Deckar;
our $VERSION = '0.01';
use 5.26.0;
use warnings;
use Object::Pad 0.66;

sub fisher_yates_shuffle {
    my ($deck) = @_;
    my $i;
    for ( $i = @$deck; --$i; ) {
        my $j = int rand( $i + 1 );
        next if $i == $j;
        @$deck[ $i, $j ] = @$deck[ $j, $i ];
    }
}

class Game::Deckar::Card {
    field $data :param :reader;
    field %meta;

    BUILD {
        my (%param) = @_;
        %meta = $param{meta}->%* if exists $param{meta};
    }

    method meta ($name) { $meta{$name} }

    method set_meta ( $name, $value ) {
        my $old = $meta{$name};
        $meta{$name} = $value;
        return sub { $self->set_meta( $name, $old ); };
    }
}

class Game::Deckar {
    use Carp 'croak';
    field %decks;

    BUILD {
        my (%param) = @_;
        if ( exists $param{decks} ) {
            for my $d ( $param{decks}->@* ) {
                $decks{$d} = [];
            }
        }
        if ( exists $param{initial} ) {
            for my $d ( keys $param{initial}->%* ) {
                push $decks{$d}->@*, $param{initial}->{$d}->@*;
            }
        }
        if ( exists $param{initial_cards} ) {
            for my $d ( keys $param{initial_cards}->%* ) {
                push $decks{$d}->@*, map {
                    Game::Deckar::Card->new(
                        data => $_,
                        ( exists $param{meta} ? ( meta => $param{meta} ) : () )
                    )
                } $param{initial_cards}->{$d}->@*;
            }
        }
    }

    method add_deck ($name) {
        croak 'deck already exists' if exists $decks{$name};
        $decks{$name} = [];
        return sub { $self->del_deck($name); };
    }

    method del_deck ($name) {
        croak 'no such deck' unless exists $decks{$name};
        croak 'deck is not empty' if $decks{$name}->@*;
        delete $decks{$name};
        return sub { $self->add_deck($name); };
    }

    method collect ( $name, @rest ) {
        croak 'no such deck'     unless exists $decks{$name};
        croak 'not enough decks' unless @rest;
        # need also the name or reference to revert to... a list of pairs
        my @ncards = [ $name, [ $decks{$name}->@* ] ];
        for my $d (@rest) {
            next if $d eq $name;    # so can collect on "get_decks"
                # unknown decks are ignored as restoring halfway through a
                # list of decks would be annoying. maybe a subsequent
                # release could rollback and throw an error
            next unless exists $decks{$d};
            push @ncards, [ $d, [ $decks{$d}->@* ] ];
            # cards are put onto the "top" of the target deck, which is
            # how some humans might do it with real stacks of cards
            unshift $decks{$name}->@*, splice $decks{$d}->@*;
        }
        return sub {
            for my $r (@ncards) { $decks{ $r->[0] } = $r->[1] }
        };
    }

    method deal ( $src, $dst, $index = 0, $top = 1 ) {
        croak 'no such deck'
          unless exists $decks{$src} and exists $decks{$dst};
        croak 'too few cards' unless $decks{$src}->@*;
        if ( $index != 0 ) {
            croak 'index out of range'
              if $index < 0
              or $index > $decks{$src}->$#*;
        }
        my $card = splice $decks{$src}->@*, $index, 1;
        my $dref = $decks{$dst};
        splice $dref->@*, ( $top ? 0 : $dref->@* ), 0, $card;
        return $card, sub {
            splice $decks{$src}->@*, $index, 0, splice $dref->@*,
              ( $top ? 0 : $dref->$#* ), 1;
        };
    }

    method empty ($name) {
        croak 'no such deck'  unless exists $decks{$name};
        croak 'deck is empty' unless $decks{$name}->@*;
        my @orig = $decks{$name}->@*;
        $decks{$name} = [];
        return sub { $decks{$name} = \@orig; };
    }

    method get_decks () {
        croak 'no decks' unless %decks;
        return sort keys %decks;
    }

    method get ($name) {
        croak 'no such deck' unless exists $decks{$name};
        return $decks{$name};
    }

    method shuffle ($name) {
        croak 'no such deck'  unless exists $decks{$name};
        croak 'deck is empty' unless $decks{$name}->@*;
        my @copy = $decks{$name}->@*;
        fisher_yates_shuffle( $decks{$name} );
        return sub { $decks{$name} = \@copy; };
    }
}

1;
__END__

=head1 Name

Game::Deckar - a module for wrangling decks of cards

=head1 SYNOPSIS

  use Game::Deckar;
  
  # Card object (optional)
  my $card = Game::Deckar::Card->new( data => "Turn Left" );
  
  my $undo = $card->set_meta( hidden => 1 );
  
  say for $card->data, $card->meta('hidden');    # "Turn Left", 1
  
  $undo->() say $card->meta('hidden');           # undef
  
  
  # Deck object
  my $deck = Game::Deckar->new(
      decks   => [qw/new player1 player2 discard/],
      initial => { new => [ "Turn Left", "Stand Up" ] },
  );
  
  $deck->shuffle('new');
  
  ( $card, $undo ) = $deck->deal( new => 'player1' );
  say $card;
  
  $undo->();    # undeal the card
  
  
  # Deck with card object wrapping (all not visible)
  $deck = Game::Deckar->new(
      decks         => [qw/new player1 player2 discard/],
      initial_cards => { new     => [ 'A' .. 'F' ] },
      meta          => { visible => 0 },
  );
  
  ( $card, $undo ) = $deck->deal( new => 'player1' );
  $card->set_meta( visible => 1 );

=head1 DESCRIPTION

Deckar provides for arrays of cards and various supporting methods to
deal with them. An optional card object allows metadata such as card
visibility or whatever to be associated with cards. Means to undo
changes are provided.

The various "decks" represent stacks (or queues) of cards, so a "deck of
cards" might be split into various named decks such as a pile to draw
from, individual decks for each of the player's hands, discard piles,
etc. Naming of these decks is left to the caller. The deck names are
assumed to be strings.

The "top" of a deck is arbitrarily index C<0>, and the bottom the last
element of the array. Therefore interactions with the top involve
C<shift>, C<unshift>, or splicing at index C<0>; interactions with the
bottom use C<push>, C<pop>, or splicing at C<@array> or C<$#array>.

=head1 CLASSES

=head2 Game::Deckar::Card

An optional container object for cards (of whatever content) that also
provides for metadata about a card, such as whether the card is visible,
counters on the card, etc.

=over 4

=item B<data>

Returns the card data.

=item B<new> B<data> => I<card-data>, [ B<meta> => { I<default metadata> } ]

Constructor. Card data must be provided. This could be a number, text
such as "Ace of Spades" or "Repulse the Monkey", or a reference to an
object or data structure. This data is only held onto by this module.

Changes made by the constructor are not available for undo.

=item B<meta> I<key>

Returns the value for the meta I<key> associated with the card.

=item B<set_meta> I<key> I<value>

Sets metadata for a card. Returns an undo code reference.

=back

=head2 Game::Deckar

This class organizes cards into decks and provides methods for moving
and shuffling cards. The caller can either use their own card data or
instead host that data within C<Game::Deckar::Card> objects.

=over 4

=item B<add_deck> I<name>

Adds an empty named deck. Returns an undo code reference.

=item B<collect> I<target> I<source1> [I<source2> ..]

Moves all cards from the source deck(s) onto the top of the target deck.
The target deck can be specified in the source list and will be ignored.
Returns an undo code reference.

=item B<del_deck> I<name>

Deletes a named deck. Decks must be empty to be deleted. Returns an undo
code reference.

=item B<deal> I<source-deck> I<dest-deck> [ I<index> ] [ I<to-top?> ]

Deals from by default the top of the source deck to the destination
deck. I<index> specifies the index used to pick from the source deck,
C<0> or the top by default. The I<to-top?> boolean controls whether the
card goes to the top of the destination (the default) or to the bottom.

Returns the card dealt (so that card metadata can be changed, if need
be) and an undo code reference.

=item B<empty> I<name>

Removes all cards from the named deck. Returns an undo code reference.

=item B<get> I<name>

Returns an array reference to the cards in the deck I<name>. The
contents of the array should not be modified; if you do modify it,
undo code references may do unexpected things, unless you also handle
that yourself.

The count of elements in a deck can be obtained by using the array
reference in scalar context.

  my $hand  = $deck->get('player1');
  my $count = @$hand;

=item B<get_decks>

Returns a sorted list of the decks present in the object.

=item B<new>

Constructor. With I<decks> creates those deck names from the list given.
With I<initial> puts the given card lists into the given deck name. With
I<initial-card> and possibly also I<meta> does the same as I<initial>
but wraps the cards in C<Game::Deckar::Card> objects.

See the L</"SYNOPSIS"> or the C<t/> directory code for examples.

Changes made by the constructor are not available for undo.

=item B<shuffle> I<name>

Shuffles the deck. Returns an undo code reference.

=back

=head1 FUNCTION

Not exported.

=over 4

=item B<fisher_yates_shuffle>

Used to in-place shuffle decks. Uses Perl's C<rand()> for the
"random" numbers.

=back

=head1 BUGS

It's new code, so various necessary methods are likely missing.

=head1 SEE ALSO

L<Games::Cards> however the documentation claimed that undo was not
available following a desk shuffle, and the module looked biased towards
a conventional deck of cards.

=head1 COPYRIGHT AND LICENSE

Copyright 2023 Jeremy Mates

This program is distributed under the (Revised) BSD License:
L<https://opensource.org/licenses/BSD-3-Clause>

=cut
