package OpenInteract::Handler::Page;

# $Id: Page.pm,v 1.42 2003/05/18 20:08:57 lachoy Exp $

use strict;
use Class::Date ();
use File::Basename;
use File::Spec;
use OpenInteract::CommonHandler qw( OK ERROR );
use OpenInteract::Handler::GenericDispatcher qw( DEFAULT_SECURITY_KEY );
use SPOPS::Secure qw( :level );

@OpenInteract::Handler::Page::ISA     = qw( OpenInteract::CommonHandler  SPOPS::Secure );
$OpenInteract::Handler::Page::VERSION = sprintf("%d.%02d", q$Revision: 1.42 $ =~ /(\d+)\.(\d+)/);
$OpenInteract::Handler::Page::author            = 'chris@cwinters.com';
$OpenInteract::Handler::Page::default_method    = 'show';
@OpenInteract::Handler::Page::forbidden_methods = ();
%OpenInteract::Handler::Page::security          = (
   DEFAULT_SECURITY_KEY() => SEC_LEVEL_WRITE,
   show   => SEC_LEVEL_NONE, notify  => SEC_LEVEL_READ,
);

# 52 weeks -- default expiration for page

use constant DEFAULT_EXPIRE  => '365D';

# Use this to check whether the file retrieved is displayable in the
# browser and in the normal template setup; others (pdf, ps, mov,
# etc.) get sent to the user directly

my %DISPLAY_TYPES = map { $_ => 1 } ( 'text/html' );

# Use this to separate your single document into multiple pages

my $PAGE_SEPARATOR = '<!--PAGE-->';


sub MY_PACKAGE                 { return 'base_page' }
sub MY_HANDLER_PATH            { return '/Page' }
sub MY_OBJECT_TYPE             { return 'page' }
sub MY_OBJECT_CLASS            { return OpenInteract::Request->instance->page }

sub MY_SEARCH_FORM_TEMPLATE    { return 'page_search_form' }
sub MY_SEARCH_FORM_TITLE       { return 'Search Pages' }
sub MY_ALLOW_SEARCH_FORM       { return 1 }

sub MY_SEARCH_RESULTS_TEMPLATE { return 'page_search_results' }
sub MY_SEARCH_RESULTS_TITLE    { return 'Page Search Results' }
sub MY_SEARCH_FIELDS           { return qw( title author keywords main_template ) }
sub MY_SEARCH_FIELDS_EXACT     { return qw( is_active ) }
sub MY_ALLOW_SEARCH            { return 1 }

sub MY_ALLOW_CREATE            { return 1 }
sub MY_OBJECT_CREATE_SECURITY  { return SEC_LEVEL_WRITE }

sub MY_EDIT_RETURN_URL         { return '/' }
sub MY_EDIT_FAIL_TASK          { return 'show' }

# Use this if you want to display the page just edited
#sub MY_EDIT_DISPLAY_TASK       { return 'show' }

# Use this if you want to display the status of the page just edited
# -- good for uploads vs. in-place edits
sub MY_EDIT_DISPLAY_TASK       { return '_edit_status' }

sub MY_EDIT_FIELDS             { return qw( location title author keywords
                                            boxes main_template notes content
                                            storage content_location ) }
sub MY_EDIT_FIELDS_TOGGLED     { return qw( is_active template_parse ) }
sub MY_EDIT_FIELDS_DATE        { return qw( active_on expires_on ) }
sub MY_ALLOW_EDIT              { return 1 }

sub MY_REMOVE_FAIL_TASK        { return 'show' }
sub MY_REMOVE_DISPLAY_TASK     { return 'show_remove' }
sub MY_ALLOW_REMOVE            { return 1 }

sub MY_ALLOW_NOTIFY            { return 1 }
sub MY_ALLOW_WIZARD            { return undef }


sub admin   { return $_[0]->actions( $_[1] ) }
sub actions {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    $R->{page}{title} = "Page Administrative Actions";
    return $R->template->handler( {}, {},
                                  { name => 'base_page::admin_actions' } );
}


my %HELP_MAP = ( upload => 'page_upload_help',
                 rename => 'page_rename_help' );


sub help {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $type = $class->_read_field( $R->apache, 'type' ) || 'upload';
    my $tmpl_name = $HELP_MAP{ $type };
    $R->{page}{title} = "Page Administrative Actions";
    $R->{page}{_template_name_} = 'base_page::page_base_help';
    return $R->template->handler( {}, {},
                                  { name => "base_page::$tmpl_name" } );
}

# Overrides entry in OpenInteract::Handler::GenericDispatcher

sub _get_task {
    my ( $class ) = @_;
    my $R = OpenInteract::Request->instance;
    my ( $task );
    $task = 'show'  if ( $R->{path}{full}->[0] !~ /^page$/i );
    $task ||= lc shift @{ $R->{path}{current} } ||
              $OpenInteract::Handler::Page::default_method;
    $R->DEBUG && $R->scrib( 1, "Task found from _get_task: [$task]" );
    return $task;
}


sub _search_criteria_customize {
    my ( $class, $criteria ) = @_;
    my $R = OpenInteract::Request->instance;
    my $table = $R->page->table_name;

    # Administrators can pick whether to find active pages or not,
    # everyone else has to only search active pages. (Note: don't use
    # '_read_field_toggled' here because we want to ignore the
    # parameter if it wasn't chosen.

    if ( $R->{auth}{is_admin} ) {
        my $active_choice = $class->_read_field( $R->apache, 'is_active' );
        $criteria->{"$table.is_active"} = $active_choice if ( $active_choice );
    }
    else {
        $criteria->{"$table.is_active"} = 'yes';
    }
}


# Retrieve all directories, expanding the one we were asked to (if at
# all). Note that these are just the objects in the database, although
# there should be a corresponding content entry for every one of these
# in the filesystem or database.

sub directory_list {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;

    my $selected_dir = $class->_read_field( $R->apache, 'selected_dir' );
    my $params = { selected_dir => $selected_dir,
                   error_msg    => $p->{error_msg},
                   status_msg   => $p->{status_msg} };

    $params->{dir_list} = eval { $R->page->list_directories };
    if ( $@ ) {
        OpenIntereact::Error->set( SPOPS::Error->get );
        $R->throw({ code => 403 });
        $params->{error_msg} .= "\nCannot retrieve directories: $@";
        $params->{dir_list} = [];
    }

    # Store the pages found using the directory as a key pointing to a
    # listref of files it contains

    if ( $selected_dir ) {
        $params->{children_files} = $R->page->fetch_iterator({
                                               where => 'directory = ?',
                                               value => [ $selected_dir ] });
    }

    $R->{page}{title} = 'Document Listing';
    return $R->template->handler( {}, $params,
                                  { name => 'base_page::page_directory_list' } );
}

########################################
# REMOVE DISPLAY

sub show_remove {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $location = $class->_read_field( $R->apache, 'location' );
    unless ( $location ) {
        return $class->search_form( $p );
    }
    my $dirname = File::Basename::dirname( $location );
    $p->{location} = $dirname;
    return $class->show( $p );
}

########################################
# CREATE SUBDIR

# Display form to add subdirectory to $parent_dir

sub specify_subdirectory {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $parent_dir = $class->_read_field( $R->apache, 'parent_directory' );
    unless ( $class->_check_location_writable( $parent_dir ) ) {
        $R->scrib( 0, "Insufficient permissions to add subdirectory" );
        my $error_msg = "Insufficient permissions to add subdirectory. No action taken.";
        return $R->template->handler( {}, { error_msg => $error_msg },
                                      { name => 'base_page::directory_edit_status' } );
    }
    return $R->template->handler( {}, { parent_directory => $parent_dir },
                                  { name => 'base_page::directory_form_simple' } );
}


# Add subdirectory $cleaned_dir to $parent_dir

sub add_subdirectory {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $parent_dir = $class->_read_field( $R->apache, 'parent_directory' );
    unless ( $class->_check_location_writable( $parent_dir ) ) {
        $R->scrib( 0, "Insufficient permissions to add subdirectory" );
        my $error_msg = "Insufficient permissions to add subdirectory. No action taken.";
        return $R->template->handler( {}, { error_msg => $error_msg },
                                      { name => 'base_page::directory_edit_status' } );
    }

    my $cleaned_dir = $R->page->clean_filename(
                              $class->_read_field( $R->apache, 'directory' ) );
    $cleaned_dir =~ s/\.//;
    my $page_dir = File::Spec->catdir( $parent_dir, $cleaned_dir );
    my $display_path = join( '/', $parent_dir, $cleaned_dir );
    $display_path =~ s|/+|/|g;
    my $full_dir = File::Spec->catdir( $R->config->get_dir( 'html' ), $page_dir );
    my %params = ( parent_directory  => $parent_dir,
                   created_directory => $display_path,
                   action            => 'add_subdirectory' );

    eval { File::Path::mkpath( $full_dir, undef, 0775 ) };
    if ( $@ ) {
        $params{error_msg} = "Failed to create [$display_path]: $@";
    }
    else {
        $params{status_msg} = "Directory [$display_path] created ok";
    }
    return $R->template->handler( {}, \%params,
                                  { name => 'base_page::directory_edit_status' } );
}


# NOTE: This removes a directory and all files underneath. This can
# obviously be very dangerous, so you should ensure the user doesn't
# call this accidentally by setting the parameter
# 'remove_directory_confirm' to 'yes'.

sub remove_directory {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $confirm = lc $class->_read_field( $R->apache, 'remove_directory_confirm' );
    my $directory = $R->page->clean_name(
                              $class->_read_field( $R->apache, 'directory' ) );

    # Get rid of the trailing separator so we can find the pages with
    # this directory value in the database (they always have the
    # trailing separator removed)

    $directory =~ s|/$||;

    if ( ! $directory or $directory eq '/' ) {
        $R->scrib( 0, "Empty/root directory given [$directory]; no action taken" );
        my $error_msg = 'Directory removal cancelled: must specify a non-root directory';
        return $R->template->handler( {}, { error_msg => $error_msg,
                                            action    => 'remove_directory' },
                                      { name => 'base_page::directory_edit_status' } );
    }

    if ( $confirm eq 'no' ) {
        $R->DEBUG && $R->scrib( 1, "Directory removal cancelled" );
        return $class->show({ location   => $directory,
                              status_msg => 'Directory removal cancelled' });
    }

    if ( $confirm ne 'yes' ) {
        $R->DEBUG && $R->scrib( 1, "Need confirmation for directory removal" );
        return $R->template->handler( {}, { directory => $directory },
                                      { name => 'base_page::directory_remove_confirm' } );
    }

    unless ( $class->_check_location_writable( $directory ) ) {
        $R->scrib( 0, "Insufficient permissions to remove directory" );
        my $error_msg = "Insufficient permissions to remove directory. No action taken.";
        return $R->template->handler( {}, { error_msg => $error_msg },
                                      { name => 'base_page::directory_edit_status' } );
    }

    $R->DEBUG && $R->scrib( 1, "Removal of dir [$directory] confirmed by user" );
    eval {
        my $pages_in_dir = $R->page->fetch_group({ where => 'directory LIKE ?',
                                                   value => [ "$directory%" ] });
        foreach my $page ( @{ $pages_in_dir } ) {
            $R->DEBUG && $R->scrib( 1, "Removing page [$page->{location}]" );
            $page->remove;
            $R->DEBUG && $R->scrib( 1, "Removed page [$page->{location}] ok" );
        }
        my $full_dir = File::Spec->catdir( $R->config->get_dir( 'html' ), $directory );
        $R->DEBUG && $R->scrib( 1, "Removing directory [$full_dir] from filesystem" );
        File::Path::rmtree( $full_dir );
        $R->DEBUG && $R->scrib( 1, "Removed directory [$full_dir] ok" );
    };
    if ( $@ ) {
        $R->scrib( 0, "Error in removal process: $@" );
        my $error_msg = "Failed to remove directory tree. Files may be in " .
                     "inconsistent state. (Error: $@)";
        return $R->template->handler( {}, { error_msg => $error_msg,
                                            action    => 'remove_directory' },
                                      { name => 'base_page::directory_edit_status' } );
    }
    $R->DEBUG && $R->scrib( 1, "Removal of directory [$directory] ok" );
    my @pieces = split( '/', $directory );
    pop @pieces;
    my $view_directory = join( '/', @pieces );
    $R->DEBUG && $R->scrib( 1, "Trying to view directory [$view_directory]" );
    return $R->template->handler( {}, { status_msg       => "Removed directory [$directory] ok",
                                        directory        => $directory,
                                        parent_directory => $view_directory,
                                        action           => 'remove_directory' },
                                  { name => 'base_page::directory_edit_status' } );
}


########################################
# RENAME FILE/PAGE

sub specify_rename {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $location = $class->_read_field( $R->apache, 'location' );
    unless ( $class->_check_location_writable( $location ) ) {
        $R->scrib( 0, "Insufficient permissions to rename file" );
        my $error_msg = "Insufficient permissions to rename file. No action taken.";
        return $R->template->handler( {}, { error_msg => $error_msg },
                                      { name => 'base_page::page_edit_status' } );
    }
    my ( $error_msg );
    my $page = eval { $R->page->fetch( $location ) };
    if ( $@ ) {
        $error_msg = "Failed to retrieve page for specified location [$location]";
    }
    return $R->template->handler( {}, { page      => $page,
                                        error_msg => $error_msg },
                                  { name => 'base_page::page_form_rename' } );
}

sub rename_file {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $old_location = $class->_read_field( $R->apache, 'old_location' );
    my $new_location = $R->page->clean_name(
                              $class->_read_field( $R->apache, 'new_location' ) );
    $R->DEBUG && $R->scrib( 1, "Trying to rename [$old_location] -> [$new_location]" );
    unless ( $class->_check_location_writable( $old_location ) && 
             $class->_check_location_writable( $new_location ) ) {
        $R->scrib( 0, "Insufficient permissions to rename file" );
        my $error_msg = "Insufficient permissions to rename file. No action taken.";
        return $R->template->handler( {}, { error_msg => $error_msg },
                                      { name => 'base_page::page_edit_status' } );
    }

    my $old_page = $R->page->fetch( $old_location );
    unless ( $old_page ) {
        $R->scrib( 0, "Old location [$old_location] does not exist" );
        my $error_msg = "Cannot rename [$old_location] to [$new_location]: " .
                     "[$old_location] does not exist";
        return $R->template->handler( {}, { error_msg => $error_msg },
                                      { name => 'base_page::page_edit_status' } );
    }
    $R->DEBUG && $R->scrib( 1, "Old location fetched ok" );

    my $new_page = $R->page->fetch( $new_location );
    if ( $new_page ) {
        $R->scrib( 0, "New location [$new_location] already exists" );
        my $error_msg = "Cannot rename [$old_location] to [$new_location]: " .
                     "[$new_location] already exists!";
        return $R->template->handler( {}, { error_msg => $error_msg },
                                      { name => 'base_page::page_edit_status' } );
    }
    $R->DEBUG && $R->scrib( 1, "New location does not exist (ok)" );

    $old_page->{location} = $new_location;
    eval { $old_page->rename_content( $old_location ) };
    if ( $@ ) {
        $R->scrib( 0, "Error renaming content: $@" );
        my $error_msg = "Cannot rename [$old_location] to [$new_location]: " .
                        "Failed to rename filesystem content to new location: $@.";
        return $R->template->handler( {}, { error_msg => $error_msg },
                                      { name => 'base_page::page_edit_status' } );
    }
    $R->DEBUG && $R->scrib( 1, "Page content renamed successfully" );

    $R->DEBUG && $R->scrib( 1, "Saving page with new location [$old_page->{location}]" );
    eval { $old_page->save({ use_id => $old_location }) };
    if ( $@ ) {
        $R->scrib( 0, "Error saving page object: $@" );
        my $error_msg = "Cannot rename [$old_location] to [$new_location]: " .
                        "Error saving page with new location: $@";
        return $R->template->handler( {}, { error_msg => $error_msg },
                                      { name => 'base_page::page_edit_status' } );
    }
    $R->DEBUG && $R->scrib( 1, "Page saved with new location ok" );

    return $R->template->handler( {}, { status_msg => "File renamed ok",
                                        directory  => $old_page->{directory},
                                        location   => $old_page->{location} },
                                  { name => 'base_page::page_edit_status' } );
}

sub _check_location_writable {
    my ( $class, $location ) = @_;
    my $R = OpenInteract::Request->instance;
    my $level = $R->page->check_security({ object_id => $location,
                                           user      => $R->{auth}{user},
                                           group     => $R->{auth}{group} });
    return ( $level >= SEC_LEVEL_WRITE );
}

########################################

# Override to first check for 'old_location' and THEN the normal ID

sub fetch_object {
    my ( $class, $id, @id_field ) = @_;
    my $R = OpenInteract::Request->instance;
    my $existing_location = $class->_read_field( $R->apache, 'old_location' );
    my $fetch_id = ( $id =~ m|^/| ) ? $id : "/$id";
    $R->DEBUG && $R->scrib( 1, "Pre-fetch: [Exist: $existing_location]",
                            "[New: $fetch_id]" );
    return ( $existing_location )
             ? $class->SUPER::fetch_object( $existing_location, @id_field )
             : $class->SUPER::fetch_object( $fetch_id, @id_field );
}


# Yes, this is out of the normal order. It's just that show() is so
# big and includes so much stuff...

sub _edit_customize {
    my ( $class, $page, $old_data ) = @_;
    my $R = OpenInteract::Request->instance;

    my %opts = ();

    # See if the upload should be there -- note that
    # _handle_uploaded_file() sets the MIME type for us and sets the
    # filename, size and type reported by the upload in tmp_upload

    my $has_upload = $class->_read_field_toggled( $R->apache, 'use_upload' );
    if ( $has_upload eq 'yes' ) {
        $R->DEBUG && $R->scrib( 1, "Handling file upload" );
        my ( $type, $msg ) = $class->_handle_uploaded_file( $page );
        return ( $type, $msg ) if ( $type );
    }

    # See if we're using the 'simple' form, and if so set the location
    # of the file from the filename uploaded (after cleaning)

    my $form_type = $class->_read_field( $R->apache, 'form_type' ) || 'normal';
    if ( $form_type eq 'simple' ) {
        my $dir = $class->_read_field( $R->apache, 'directory' );
        unless ( $dir ) {
            my $error_message = "Failed to upload file. No directory specified.";
            return ( ERROR, $error_message );
        }
        $dir =~ s|/$||;
        my $filename = $R->page->clean_filename( $page->{tmp_upload}{filename} );
        $page->{location} = join( '/', $dir, $filename );
        $R->DEBUG && $R->scrib( 1, "Set location to [$page->{location}] from",
                                   "filename reported by client [$filename]" );

        # Check to see if this location is already in the
        # database. (For non-simple pages this is done in the normal
        # editing sequence.) If so, set the saved-status of this to
        # true so the update takes place.

        unless ( $page->is_saved ) {
            my $existing_page = eval { $R->page->fetch( $page->{location} ) };
            $page->has_save if ( $existing_page );
        }
    }

    # If we're not using the simple form, check to see if the user
    # changed the location, then we need to set the ID so the UPDATE
    # works properly.

    else {
        unless ( $page->{location} =~ m|^/| ) {
            $page->{location} = "/$page->{location}";
        }
        if ( $old_data->{location} and
            $page->{location} ne $old_data->{location} ) {
            $opts{use_id} = $old_data->{location};
            $R->DEBUG && $R->scrib( 1, "This should be an UPDATE where we change ",
                                    "the primary key from [$old_data->{location}] ",
                                    "to [$page->{location}]" );
        }
    }

    # Ensure that the location is clean

    $R->DEBUG && $R->scrib( 1, "Location before clean [$page->{location}]" );
    $page->{location} = $R->page->clean_location( $page->{location} );
    $R->DEBUG && $R->scrib( 1, "Location after clean [$page->{location}]" );

    # Default the expires_on field

    unless ( $page->{expires_on} ) {
        my $expire_date = Class::Date->now + DEFAULT_EXPIRE;
        $page->{expires_on} = $expire_date->strftime( '%Y-%m-%d' );;
    }

    $page->{mime_type} ||= 'text/html';

    # Non-displayable docs always get saved to the filesystem (for
    # now); for these docs we also need to remove 'content' from the
    # list of fields to be processed by the FullText indexer

    unless ( $class->_is_displayable( $page->{mime_type} ) ) {
        $R->DEBUG && $R->scrib( 1, "Not displayable [$page->{mime_type}];",
                                   "don't scan content for index" );
        $page->{is_file} = 'yes';
        if ( $page->CONFIG->{fulltext_field} ) {
            $opts{fulltext_field} = [ grep ! /^content$/, @{ $page->{fulltext_field} } ];
        }
    }

    return ( OK, \%opts );
}

# If this is a successful update and the item's location has been
# changed, we need to tell the content implementation class to rename
# the content and do whatever other actions it requires.

sub _edit_post_action_customize {
    my ( $class, $page, $old_data ) = @_;
    if ( $page->{location} ne $old_data->{location} ) {
        $page->rename_content( $old_data->{location} );
    }
    return 1;
}


sub _edit_status {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    $p->{directory} = $p->{page}{directory};
    $p->{location}  = $p->{page}{location};
    $R->{page}{title} = 'Status of page edit';
    return $R->template->handler( {}, $p, { name => 'base_page::page_edit_status' } );
}


# Why do we set the content-type when returning errors? See note on
# error content-type forcing in POD...

sub show {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    $R->DEBUG && $R->scrib( 1, "Edit status of page: [edit: $p->{edit}] ",
                               "[new object: $p->{is_new_object}]" );

    my $page = $p->{page};
    $page  ||= $R->page->new()  if ( $p->{is_new_object} );

    my ( $location );

    # Try and find a page object (just the metadata) matching with our
    # location. Note that page_by_location() will also treat
    # $location as a directory request, where appropriate

    unless ( $page ) {
        $location = $class->_get_location( $p );
        my $item = eval { $R->page->page_by_location( $location ) };
        if ( $@ ) {
            $R->scrib( 0, "Caught error when fetching page by location [$location]: $@" );
            $class->_fail_page_fetch( $location, $@ );
        }
        if ( $item and $item->isa( $R->page_directory ) ) {
            $R->scrib( 1, "Returned item for location [$location] is a ",
                          "page_directory object; run dir handler" );
            return $class->_run_directory_handler( $item, $location, $p );
        }
        $page = $item;
    }

    # Bail if we're not creating a new page and haven't found a page
    # to display yet

    my $do_edit = $p->{edit} || $class->_read_field( $R->apache, 'edit' );
    unless ( $page or $do_edit ) {
        $R->{page}{content_type} = 'text/html';
        $location ||= $page->{location};
        $R->throw({ code => 314, system_msg => $location });
    }

    # See if we're supposed to edit

    if ( $do_edit ) {
        $page->{location} ||= $class->_get_location;
        return $class->show_editable_page( $page, $p );
    }

    # If we specified that we're going to send a separate file to the
    # user (usually not HTML, text, etc.) then set the information and
    # quit processing

    unless ( $class->_is_displayable( $page->{mime_type} ) ) {
        return $class->show_nondisplayable_page( $page, $p );
    }

    return $class->show_displayable_page( $page, $p );
}

sub show_displayable_page {
    my ( $class, $page, $p ) = @_;
    die "Must pass page to display\n" unless ( ref $page );

    my $R = OpenInteract::Request->instance;

    # Ensure the page is active

    unless ( $class->_is_active( $page ) ) {
        $R->DEBUG && $R->scrib( 1, "Page is not currently active; return error" );
        $R->{page}{title} = 'Page not active';
        return '<h2 align="center">Not Active</h2><p>Sorry, this page is not active.</p>';
    }

    # Follow the alias chain to its end

    while ( $page->{storage} eq 'alias' ) {
        $page = eval { $R->page->page_by_location( $page->{content_location} ) };
        if ( $@ or ! $page ) {
            $R->scrib( 0, "Location for alias [$page->{content_location}]",
                          "wasn't found: $@" );
            return "Request was for an alias, but aliased page unavailable.";
        }
    }

    $R->DEBUG && $R->scrib( 1, "Display [$page->{location}] as normal HTML" );
    $page->fetch_content;

    # Use page metadata to modify display

    $R->{page}{title} = $page->{title};
    $R->{page}{_template_name_} = $page->{main_template};
    $class->_add_object_boxes( $page, $p );

    # Grab content we're actually going to show

    my $display_content = $class->_split_pages( $page );

    return $display_content if ( $page->{template_parse} eq 'no' );
    $p->{page} = $page;
    return $R->template->handler( {}, $p, { text => \$display_content } );
}


sub show_editable_page {
    my ( $class, $page, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    $page ||= $R->page->new;


    # If this is an editable doc, fetch the content, otherwise,
    # mark it as non-editable to the form

    $page->{storage} ||= 'file';
    if ( $class->_is_displayable( $page->{mime_type} ) ) {
        $page->fetch_content if $page->is_saved;
    }
    else {
        $p->{non_editable} = 1;
    }
    $R->DEBUG && $R->scrib( 1, "This page should be in an editable form;",
                               "uneditable content status [$p->{non_editable}]" );
    $p->{page} = $page;

    my $tmpl_name = 'page_form';

    # Check and see if this is a request to upload the page using the
    # 'simple' form

    my $form_type = $p->{form_type} ||
                    $class->_read_field( $R->apache, 'form_type' ) ||
                    'normal';
    if ( $form_type eq 'simple' ) {
        $tmpl_name = 'page_form_simple';
        $p->{directory} = $page->{location};
    }

    $R->{page}{title}   = 'Edit a Document';
    return $R->template->handler( {}, $p, { name => "base_page::$tmpl_name" } );
}


sub show_nondisplayable_page {
    my ( $class, $page ) = @_;
    my $R = OpenInteract::Request->instance;
    $R->{page}{content_type}   = $page->{mime_type};
    $R->{page}{send_file}      = join( '', $R->CONFIG->get_dir( 'html' ), $page->{location} );
    $R->{page}{send_file_size} = $page->{size};
    $R->DEBUG && $R->scrib( 1, "File being retrieved is not directly displayable.",
                               "Set 'send_file' to [$page->{location}]" );
    return undef;
}


sub _fail_page_fetch {
    my ( $class, $location, $error ) = @_;
    my $R = OpenInteract::Request->instance;
    $R->{page}{content_type} = 'text/html';
    $R->DEBUG && $R->scrib( 1, "Could not retrieve page. Error [$error]" );
    if ( $error =~ /^security/ ) {
        $R->throw({ code => 303 });
    }
    elsif ( $error =~ /^access/ ) {
        return "<h2>Cannot Access</h2><p>Failure accessing page.</p>";
    }
    return "<h2>Unknown Error</h2><p>Could not fetch page -- unknown failure.</p>";
}


sub _run_directory_handler {
    my ( $class, $page_directory, $location, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    $p->{page_directory} = $page_directory;
    $p->{directory}      = $location;
    my ( $dir_class, $dir_method ) = $R->lookup_action( $page_directory->{action} );
    if ( $dir_class and $dir_method ) {
        return $dir_class->$dir_method( $p );
    }
    die "Tried to execute directory handler for ($page_directory->{directory}) ",
        "but there is no mapping for ($page_directory->{action}).\n";
}


# True means page is displayable in browser, false means it's not. We
# treat an empty mime_type as an HTML page. (Might change)

sub _is_displayable {
    my ( $class, $mime_type) = @_;
    return 1 unless ( $mime_type );
    return 1 if ( $DISPLAY_TYPES{ $mime_type } );
    return undef;
}


# Grab the location from whatever is available -- passed parameters,
# GET/POST parameters, or the original path. Once found, clean it up

sub _get_location {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my ( $location );
    $location = $p->{page}{location} if ( $p->{page} );
    unless ( $location ) {
        $location   = $p->{location} ||
                      $class->_read_field( $R->apache, 'location' );
        $location ||= $R->{path}{original} if ( $R->{path}{original} !~ m|^/page\b|i );
    }

    # Clean up the location

    $location = $R->page->clean_location( $location );
    return $location;
}


# A page can have one or more tags that declare it wants itself split
# into multiple pieces for display. This routine does the
# splitting. This is still under development...

sub _split_pages {
    my ( $class, $page ) = @_;
    my $R = OpenInteract::Request->instance;

    # Split the page into separate pages -- first check and see if the
    # document IS paged, then do the splitting and other contortions

    if ( $page->{content} =~ /$PAGE_SEPARATOR/ ) {
        my @text_pages      = split /$PAGE_SEPARATOR/, $page->{content};
        my $page_num        = $class->_read_field( $R->apache, 'pagenum' ) || 1;
        my $this_page       =  $text_pages[ $page_num - 1 ];
        my $total_pages     = scalar @text_pages;
        my $current_pagenum = $page_num;
        $this_page .= <<PCOUNT;
     <p align="right"><font size="-1">
     [%- PROCESS page_count( total_pages     = $total_pages,
                             url             = '$page->{location}',
                             current_pagenum = $current_pagenum ) -%]
     </font></p>
PCOUNT
       return $this_page;
    }
    return $page->{content};
}


sub _add_object_boxes {
    my ( $class, $page, $p ) = @_;
    my $R = OpenInteract::Request->instance;

    my $box_string = $page->{boxes};

    # Add boxes as necessary -- names beginning with a '-' should be
    # tagged for removal

    if ( $box_string ) {
        $box_string =~ s/^\s+//;
        my @boxes = split /\s+/, $box_string;
        foreach my $box_name ( @boxes ) {
            next if ( $box_name =~ /^\s*$/ );
            $R->DEBUG && $R->scrib( 1, "Adding box name [$box_name] from page definition" );
            my $box_params = { name => $box_name };
            if ( $box_name =~ s/^\-// ) {
                $box_params->{name}   = $box_name;
                $box_params->{remove} = 'yes';
            }
            push @{ $R->{boxes} }, $box_params
        }
    }

    # If this person has WRITE access to the module, give them a box
    # so they can edit/remove this document

    if ( $p->{level} >= SEC_LEVEL_WRITE ) {
        push @{ $R->{boxes} }, { name   => 'edit_document_box',
                                 params => { page => $page } };
    }
    return undef;
}


sub _is_active {
    my ( $class, $page ) = @_;
    my $R = OpenInteract::Request->instance;

    if ( $page->{is_active} eq 'no' ) {
        $R->DEBUG && $R->scrib( 1, "Page NOT active: 'is_active' is false" );
        return undef;
    }
    unless ( $page->{active_on} ) {
        $R->DEBUG && $R->scrib( 1, "Page is active: there is no 'active_on' date" );
        return 1;
    }

    my $active  = Class::Date::date([ split /\D/, $page->{active_on} ]);
    my $now     = Class::Date->now;

    if ( $active > $now ) {
        $R->DEBUG && $R->scrib( 1, "Page NOT active: 'active_on' check failed" );
        return undef;
    }

    # OK if there's no expiration date, and the active date is ok

    unless ( $page->{expires_on} ) {
        $R->DEBUG && $R->scrib( 1, "Page is active: 'active_on' check ok, no expiration" );
        return 1;
    }

    my $expires = Class::Date::date([ split /\D/, $page->{expires_on} ]);
    if ( $now > $expires ) {
        $R->DEBUG && $R->scrib( 1, "Page NOT active: 'expires_on' check failed" );
        return undef;
    }

    return 1;
}


sub _handle_uploaded_file {
    my ( $class, $page ) = @_;
    my $R = OpenInteract::Request->instance;
    $R->DEBUG && $R->scrib( 1, "User is requesting content from uploaded file" );
    my $upload  = $R->apache->upload( 'content_upload' );
    unless ( $upload ) {
        my $error_msg = 'You checked off that you wanted to upload a ' .
                        'file but did not upload one. Why are you teasing me?';
        my %error_opts = ( method => 'show', error_msg => $error_msg, edit => 1 );
        return ( ERROR, \%error_opts );
    }
    $page->{tmp_upload} = { filename => $upload->filename,
                            size     => $upload->size,
                            type     => $upload->type };
    $R->DEBUG && $R->scrib( 1, "Upload seems to be retrieved ok. Here is some info:\n",
                               "Filename: [$page->{tmp_upload}{filename}] Size:",
                               "$page->{tmp_upload}{size}] Type: [$page->{tmp_upload}{type}]" );
    $page->{size} = $upload->size;
    if ( $class->_is_displayable( $page->{mime_type} ) ) {
        my $fh = $upload->fh;
        local $/ = undef;
        my $content = <$fh>;
        $page->{content} = \$content;
    }
    else {
        $page->{content} = $upload->fh;
    }
    $page->{mime_type} = $R->page->mime_type_by_extension( $upload->filename ) ||
                         $page->{tmp_upload}{type};
    return;
}

1;

__END__

=pod

=head1 NAME

OpenInteract::Handler::Page - Display HTML pages and other documents from the database and/or filesystem

=head1 SYNOPSIS

=head1 DESCRIPTION

Displays a 'static' page from information in the database. The URL to
the page looks like a normal page rather than a database call or other
GET request, although it B<can> look like a GET request if you want it
to.

=head2 Error Content-Type Forcing

We have to force the content-type when returning an error in C<show()>
because the user might have requested a file that actually exists in
the filesystem and which Apache has already mapped a content-type. You
will know when this happens because you will be prompted to d/l the
file or a plugin (like Acrobat Reader) will try to display it, but the
*actual* content will be plain old HTML...

=head1 METHODS

We use L<OpenInteract::CommonHandler|OpenInteract::CommonHandler> but
override the C<show()> method for our special needs.

B<directory_list>: implemented in this class

B<search_form>: implemented in
L<OpenInteract::CommonHandler|OpenInteract::CommonHandler>

B<search>: implemented in
L<OpenInteract::CommonHandler|OpenInteract::CommonHandler>

B<show>: implemented in this class

B<edit>: implemented in
L<OpenInteract::CommonHandler|OpenInteract::CommonHandler>

B<remove>: implemented in
L<OpenInteract::CommonHandler|OpenInteract::CommonHandler>

B<notify>: implemented in
L<OpenInteract::CommonHandler|OpenInteract::CommonHandler>

=head1 BUGS

B<Date Format Assumptions>

For the active_on/expires_on dates in the page object, we assume the
date is formatted yyyy*mm*dd. This may not hold true in all cases. It
would probably be useful to create an SPOPS add-on that creates a
Class::Date (or Time::Piece, or whatever) object for every date field
in an object on a fetch().

=head1 TO DO

Nothing known.

=head1 SEE ALSO

L<OpenInteract::CommonHandler|OpenInteract::CommonHandler>

=head1 COPYRIGHT

Copyright (c) 2001-2002 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
