package Template::Extract::Run;
$Template::Extract::Run::VERSION = '0.50';
use strict;
use warnings;
use 5.006;

our ($DEBUG);
my ( %loop, $cur_loop, $data );

sub new {
    my $class = shift;
    my $self  = {};
    return bless( $self, $class );
}

sub run {
    my ( $self, $regex, $document, $ext_data ) = @_;

    $self->_init($ext_data);
    ## no critic

    defined($document) or return undef;
    defined($regex)    or return undef;

    {
        use re 'eval';
        return $data if $document =~ /$regex/s;
    }

    return undef;
}

# initialize temporary variables
sub _init {
    %loop     = ();
    $cur_loop = undef;
    $data     = $_[1] || {};
}

sub _enter_loop {
    $cur_loop = $loop{ $_[1] } ||= {
        name  => $_[0],
        id    => $_[1],
        count => -1,
    };
    $cur_loop->{count}++;
    $cur_loop->{var} = {};
    $cur_loop->{pos} = {};
}

sub _leave_loop {
    my ( $obj, $key, $vars ) = @_;

    ref($obj) eq 'HASH' or return;
    my $old;
    if (exists $obj->{$key}) {
        $old = $obj->{$key};
    }
    ref($old) eq 'ARRAY' or return;

    print "Validate: [$old $key @$vars]\n" if $DEBUG;

    my @new;

  OUTER:
    foreach my $entry (@$old) {
        next unless %$entry;
        foreach my $var (@$vars) {

            # If it's a foreach, it needs to not match or match something.
            if ( ref($var) ) {
                next if !exists( $entry->{$$var} ) or @{ $entry->{$$var} };
            }
            else {
                next if exists( $entry->{$var} );
            }
            next OUTER;    # failed!
        }
        push @new, $entry;
    }

    delete $_[0]{$key} unless @$old = @new;
}

sub _adjust {
    my ( $obj, $val ) = ( shift, pop );

    foreach my $var (@_) {
        $obj = $obj->{$var} ||= {};
    }
    return ( $obj, $val );
}

sub _traverse {
    my ( $obj, $val ) = ( shift, shift );

    my $depth = -1;
    while ( my $id = pop(@_) ) {
        my $var = $loop{$id}{name};
        my $index = $loop{ $_[-1] || $val }{count};
        $obj = $obj->{$var}[$index] ||= {};
    }
    return $obj;
}

sub _ext {
    my ( $var, $val, $num ) = splice( @_, 0, 3 );
    my $obj = $data;

    if (@_) {
        print "Ext: [ $$val with $num on $-[$num]]\n" if ref($val) and $DEBUG;

        # fetch current loop structure
        my $cur = $loop{ $_[0] };

        # if pos() changed, increment the iteration counter
        $cur->{var}{$num}++
          if ( ( $cur->{pos}{$num} ||= -1 ) != $-[$num] )
          or ref $val and $$val eq 'leave_loop';

        # remember pos()
        $cur->{pos}{$num} = $-[$num];

        my $iteration = $cur->{var}{$num} - 1;
        $obj = _traverse( $data, @_ )->{ $cur->{name} }[$iteration] ||= {};
    }

    ( $obj, $var ) = _adjust( $obj, @$var );

    if ( !ref($val) ) {
        $obj->{$var} = $val;
    }
    elsif ( $$val eq 'leave_loop' ) {
        _leave_loop( $obj, @$var );
    }
    else {
        $obj->{$var} = $$$val;
    }
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Template::Extract::Run - Apply compiled regular expressions on documents

=head1 VERSION

version 0.50

=head1 SYNOPSIS

    use Template::Extract::Run;
    use Data::Dumper;

    open FH, '<', 'stored_regex' or die $!;
    my $regex = join('', <FH>);
    close FH;

    my $document = << '.';
    <html><head><title>Great links</title></head><body>
    <ul><li><A HREF="http://slashdot.org">News for nerds.</A>: A+ - nice.
    this text is ignored.</li>
    <li><A HREF="http://microsoft.com">Where do you want...</A>: Z! - yeah.
    this text is ignored, too.</li></ul>
    .

    print Data::Dumper::Dumper(
        Template::Extract::Run->new->run($regex, $document)
    );

=head1 DESCRIPTION

This module applies a regular expression generated by
B<Template::Extract::Compile> to a document.

=head1 METHODS

=head2 new()

Constructor.  Currently takes no parameters.

=head2 run($regex, $document, \%values)

Applying C<$regex> on C<$document> and returning the resulting C<\%values>.
This process does not make use of the Template Toolkit or any other modules.

=head1 SEE ALSO

L<Template::Extract>, L<Template::Extract::Compile>

=head1 AUTHORS

Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT

Copyright 2004, 2005, 2007 by Audrey Tang E<lt>cpan@audreyt.orgE<gt>.

This software is released under the MIT license cited below.

=head2 The "MIT" License

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.

=for :stopwords cpan testmatrix url bugtracker rt cpants kwalitee diff irc mailto metadata placeholders metacpan

=head1 SUPPORT

=head2 Websites

The following websites have more information about this module, and may be of help to you. As always,
in addition to those websites please use your favorite search engine to discover more resources.

=over 4

=item *

MetaCPAN

A modern, open-source CPAN search engine, useful to view POD in HTML format.

L<https://metacpan.org/release/Template-Extract>

=item *

RT: CPAN's Bug Tracker

The RT ( Request Tracker ) website is the default bug/issue tracking system for CPAN.

L<https://rt.cpan.org/Public/Dist/Display.html?Name=Template-Extract>

=item *

CPANTS

The CPANTS is a website that analyzes the Kwalitee ( code metrics ) of a distribution.

L<http://cpants.cpanauthors.org/dist/Template-Extract>

=item *

CPAN Testers

The CPAN Testers is a network of smoke testers who run automated tests on uploaded CPAN distributions.

L<http://www.cpantesters.org/distro/T/Template-Extract>

=item *

CPAN Testers Matrix

The CPAN Testers Matrix is a website that provides a visual overview of the test results for a distribution on various Perls/platforms.

L<http://matrix.cpantesters.org/?dist=Template-Extract>

=item *

CPAN Testers Dependencies

The CPAN Testers Dependencies is a website that shows a chart of the test results of all dependencies for a distribution.

L<http://deps.cpantesters.org/?module=Template::Extract>

=back

=head2 Bugs / Feature Requests

Please report any bugs or feature requests by email to C<bug-template-extract at rt.cpan.org>, or through
the web interface at L<https://rt.cpan.org/Public/Bug/Report.html?Queue=Template-Extract>. You will be automatically notified of any
progress on the request by the system.

=head2 Source Code

The code is open to the world, and available for you to hack on. Please feel free to browse it and play
with it, or whatever. If you want to contribute patches, please send me a diff or prod me to pull
from your repository :)

L<https://github.com/shlomif/template-extract>

  git clone git://github.com/shlomif/template-extract.git

=head1 AUTHOR

Shlomi Fish <shlomif@cpan.org>

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website
L<https://github.com/shlomif/template-extract/issues>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2003 by Audrey Tang.

This is free software, licensed under:

  The MIT (X11) License

=cut
