package Unicode::Collate::CJK::Pinyin;

use 5.006;
use strict;

our $VERSION = '0.65';

my %u2p;
my $wt = 0x8000;

while (<DATA>) {
    last if /^__END__/;
    my @c = split;
    for my $c (@c) {
	next if !$c;
	$u2p{hex($c)} = $wt;
	$wt++;
    }
}

sub weightPinyin {
    my $u = shift;
    return exists $u2p{$u} ? $u2p{$u} : undef;
}

1;
__DATA__
3105 3106 3107 3108 3109 310A 310B 310C 310D 310E
310F 3110 3111 3112 3113 3114 3115 3116 3117 3118
3119 311A 311B 311C 311D 311E 311F 3120 3121 3122
3123 3124 3125 3126 3127 3128 3129 5416 963F 554A
9515 9312 55C4 5391 54CE 54C0 5509 57C3 6328 6B38
6EBE 953F 9384 5540 6371 7691 51D2 5D66 6EB0 560A
6571 6573 769A 764C 6BD0 6639 5A3E 55F3 77EE 853C
8EB7 566F 6FED FA1F 85F9 8B6A 972D 9744 827E 4F0C
7231 7839 784B 9698 55CC 5867 5AD2 611B 788D 53C6
66A7 7477 50FE 58D2 5B21 61D3 8586 61DD 66D6 74A6
8CF9 9932 9D31 76A7 77B9 99A4 7919 8B7A 9440 9C6B
9749 5B89 4F92 5CD6 6849 6C28 5EB5 83F4 8C19 5A95
843B 844A 75F7 8164 9E4C 84ED 8A9D 978C 978D 76E6
8AF3 95C7 99A3 9B9F 76EB 9D6A 97FD 9D95 73B5 557D
96F8 57B5 4FFA 5535 57EF 94F5 968C 63DE 667B 7F6F
92A8 72B4 5CB8 6309 6D1D 834C 6848 80FA 8C7B 5813
5A69 6697 8C8B 5111 930C 9EEF 80AE 9AAF 5C87 6602
663B 678A 76CE 91A0 51F9 5773 5787 67EA 8EEA 720A
6556 53AB 969E 55F7 55F8 5D85 5ED2 6EF6 7352 7353
9068 646E 71AC 7488 851C 78DD 7FF1 8071 87AF 7FF6
8B37 7FFA 9CCC 93D6 9C32 9DD4 9F07 629D 82BA 62D7
8884 5AAA 957A 5ABC 8956 5C99 6277 5CB0 50B2 5961
5965 5AEF 6160 9A9C 5967 6F9A 58BA 5DB4 6FB3 61CA
64D9 8B38 93CA 9A41 516B 4EC8 5DF4 53ED 6252 6733
7390 5427 593F 5C9C 82AD 75A4 54F5 634C 7B06 7C91
7D26 7F93 8686 91DF 8C5D 9C83 9B5E 53D0 72AE 629C
577A 59AD 62D4 8307 70A6 7679 80C8 91DB 83DD 8A59
8DCB 8EF7 98B0 9B43 58A2 9F25 628A 94AF 9200 9776
575D 5F1D 7238 57BB 7F62 E83D 8DC1 9C85 7F77 E83C
9B81 8987 77F2 9738 58E9 705E 6B1B 6300 63B0 767D
767E 4F70 67CF 6822 636D 7AE1 7CA8 7D54 6446 64FA
896C 5457 5E8D 62DD 8D25 62DC 5504 6557 7308 7A17
7CBA 9781 85AD 8D01 97DB 5161 74F8 6273 653D 670C
73ED 822C 9881 6591 642C 6592 9812 7622 8781 878C
8929 764D 8FAC 962A 5742 5C85 6604 677F 7248 74EA
94A3 7C84 8228 9211 8742 9B6C 95C6 529E 534A 4F34
626E 59C5 6011 62CC 7ECA 79DA 6E74 7D46 9261 977D
8FA6 74E3 90A6 5CC0 57B9 5E2E 6360 6886 6D5C 90AB
5E47 5E5A 7E0D 5E6B 97A4 7ED1 7D81 699C 7253 8180
73A4 868C 508D 68D2 68D3 7865 8C24 585D 5FAC 7A16
84A1 872F 78C5 9551 8255 8B17 938A 52F9 5305 4F68
5B62 82DE 80DE 525D 7B23 7172 9F85 8554 8912 95C1
8943 9F59 7A87 5AD1 96F9 5B9D 6009 9971 4FDD 9E28
73E4 5821 5822 5AAC 8446 5BDA 98F9 98FD 8913 99C2
9CF5 7DE5 9D07 8CF2 85F5 5BF3 5BF6 974C 52FD 62A5
62B1 8C79 8DB5 94C7 83E2 888C 5831 924B 9C8D 9764
9AB2 66B4 9AF1 8663 9B91 5124 66D3 7206 5FC1 9464
8421 9642 5351 676F 76C3 686E 60B2 63F9 7891 9E4E
85E3 9D6F 55BA 5317 9273 8D1D 72C8 8C9D 90B6 5907
6601 726C 82DD 80CC 94A1 4FFB 500D 6096 72FD 88AB
505D 5079 6896 73FC 9101 5099 50C3 60EB 7119 7432
8EF0 8F88 6102 789A 7999 84D3 86FD 7295 8919 8A96
9AB3 8F29 92C7 618A 7CD2 97B4 943E 5954 6CCD 8D32
5034 6E00 9029 7287 8CC1 951B 931B 672C 82EF 5959
755A 694D 574C 6379 6873 7B28 64AA 7356 8F3D 4F3B
794A 595F 5D29 7EF7 7D63 958D 5D6D 75ED 5623 7DB3
7E43 752D 57C4 57F2 83F6 7423 742B 979B 6CF5 8FF8
902C 8DF0 5874 750F 955A 8E66 93F0 7680 5C44 506A
6BF4 903C 8C4D 8795 9CBE 939E 9D56 9C0F 67F2 8378
9F3B 5B36 5315 6BD4 5936 673C 4F4A 5421 59A3 6C98
7595 5F7C 67C0 79D5 4FFE 7B14 7C83 7C8A 822D 555A
7B46 9119 805B 8C8F 5302 5E01 5FC5 6BD5 95ED 4F56
5752 5E87 8BD0 90B2 59BC 602D 6788 7540 7541 82FE
54D4 6BD6 73CC 75AA 80C7 835C 965B 6BD9 72F4 7562
8890 94CB 5A62 5EB3 655D 6890 8406 841E 9587 9589
581B 5F3B 5F3C 610A 610E 6E62 7695 7986 7B5A 8A56
8CB1 8D51 55F6 5F43 6945 6ED7 6EED 714F 75F9 75FA
8177 84D6 84FD 870C 88E8 8DF8 8F9F 924D 959F 98F6
5E63 5F0A 719A 7359 78A7 7A2B 7B85 7B86 7DBC 853D
912A 999D 5E64 6F77 7358 7F7C 8945 99DC 9AF2 58C1
5B16 5EE6 7BE6 7BF3 7E2A 859C 89F1 907F 9B85 6583
6FDE 81C2 8E55 979E 9AC0 5970 74A7 9128 9946 7E74
895E 8963 93CE 97B8 97E0 8E83 8E84 9B53 8D14 9434
9A46 9DDD 9DE9 9F0A 8FB9 782D 7B3E 7335 7F16 8439
7178 7251 7502 7BAF 7DE8 8759 7371 9089 937D 9CCA
908A 97AD 9BFE 9BFF 7C69 709E 8D2C 6241 7A86 533E
8CB6 60FC 78A5 7A28 890A 7CC4 9D18 85CA 535E 5F01
5FED 6283 6C73 6C74 82C4 91C6 5CC5 62DA 4FBF 53D8
5909 662A 898D 5FA7 63D9 7F0F 904D 959E 8FA1 7DF6
8251 9828 8FA7 8FA8 8FA9 8FAA 8FAB 8FAE 8FAF 8B8A
706C 6753 5F6A 6807 98D1 9A89 9ADF 6DF2 730B 813F
5882 5E56 6EEE 8508 98AE 9AA0 6A19 719B 8198 9E83
762D 9556 98D9 98DA 5126 98B7 700C 85E8 8B24 7202
81D5 8D06 93E2 7A6E 9573 98C6 98C7 98C8 98CA 9A43
9463 9A6B 8868 5A4A 88F1 8AD8 893E 9336 6AA6 4FF5
647D 9CD4 9C3E 618B 9CD6 9C49 9F08 864C 9F9E 5225
522B 5487 8382 86C2 5FB6 8952 87DE 8E69 762A 765F
5F46 6C43 90A0 780F 5BBE 5F6C 50A7 658C 6915 6EE8
7F24 69DF 7478 8C69 8CD3 8CD4 9554 5110 6FD2 6FF1
6FF5 8668 8C73 74B8 7015 9726 7E7D 8819 944C 986E
6C1E 6448 6BA1 8191 9AE9 64EF 9B02 6BAF 81CF 9ACC
9B13 9AD5 9B22 51AB 4ECC 6C37 51B0 5175 681F 63A4
68B9 92F2 6AB3 4E19 90B4 9643 6032 62A6 79C9 82EA
661E 663A 67C4 70B3 997C 772A 7A89 86C3 68C5 7980
9235 927C 9786 9905 9920 71F7 5E76 4E26 4F75 5E77
57AA 5EB0 5002 6824 75C5 7ADD 504B 50A1 5BCE 6452
8A81 9BA9 9750 7676 62E8 6CE2 7677 73BB 5265 76CB
7835 889A 88AF 94B5 997D 5575 7D34 7F3D 8116 83E0
88B0 7886 9262 50E0 5D93 64A5 64AD 9911 78FB 8E73
9A4B 9C4D 4EE2 4F2F 5B5B 72BB 9A73 5E1B 6CCA 72DB
74DD 82E9 4FBC 52C3 67ED 80C9 90E3 4EB3 632C 6D61
74DF 79E1 94B9 E84B 94C2 6872 6DFF 8236 535A 6E24
6E50 8467 9E41 613D 640F 733C 9238 9251 998E 9C8C
50F0 717F 7254 7B94 818A 824A 999B 99C1 8E23 92CD
9548 58C6 8584 999E 99EE 9B8A 894F 8C70 5697 61EA
7921 7C19 939B 993A 9D53 72A6 9AC6 9AC9 6B02 896E
7934 946E 86BE 8DDB 7BA5 7C38 5B79 64D8 6A97 7CEA
8B52 8617 8514 5CEC 5EAF 900B 94B8 6661 923D 8AA7
9914 8F50 91AD 535C 535F 8865 54FA 6355 88DC 9CEA
735B 9D4F 9E14 4E0D 5E03 4F48 5425 6B65 5498 6016
6B68 6B69 949A 52CF 57D7 6091 6357 8379 90E8 57E0
74FF 9208 5ECD 8500 8E04 90F6 7BF0 9922 7C3F 5693
64E6 6503 7924 7938 906A 56C3 5072 731C 624D 6750
8D22 8CA1 621D 88C1 7E94 91C7 5038 57F0 5A47 5BC0
5F69 63A1 776C 8DF4 7DB5 8E29 83DC 68CC 8521 7E29
4E72 53C2 53C3 53C4 98E1 9A96 53C5 55B0 6E4C 50AA
5B20 9910 9A42 6B8B 8695 60ED 6B98 615A 8745 6159
8836 883A 60E8 6701 6158 5646 61AF E835 7A47 9EEA
9EF2 707F 7CB2 510F 6FAF 8592 71E6 74A8 8B32 7218
4ED3 4EFA 4F27 6CA7 82CD 9E27 5009 8231 5096 51D4
5D62 6EC4 734A 84BC 6FF8 8259 87A5 7F49 9DAC 5328
8535 85CF 6B0C 9476 8CF6 64A1 64CD 7CD9 66FA 66F9
5608 5D86 6F15 84F8 69FD 893F 825A 87AC 93EA 8279
8278 8349 613A 61C6 9A32 808F 9135 8959 9F1C 518A
518C 4FA7 5395 607B 62FA 6D4B 835D 6547 755F 5074
53A0 7B27 7CA3 8417 5EC1 60FB 6E2C 7B56 8434 7B5E
7B74 84DB 5884 7BA3 61A1 7C0E 5D7E 81A5 5C91 68A3
6D94 7B12 564C 5C42 5C64 5D92 7AF2 9A53 8E6D 785B
7873 5CBE 7320 4E7D 53C9 6260 6271 8286 6748 809E
81FF 633F 8A0D 505B 55CF 63D2 63F7 9987 929F 9538
8256 7580 9364 9388 9937 79C5 579E 67E5 67FB 832C
8336 5D56 643D 7339 976B 69CE 8A67 5BDF 78B4 8928
6AAB 8869 8E45 9572 9454 597C 6C4A 5C94 4F98 8BE7
524E 59F9 5DEE 7D01 8A6B 62C6 9497 91F5 72B2 4FAA
67F4 7961 8C7A 558D 5115 831D 867F 52D1 8883 7625
8806 56C6 8FBF 89C7 68B4 63BA 6400 8998 88E7 647B
92D3 5E68 895C 6519 5A75 8C17 5B71 68CE 6E79 7985
998B 5B0B 7158 7F20 50DD 7351 8749 874A 8A97 92CB
5103 5EDB 6F79 6F7A 7DFE 78DB 79AA 6BDA 913D 9561
700D 87EC 5133 5296 7E75 87FE 9141 56B5 58E5 5DC9
703A 6B03 7E8F 7E92 8E94 9575 826C 8B92 9471 995E
4EA7 522C 65F5 4E33 6D50 5257 8C04 7522 7523 94F2
9610 8487 5277 5D7C 644C 6EFB 5E5D 8546 8AC2 95B3
71C0 7C05 5181 7E5F 91A6 8B42 93DF 95E1 56C5 705B
8B87 5FCF 785F 6472 61F4 98A4 61FA 7FBC 97C2 986B
4F25 660C 5000 5A3C 6DD0 7316 83D6 960A 667F 6919
7429 88EE 9520 9329 95B6 9CB3 9BE7 9F1A 957F 4EE7
514F 80A0 82CC 9577 9578 5C1D 507F 5E38 5F9C 74FA
8407 751E 8178 5617 5872 5AE6 747A 8193 92FF 511F
5690 87D0 9CBF 93DB 9C68 5382 573A 6636 60DD 5834
655E 50D8 53B0 5EE0 6C05 92F9 6005 739A 7545 5021
9B2F 5531 60B5 7452 66A2 757C 8AAF 97D4 6284 5F28
600A 6B29 949E 8A2C 712F 8D85 9214 7E5B 724A 6641
5DE2 5DE3 671D 911B 9F0C 6F05 5632 6A14 6F6E 7AB2
7F7A 8F48 9F02 8B3F 5435 7092 7727 717C 9EA8 5DD0
4EE6 4EEF 8016 89D8 8F66 4F21 8ECA 4FE5 7817 5513
8397 7868 86FC 626F 5056 64A6 5972 5C6E 5F7B 577C
8FE0 70E2 70F2 8045 63A3 7869 9819 5FB9 64A4 6F88
52F6 77AE 7221 62BB 90F4 68FD 741B 55D4 7D9D 778B
8AC3 8CDD 8B13 5C18 81E3 5FF1 6C89 8FB0 9648 8FE7
831E 5BB8 70E5 8380 8390 9673 6550 6668 8A26 8C0C
8ED9 6116 63E8 9202 7141 852F 5875 6A04 760E 9703
87B4 8AF6 85BC 9E8E 66DF 9DD0 8DBB 7876 789C 588B
5926 78E3 8E38 8D02 95EF 886C 75A2 79F0 9F80 8D81
8D82 6987 7A31 9F53 9F54 512D 56AB 8C36 6AEC 896F
8B96 9637 6CDF 8670 67FD 722F 68E6 6D7E 5041 86CF
94DB 725A 7424 8D6A 6186 645A 9757 6490 6491 7DFD
6A55 77A0 8D6C 9833 6A89 7AC0 7A6A 87F6 93F3 93FF
9953 943A 4E1E 6210 673E 5448 627F 67A8 8BDA 90D5
4E57 57CE 5A0D 5BAC 5CF8 6D06 837F 4E58 57D5 6330
73F9 8100 6381 73F5 7A9A 812D 94D6 5818 60E9 68D6
6909 7A0B 7B6C 7D7E 88CE 584D 5856 6E97 7880 8AA0
757B 9172 92EE 6195 6F82 6F84 6A59 6A99 9BCE 7013
61F2 9A2C 4FB1 5F8E 609C 901E 9A8B 5EB1 7748 9A01
79E4 5403 599B 6758 4F99 54E7 5F68 80F5 86A9 9E31
74FB 7735 7B1E 7C9A 55AB 8A35 55E4 5AB8 645B 75F4
7D7A 5644 779D 8ABA 87AD 9D1F 9D44 7661 9B51 9F5D
6521 9EB6 5F72 9ED0 5F1B 6C60 9A70 5758 8FDF 5CBB
6CDC 830C 6301 7AFE 834E 4FFF 6B6D 5319 6DD4 801B
86B3 8D7F 7B42 8CBE 9045 8D8D 905F 99B3 5880 6F26
8E1F 9072 7BEA 8B18 5C3A 53FA 544E 8094 4F88 5376
9F7F 5791 62F8 80E3 6065 803B 8687 88B3 8C49 6B3C
6B6F 88B2 88ED 9279 892B 9F52 5F73 53F1 65A5 707B
8D64 996C 62B6 8FE3 52C5 605C 70BD 7FC4 7FC5 6555
70FE 75D3 557B 6E41 98ED 50BA 75F8 815F 8DEE 9253
96F4 618F 7FE4 906B 9290 6157 761B 7FE8 71BE 61D8
7CE6 8DA9 994E 9D92 9DD8 5145 51B2 5FE1 6C96 833A
6D7A 73EB 7FC0 8202 5603 644F 5FB8 6183 61A7 885D
7F7F 825F 8E56 866B 5D07 5D08 9680 6F34 8908 7DDF
8769 87F2 721E 5BA0 57EB 5BF5 94F3 63F0 9283 62BD
E837 7D2C 640A 7633 7BD8 72A8 72AB 4EC7 4FE6 5E31
6826 60C6 7EF8 83D7 6906 7574 7D52 6101 7697 7A20
7B79 88EF 8A76 9167 916C 7DA2 8E0C 5114 96D4 5B26
5E6C 61E4 85B5 71FD 96E0 7587 7C4C 8E8A 91BB 8B8E
8B90 4E11 4E12 541C 677D 4FB4 5062 7785 919C 77C1
9B57 81ED 81F0 905A 6BA0 51FA 5C80 521D 6474 6A17
8C99 9F63 520D 9664 82BB 53A8 6EC1 84A2 8C60 9504
698B 8021 84AD 870D 8D8E 924F 96CF 7293 854F 5EDA
7BE8 92E4 6A71 61E8 5E6E 6AC9 87F5 8E87 96DB 6AE5
8E70 9DB5 8E95 6775 7840 6918 50A8 696E 7987 695A
891A 6FCB 5132 6A9A 74B4 790E 9F6D 9F7C 4E8D 51E6
5904 7ACC 6035 62C0 7ECC 8C56 6B2A 7AD0 4FF6 654A
57F1 73FF 7D40 8655 5097 7421 9110 6410 6EC0 89E6
8E00 95A6 510A 563C 8AD4 61B7 6A7B 65B6 6B5C 81C5
9EDC 89F8 77D7 640B 8197 63E3 E852 E850 555C 562C
8E39 5DDB 5DDD 6C1A 7A7F 5276 744F 4F20 8221 8229
8239 570C 732D 9044 50B3 693D 6B42 66B7 7BC5 8F32
821B 8348 5598 50E2 6C4C 4E32 7394 948F 91E7 8CD7
9DA8 5205 56F1 75AE 7A93 7A97 724E 6450 7255 7621
7ABB 5E8A 7240 5647 50B8 6F3A 78E2 95D6 521B 6006
5231 524F 5259 5275 6134 5439 708A 9FA1 5782 5015
57C0 6858 9672 6376 83D9 6425 68F0 8144 69CC 9524
7BA0 9318 939A 9840 65FE 6776 6625 8405 583E 5A8B
6699 693F 69C6 7443 7BBA 877D 6A41 8F34 6AC4 E858
9C06 9D9E 7EAF 9659 5507 6D71 7D14 83BC 6DF3 8123
6E7B 7289 6EE3 8493 9E51 6F18 84F4 9187 9195 931E
9BD9 9D89 5046 8436 60F7 7776 8CF0 8E33 8822 8E14
6233 8FB6 8FB5 5A15 5A16 60D9 6DB0 7EF0 9034 814F
8F8D 916B 7DBD 8DA0 8F1F 9F8A 64C9 78ED 6B60 56BD
9F6A 9461 9F71 5472 73BC 75B5 8D80 5068 7E12 9AB4
8BCD 73C1 5790 67CC 7960 8308 8328 5832 74F7 8A5E
8F9D 6148 7506 8F9E 9236 78C1 96CC 9E5A 7CCD 8FA4
98FA 9908 5B28 6FE8 858B 9D1C 7920 8800 8FAD 9DBF
9DC0 6B64 4F4C 6CDA 7689 8DD0 673F 6B21 4F7D 523A
523E 5E9B 8326 6828 83BF 7D58 86D3 8D50 8786 8CDC
55ED 4ECE 5306 56EA 82C1 5FE9 679E 8310 6031 5F9E
60A4 68C7 7127 8471 6964 6F17 8061 84EF 8525 9AA2
66B0 6A05 6A2C 719C 747D 7481 7DEB 8066 806A 779B
7BF5 8070 87CC 7E71 93E6 9A18 9A44 4E1B 5F93 5A43
5B6E 5F96 60B0 6DD9 742E 6152 6F0E 6F40 6F48 6F68
8AB4 8CE8 8CE9 6A37 85C2 53E2 7047 6B09 721C 6181
8B25 51D1 6E4A 6971 8160 8F8F 8F33 7C97 89D5 9E81
9E84 9EA4 5F82 6B82 4FC3 731D 5AA8 9162 7604 851F
8A8E 8D97 5648 61B1 8E27 918B 762F 7C07 7E2C 8E59
9F00 8E74 8E75 9863 6C46 64BA 9569 8E7F 651B 8EA5
9479 6505 6AD5 5DD1 6B11 7A73 7A9C 71B6 7BE1 6BA9
7BF9 7C12 7AC4 7228 5D14 50AC 51D7 7F1E 5894 5D89
615B 6467 69B1 69EF 7355 78EA 7E17 93D9 4E7C 6F3C
7480 8DA1 76A0 4F1C 5FF0 75A9 5005 7D23 7FC6 8103
8106 5550 555B 60B4 6DEC 8403 6BF3 7120 7601 7CB9
7DB7 7FE0 81B5 81AC 7AC1 894A 9847 81CE 90A8 6751
76B4 58AB 6F8A 7AF4 5B58 62F5 8E06 520C 5FD6 5BF8
540B 7C7F 6413 7473 9073 78CB 64AE 8E49 919D 9ACA
8658 5D6F 5D73 75E4 7749 77EC 84AB 8516 9E7E 9E7A
9F79 811E 5249 5252 539D 590E 632B 839D 83A1 63AA
902A 68E4 9509 84CC 9519 92BC 932F 5491 54D2 8037
7563 642D 55D2 8921 5660 58B6 6498 939D 8FBE 8FCF
8FD6 547E 59B2 601B 6C93 57AF 709F 7F8D 8345 8359
5273 5312 7B2A 9039 6E9A 7B54 8A5A 9054 8DF6 7629
977C 8598 9791 71F5 7E68 87FD 9389 8E82 943D 97C3
9F96 9F98 6253 5927 4EA3 7714 6A7D 5446 5454 7343
61DB 6B79 50A3 4EE3 6C4F 8F6A 4FA2 5788 5CB1 5E12
7519 7ED0 8FE8 5E26 5F85 6020 67CB 6B86 73B3 8D37
5E2F 8ED1 57ED 5E36 7D3F 86AE 888B 8EDA 902E 8CB8
8EE9 7447 5ED7 53C7 66C3 7DFF 9B98 9D0F 6234 825C
9EDB 7C24 8E5B 703B 9734 8976 9EF1 9746 4E39 5989
5355 62C5 5358 7708 7803 803C 803D 90F8 8043 8EAD
9156 55AE 5A85 6B9A 7605 5330 7BAA 891D 9132 9815
510B 52EF 64D4 6BAB 7649 894C 7C1E 8078 4F14 5210
72DA 73AC 74ED 80C6 8874 75B8 7D1E 63B8 4EB6 99BE
64A3 6FB8 9ED5 81BD 65E6 4F46 5E0E 6C8A 6CF9 8BDE
67E6 758D 8A11 5556 5557 5F39 60EE 6DE1 840F 86CB
557F 6C2E 8145 8711 89DB 7A9E 8A95 50E4 5649 9AE7
563E 5F48 619A 61BA 6FB9 79AB 9924 99F3 9D20 7514
765A 56AA 8D09 972E 994F 5F53 73F0 88C6 7B5C 7576
5105 5679 6FA2 74AB 8960 7C39 8261 87F7 6321 515A
8C20 64CB 8B61 9EE8 6529 7059 6B13 8B9C 6C39 51FC
5735 5B95 7800 57B1 8361 6863 83EA 5A78 74FD 903F
96FC 6F52 78AD 778A 8569 8DA4 58CB 6A94 7497 76EA
7911 7C1C 862F 95E3 5200 5202 53E8 5C76 5FC9 6737
6C18 8220 91D6 9C7D 9B5B 636F 5BFC 5C9B 9666 5012
5BB2 5CF6 6363 7977 7982 6417 969D 5D8B 5D8C 69DD
5C0E 96AF 58D4 5DB9 64E3 8E48 79B1 5230 60BC 7118
76D7 83FF 6921 76DC 9053 7A32 7FE2 5675 7A3B 885C
6AA4 885F 71FE 7FFF 8EC7 74D9 7E9B 6074 5F97 6DC2
60B3 60EA 951D 561A 5FB3 5FB7 9340 7684 63FC 6265
627D 706F 767B 8C4B 5654 5B01 71C8 7492 7AF3 7C26
8260 89B4 8E6C 7B49 6225 9093 50DC 51F3 9127 96A5
58B1 5D9D 77AA 78F4 956B 6AC8 9419 4EFE 4F4E 5943
5F7D 889B 5572 57DE 7F9D 9684 5824 8D86 5600 6EF4
955D 78FE 9349 97AE 93D1 5EF8 72C4 8091 7C74 82D6
8FEA 5519 654C 6DA4 837B 6891 7B1B 89CC 976E 6ECC
9AE2 5AE1 850B 8510 9814 9B61 6575 7BF4 5681 85E1
8C74 7CF4 89BF 9E10 6C10 538E 8BCB 90B8 963A 5467
577B 5E95 5F24 62B5 62DE 67E2 7274 7825 638B 83E7
89DD 8A46 8EE7 805C 9AB6 9BF3 5730 5F1A 5754 5F1F
65F3 6755 7393 601F 67A4 82D0 4FE4 5E1D 57CA 5A23
9012 9013 5059 5547 688A 710D 7731 7976 7B2C 83C2
8C1B 91F1 5A82 68E3 7747 7F14 8482 50C0 7998 8163
905E 926A 99B0 5891 58AC 6455 78B2 8515 8743 9070
6178 750B 7DE0 5DB3 8AE6 8E36 87AE 55F2 6541 6382
508E 53A7 5D6E 6EC7 69D9 7628 98A0 8E4E 5DC5 985A
985B 766B 5DD3 5DD4 6527 7672 9F7B 5178 594C 70B9
5A70 655F 6923 7898 84A7 8547 8E2E 9EDE 7535 4F43
7538 963D 576B 5E97 57AB 6242 73B7 94BF 5538 5A5D
60E6 6DC0 5960 7414 6BBF 8714 923F 96FB 588A 58C2
6A42 6FB1 975B 78F9 765C 7C1F 9A54 5201 53FC 6C48
521F 866D 51CB 595D 5F34 5F6B 86C1 7431 8C82 7889
9CED 6BA6 7797 96D5 9B89 9CB7 7C13 9F26 9BDB 9D70
625A 5C4C 5F14 4F04 540A 9493 7A8E 8A0B 8C03 6389
91E3 94DE 921F 7AE8 84E7 92B1 96FF 8ABF 7639 7AB5
92FD 85CB 9443 7239 8DCC 893A 82F5 8FED 57A4 5CCC
604E 6315 7ED6 80C5 74DE 7723 800A 5551 621C 8C0D
558B 581E 5E49 60F5 63F2 7573 7D70 800B 81F7 8A44
8D83 53E0 6B9C 7243 7252 957B 5D7D 789F 8728 890B
8253 8776 7582 8ADC 8E40 9CBD 66E1 66E2 9C08 7589
758A 6C0E 54CB 6633 7730 56B8 4E01 4EC3 53EE 5E04
738E 753C 7594 76EF 9489 8035 914A 91D8 976A 5975
9876 9802 9F0E 5D7F 9F11 85A1 9424 8BA2 5FCA 9964
77F4 5B9A 8A02 98E3 5576 8423 6917 815A 7887 952D
78A0 8062 8063 9320 78F8 9841 4E1F 4E22 94E5 98A9
92A9 4E1C 51AC 549A 5CBD 6771 82F3 6638 6C21 5032
9E2B 57EC 5A3B 5D2C 6DB7 7B17 83C4 6C2D 8740 9B97
9F15 9BDF 9D87 9DAB 8463 58A5 5B1E 61C2 7BBD 856B
8ACC 52A8 51BB 4F97 578C 59DB 5CD2 5CDD 606B 630F
680B 6D1E 80E8 8FF5 51CD 6219 80F4 52D5 5D20 7850
68DF 6E69 8156 50CD 8A77 99E7 9718 543A 5245 5517
90FD 515C 5160 8538 6A77 7BFC 8254 6597 4E67 9627
6296 6793 94AD 9661 86AA 9204 8C46 90D6 6D62 8373
9017 997E 9B25 68AA 6BED 8130 9158 75D8 9597 7AA6
9B26 92C0 9916 65A3 95D8 7AC7 9B2A 9B2C 9B2D 5262
9607 561F 7763 918F 95CD 6BD2 6D9C 8BFB 6E0E 691F
724D 728A 88FB 8AAD 8773 7368 9316 51DF 5335 5B3B
7006 6ADD 6BB0 7258 72A2 74C4 76BE 9A33 9EE9 8B80
8C44 8D15 97E3 9AD1 945F 97C7 97E5 9EF7 8B9F 53BE
72EC 7B03 5835 5E3E 743D 8D4C 7779 89A9 8CED 7BE4
828F 5992 675C 809A 59AC 5EA6 8370 79FA 6E21 976F
9540 8799 6BAC 934D 8827 8839 8011 5073 5A8F 7AEF
890D 9374 77ED 6BB5 65AD 5845 7F0E 846E 6934 7145
7456 8176 78AB 953B 7DDE 6BC8 7C16 935B 65B7 8E96
7C6A 53FE 5796 5806 5860 5D5F 75FD 78D3 9827 9D2D
941C 961F 5BF9 514A 514C 5151 5BFE 794B 603C 966E
968A 7893 7D90 5C0D 619D 6FE7 85B1 9566 61DF 7029
8B48 9413 8B75 5428 60C7 6566 8733 58A9 58AA 58FF
64B4 7364 5678 6489 6A54 729C 7905 8E72 8E7E 9A50
76F9 8DB8 8E89 4F05 56E4 5E89 6C8C 7096 76FE 7818
9007 949D 987F 9041 920D 815E 9813 78B7 906F 619E
6F61 71C9 8E32 591A 591B 5484 54C6 8324 525F 5D1C
6560 6BF2 88F0 5689 4EDB 593A 94CE 526B 6387 6553
655A 55A5 656A 75E5 922C 596A 51D9 8E31 9BB5 9438
6735 6736 54DA 579B 6305 6306 57F5 7F0D 692F 8D93
8EB1 8EB2 7D9E 4EB8 8EC3 9B0C 56B2 5234 5241 6CB2
964A 964F 9973 579C 5C2E 67EE 6857 5815 8235 60F0
8DE2 8DE5 8DFA 98FF 58AE 5D9E 619C 58AF 9D7D 59B8
59BF 5A3F 5C59 8BB9 542A 56EE 8FD7 4FC4 5A25 5CE8
5CE9 6D90 83AA 73F4 8A1B 7692 774B 920B 9507 9E45
86FE 78C0 8A90 92E8 981F 989D 9B64 984D 9D5D 9D5E
8B4C 6799 7808 5A40 60E1 5641 9A00 9D48 5384 6B7A
5C75 6239 5C8B 9628 5443 627C 82CA 9638 545D 7810
8F6D 54A2 54B9 57A9 59F6 5CC9 530E 6076 7828 8685
997F 5054 537E 580A 60AA 7846 8C14 8EDB 9102 960F
582E 5D3F 6115 6E42 843C 8C5F 8EF6 904C 904F 5EC5
6424 6439 7427 816D 8A7B 50EB 8741 9537 9E57 855A
907B 981E 989A 9913 5669 64DC 89A8 8AE4 95BC 9929
9354 9CC4 6B5E 984E 6AEE 9C10 9D9A 8B8D 9469 9F76
9C77 5940 6069 84BD 717E 5CCE 6441 97A5 4ED2 4E7B
65D5 513F 800C 5150 4F95 5152 9651 5CCF 6D0F 800F
834B 682D 80F9 5532 88BB 9E38 7CAB 804F 8F00 9C95
96AD 9AF5 9B9E 9D2F 8F5C 5C12 5C13 5C14 8033 8FE9
6D31 9975 682E 6BE6 73E5 94D2 723E 927A 990C 99EC
85BE 9087 8DB0 4E8C 5F0D 5F10 4F74 5235 54A1 8D30
8CAE 8848 8CB3 8A80 6A32 53D1 6CB7 767A 767C 5F42
9AEA 6A43 9197 4E4F 4F10 59C2 57A1 75BA 7F5A 8337
9600 6830 50A0 7B4F 7782 7F70 95A5 7F78 85C5 4F71
6CD5 781D 9345 704B 73D0 743A 9AEE 5E06 5FDB 72BF
756A 52EB 5643 58A6 5B0F 5E61 61A3 65D9 65DB 7FFB
85E9 8F53 98BF 7C53 98DC 9C55 51E1 51E2 51E3 5325
674B 67C9 77FE 7C75 9492 8224 70E6 8227 7B32 91E9
68E5 7169 7DD0 6A0A 8543 6A4E 71D4 74A0 81B0 85A0
7E41 894E 7E59 7FB3 8E6F 703F 792C 8629 9407 881C
9DED 53CD 4EEE 6255 8FBA 8FD4 6C3E 72AF 597F 6C4E
6CDB 996D 8303 8D29 7548 8A09 8ED3 68B5 76D5 7B35
8CA9 8EEC 98EF 98F0 6EFC 5B0E 7BC4 5B14 702A 531A
65B9 90A1 574A 82B3 678B 7265 94AB 6DD3 8684 580F
8DBD 9201 933A 9D0B 9632 59A8 623F 80AA 57C5 9C82
9B74 4EFF 8BBF 5F77 7EBA 6609 6618 74EC 7706 5023
65CA 7D21 822B 8A2A 9AE3 9DAD 653E 98DE 5983 975E
98DB 5561 5A53 5A54 6E04 7EEF 83F2 6249 7306 975F
88F6 7DCB 871A 970F 9CB1 9925 99A1 9A11 9A1B 9BE1
98DD 80A5 6DDD 6683 8153 8730 87E6 670F 80D0 532A
8BFD 595C 60B1 6590 68D0 69A7 7FE1 855C 8AB9 7BDA
5420 5E9F 676E 6CB8 72D2 80BA 6632 8D39 4FF7 5255
539E 75BF 5C5D 8409 5EC3 8CBB 75F1 9544 5EE2 855F
66CA 7648 9F23 6FF7 6AE0 9428 9745 5206 5429 5E09
7EB7 82AC 6610 6C1B 73A2 7AD5 886F 7D1B 7FC2 68FB
8A1C 8EAE 915A 9216 96F0 6706 9934 9959 5746 575F
59A2 5C8E 6C7E 678C 7083 80A6 68A4 7F92 86A0 86A1
68FC 711A 84B6 999A 96AB 58B3 5E69 8561 9B75 9CFB
6A68 71CC 71D3 8C6E 9F22 7FB5 9F16 8C76 8F52 943C
99A9 9EC2 7C89 7793 9EFA 4EFD 574B 5F05 594B 5FFF
79CE 507E 6124 7CAA 50E8 61A4 596E 81B9 7CDE 9CBC
7035 9C5D 4E30 98CE 4EF9 51E8 51EC 59A6 6CA3 6CA8
51EE 67AB 5C01 75AF 76FD 781C 98A8 5CEF 5CF0 5051
687B 70FD 7412 5D36 6E22 6E84 7326 8451 950B 6953
728E 8702 760B 78B8 50FC 7BC8 9137 92D2 6A92 8C50
93BD 93E0 9146 5BF7 7043 8634 974A 98CC 9EB7 51AF
5906 6340 6D72 9022 5838 99AE 6453 7D98 7F1D 8242
7E2B 8BBD 8982 552A 8AF7 51E4 5949 752E 4FF8 6E57
7128 7148 8D57 9CEF 9CF3 9D0C 8CF5 8615 74F0 8985
4ECF 4F5B 5772 68BB 57BA 7D11 7F36 5426 599A 7F39
7F3B 96EC 9D00 592B 4F15 909E 544B 598B 59C7 678E
739E 80A4 6024 67CE 7806 80D5 8342 886D 5A10 5C03
8374 65C9 7D28 8DBA 915C 9EB8 7A03 8DD7 9207 7B5F
7D92 911C 5B75 8C67 6577 819A 9CFA 9EA9 7CD0 9EAC
9EB1 61EF 4E40 5DFF 5F17 4F0F 51EB 7536 51B9 521C
5B5A 6276 8299 82A3 82BE 5488 5CAA 5E17 5F7F 602B
62C2 670D 6CED 7EC2 7ECB 82FB 8300 4FD8 5798 67B9
67EB 6C1F 6D11 70A5 73B8 7549 7550 7953 7F58 832F
90DB 97E8 9CEC 54F9 683F 6D6E 7557 7829 83A9 86A8
5310 6874 6DAA 70F0 7408 7B26 7B30 7D31 7D3C 7FC7
8274 83D4 8659 88B1 5E45 68F4 7D65 7F66 844D 798F
7CB0 7D8D 8240 8709 8F90 9258 925C 98AB 9CE7 6991
7A2A 7B99 8907 97CD 5E5E 6F93 8760 9AF4 9D14 8AE8
8E3E 8F3B 9B84 7641 8946 9BB2 9EFB 8965 9D69 9D9D
5452 629A 752B 5E9C 5F23 62CA 65A7 4FCC 90D9 4FEF
91DC 91E1 636C 812F 8F85 6928 7124 76D9 8151 6ECF
8705 8150 8F14 64AB 9B34 7C20 9EFC 961D 7236 8BA3
4ED8 5987 8D1F 9644 5490 577F 7ACE 961C 9A78 590D
5CCA 7954 8A03 8CA0 8D74 86A5 889D 965A 5069 51A8
526F 5A4F 5A66 86B9 5085 5A8D 5BCC 5FA9 79FF 842F
86D7 8984 8A42 8D4B 6931 7F1A 8179 9C8B 79A3 8914
8D59 7DEE 8567 875C 876E 8CE6 99D9 7E1B 8F39 9B92
8CFB 9351 9362 9CC6 8986 99A5 9C12 7324 65EE 4F3D
5620 9486 5C1C 91D3 560E 5676 9337 5C15 738D 5C2C
9B40 4F85 8BE5 90C2 9654 5793 59DF 5CD0 8344 6650
8D45 7561 7974 7D6F 9691 8A72 8C65 8CC5 8CCC 5FCB
6539 7D60 9385 4E10 4E62 5303 5304 675A 9499 76D6
6461 6E89 8462 9223 6224 6982 84CB 69E9 69EA 6F11
74C2 5E72 7518 5FD3 8289 8FC0 653C 6746 7395 809D
5769 6CD4 82F7 67D1 7AFF 75B3 9150 7C93 4E81 51F2
5C32 5C34 7B78 6F27 9CF1 5C36 5C37 9B50 4EE0 76AF
79C6 8866 8D76 6562 687F 7B34 7A08 611F 6F89 8D95
6A44 64C0 7C33 9CE1 9C64 65F0 6C75 76F0 77F8 7EC0
501D 51CE 6DE6 7D3A 8A4C 9AAD 5E79 69A6 6A8A 8D63
8D1B 7068 5188 7F53 51AE 521A 962C 5C97 7EB2 809B
5CA1 7268 7598 77FC 7F38 94A2 525B 7F61 E827 E82D
5808 6386 91ED 68E1 7285 583D 7DB1 7F41 92FC 93A0
5D17 6E2F 6760 7135 7B7B 69D3 6206 6207 768B 7F94
7F99 9AD8 7690 9AD9 81EF 6EDC 776A 69D4 777E 818F
69F9 6A70 7BD9 7CD5 993B 6ADC 97DF 9DCE 9F1B 9DF1
5930 6772 83D2 7A01 641E 7F1F 69C0 69C1 7354 7A3E
7A3F 9550 7E1E 85C1 6ABA 85F3 93AC 543F 544A 52C2
8BF0 90DC 5CFC 796E 7970 9506 7B76 66A0 799E 8AA5
92EF 6208 572A 72B5 7EA5 6213 8090 726B 7599 7271
7D07 54E5 80F3 88BC 9E3D 5272 6401 5F41 6B4C 6ED2
6228 95A4 9D10 9D1A 64F1 8B0C 9D3F 93B6 5444 4F6E
530C 630C 8316 9601 9769 654B 683C 9B32 6105 81F5
845B 86D2 86E4 88D3 9694 55DD 5865 6EC6 89E1 643F
69C5 8188 95A3 9549 9788 97D0 9ABC 8AFD 8F35 9BAF
6ACA 9398 97DA 8F55 97B7 9A14 9C2A 54FF 8238 4E2A
5404 867C 500B 784C 94EC 7B87 927B 7366 7ED9 7D66
6839 8DDF 54CF 4E98 826E 831B 63EF 6404 66F4 522F
5E9A 754A 6D6D 8015 63B6 83EE 6929 713F 7D5A 8D53
9E52 7DEA 7E06 7FAE 8CE1 7FB9 9D8A 90E0 54FD 57C2
5CFA 632D 7EE0 803F 8384 6897 7D86 9CA0 9ABE 9BC1
4E99 5829 5579 55BC 55F0 5DE5 5F13 516C 53B7 529F
653B 675B 4F9B 7CFC 80B1 5BAB 5BAE 606D 86A3 8EAC
9F9A 5311 5868 5E4A 6129 89E5 8EB3 5314 78BD 7BE2
9AF8 89F5 9F8F 9F94 5EFE 5DE9 6C5E 62F1 551D 62F2
6831 73D9 8F01 978F 5171 8D21 7FBE 8CA2 6150 7195
8D11 515D 5163 52FE 4F5D 6C9F 94A9 88A7 7F11 920E
6E9D 9264 7DF1 8920 7BDD 7C3C 97B2 97DD 5CA3 72D7
82DF 67B8 73BD 8007 8009 7B31 8008 86BC 8C7F 5778
6784 8BDF 8D2D 57A2 59E4 8329 5193 591F 5920 8A3D
5ABE 5F40 6406 8A6C 9058 96CA 69CB 7179 89CF 6480
89AF 8CFC 4F30 5495 59D1 5B64 6CBD 6CD2 67E7 8F71
5502 5503 7F5B 9E2A 7B1F 83C7 83F0 86C4 84C7 89DA
8EF1 8EF2 8F9C 9164 6BC2 9232 7B8D 7B9B 5AF4 7BD0
6A6D 9B95 9D23 8F42 9E58 9DBB 53E4 5903 6262 6C69
8BC2 8C37 80A1 5CE0 726F 9AA8 7F5F 7F96 9027 94B4
50A6 5552 6DC8 8135 86CA 86CC 5C33 6132 7138 7872
8A41 9989 9E44 69BE 9237 9F13 9F14 560F 6996 76B7
7A40 7E0E 7CD3 85A3 6FF2 81CC 9936 7014 76EC 77BD
9D60 8831 56FA 6018 6545 51C5 987E 580C 5D13 5D2E
688F 727F 68DD 797B 96C7 75FC 7A12 9522 980B 50F1
932E 9CB4 9BDD 9867 74DC 522E 82FD 80CD 9E39 6B44
713B 7171 98AA 8D8F 5280 7DFA 92BD 98B3 9D30 9A27
518E 53E7 5471 5250 526E 5569 5BE1 5366 576C 8BD6
6302 639B 7F63 7D53 7F6B 8902 8A7F 4E56 62D0 67B4
67FA 7B89 592C 53CF 602A 6060 5173 89C2 5B98 51A0
898C 500C 8416 68FA 8484 7AA4 95A2 761D 764F 89B3
95D7 9CCF 95DC 9C25 89C0 9C5E 9986 742F 75EF 7B66
7BA1 8F28 8218 9327 9928 8E80 9CE4 535D 6BCC 4E31
8D2F 6CF4 60BA 60EF 63BC 6DAB 8CAB 60B9 797C 6163
645C 6F45 9066 6A0C 76E5 7F46 96DA 93C6 704C 721F
74D8 77D4 7936 9E73 7F50 9475 9E1B 9C79 5149 706E
4F8A 7097 709A 709B 54A3 5799 59EF 6D38 832A 6844
70E1 73D6 80F1 7844 50D9 8F04 92A7 9EC6 6B1F 5E7F
5E83 72B7 5EE3 7377 81E9 4FC7 901B 81E6 6497 5F52
572D 59AB 9F9F 89C4 90BD 7688 8325 95FA 5E30 73EA
80FF 4E80 7845 7A90 88BF 898F 5AAF 691D 7470 90CC
5AE2 646B 95A8 9C91 5B00 5DB2 69FB 69FC 749D 77A1
81AD 9BAD 9F9C 5DC2 6B78 9B36 9A29 74CC 9B39 6AF7
5B84 6C3F 8F68 5E8B 4F79 5326 8BE1 9652 579D 59FD
6051 7678 8ECC 9B3C 5EAA 796A 532D 6677 6E40 86EB
89E4 8A6D 53AC 7C0B 87E1 523D 523F 6530 660B 67DC
7085 6531 8D35 6842 6922 7B40 8CB4 6E8E 84D5 8DEA
7786 528A 528C 648C 69F6 77B6 79AC 7C02 6AC3 8958
9CDC 97BC 9C56 9C65 4E28 886E 60C3 7EF2 889E 8F8A
6EDA 84D8 88F7 6EFE 7DC4 8509 78D9 7DF7 8F25 9CA7
9B8C 9BC0 68CD 68DE 7754 7774 74AD 8B34 5459 54BC
57DA 90ED 556F 581D 5D1E 6947 8052 921B 9505 588E
7611 5613 5F49 6FC4 8748 934B 5F4D 87C8 56EF 56F6
56FB 56FD 5700 570B 5E3C 63B4 8158 5E57 6156 6451
6F0D 805D 852E 8195 8662 9998 679C 60C8 6DC9 7313
83D3 9983 6901 8901 69E8 7CBF 7DB6 873E 88F9 8F20
991C 9439 8FC7 904E 8142 598E 94EA 927F 4E37 54C8
548D 55E8 5B69 9AB8 6D77 80F2 70F8 5870 917C 91A2
4EA5 9A87 5BB3 6C26 55D0 9900 99ED 99F4 56A1 995A
4E64 516F 4F44 9878 54FB 86B6 9163 9807 5AE8 8C3D
61A8 99A0 9B7D 9F3E 9097 542B 90AF 51FD 5481 80A3
51FE 8677 5505 5705 5A22 6D5B 5D21 6657 6892 6DB5
7113 5BD2 5D45 97E9 751D 7B68 7233 872C 6F8F 92E1
97D3 5388 7F55 6D6B 558A 850A 8C43 9B2B 6C49 5C7D
625E 6C57 95EC 65F1 57BE 608D 634D 6658 6D86 7302
839F 6665 6DCA 710A 7400 83E1 91EC 9588 7694 7745
50BC 86FF 9894 99AF 6496 6F22 872D 66B5 71AF 92B2
92CE 61BE 64BC 7FF0 8792 9837 9844 99FB 8B40 E821
E820 96D7 701A 862B 9DBE 592F 9B67 5994 82C0 8FD2
65BB 676D 57B3 7ED7 7B10 822A 86A2 9883 8CA5 7B55
7D4E 980F 6C86 8320 84BF 5686 8585 85A7 7AD3 869D
6BEB 6903 55E5 7346 5651 8C6A 5637 734B 512B 66CD
568E 58D5 6FE0 7C47 8814 8B79 597D 90DD 53F7 660A
6626 79CF 54E0 604F 608E 6D69 8017 6667 6DCF 5090
7693 6EC8 8055 865F 66A4 66AD 6F94 769C 769E 76A1
8583 76A5 98A2 704F 9865 9C1D 705D 515E 8BC3 5475
62B2 6B31 559D 8A36 55EC 881A 79BE 5408 4F55 52BE
548A 548C 59C0 6CB3 90C3 5CC6 66F7 67C7 72E2 76C7
7C7A 9602 9978 54EC 6546 6838 76C9 76CD 8377 555D
6DB8 6E2E 76D2 79F4 83CF 8402 86B5 9F81 60D2 7CAD
8A38 988C 6941 6BFC 8A65 8C88 8C89 924C 9616 9C84
7186 95A1 9E56 9EA7 6F95 981C 7BD5 7FEE 879B 9B7A
7909 95D4 97A8 9F55 8988 9DA1 76AC 9449 9FA2 4F6B
578E 8D3A 8894 96BA 5BC9 7103 6E7C 8CC0 55C3 7142
788B 7187 8910 8D6B 9E64 7FEF 58D1 764B 71FA 7200
9DB4 9F43 974D 974E 9E16 974F 9ED2 9ED1 563F 6F76
5B12 62EB 75D5 978E 4F77 5F88 72E0 8A6A 6068 4EA8
54FC 6099 6DA5 811D 59EE 6046 6052 6841 70C6 73E9
80FB 9E3B 6A2A 6A6B 8861 9D34 9D46 8605 9445 5548
583C 56CD 4E4A 4E65 53FF 7074 8F70 54C4 8A07 70D8
8EE3 63C8 6E39 7122 7861 8C3E 85A8 8F37 569D 9367
8F5F 4EDC 5F18 5985 7EA2 5430 5B8F 6C6F 7392 7EAE
95F3 5B96 6CD3 739C 82F0 57AC 5A02 6D2A 7AD1 7D05
836D 8679 6D64 7D18 7FC3 803E 7854 7D2D 8C39 9E3F
6E31 7AE4 7CA0 8452 8453 921C 958E 7D8B 7FDD 8C3C
6F42 9277 9783 9B5F 7BCA 92D0 5F4B 857B 9710 9EC9
971F 9D3B 9ECC 664E 55CA 8BA7 8A0C 95A7 6494 6F8B
6F92 92BE 95C0 95C2 9B28 9F41 4FAF 77E6 5589 5E3F
7334 8454 760A 777A 9297 7BCC 7CC7 7FED 9ABA 936D
9931 9BF8 543C 543D 72BC 540E 90C8 539A 5795 5F8C
6D09 9005 5019 9107 5820 8C5E 9C8E 9C98 9B9C 9C5F
4E4E 5322 864D 547C 5780 5FFD 6612 66F6 6CD8 82F8
6057 70C0 8F77 532B 553F 60DA 6DF4 8656 8EE4 96FD
5611 5BE3 6EF9 96D0 5E60 6B51 81B4 8B3C 56EB 6287
5F27 72D0 74F3 80E1 58F6 58F7 659B 7100 5596 58FA
5AA9 6430 6E56 7322 7D57 846B 695C 7173 745A 561D
851B 9E55 69F2 7BB6 7CCA 8774 885A 9B71 7E20 879C
9190 9836 89F3 9378 992C 702B 9B0D 9C17 9D98 9DA6
9DAE 4E55 6C7B 864E 6D52 552C 8400 7425 865D 6EF8
7B8E 933F 9BF1 4E92 5F16 6236 6237 6238 51B1 51B4
8290 5E0D 62A4 6C8D 6CAA 5CB5 6019 623D 6608 6791
795C 7B0F 7C90 5A5F 6248 74E0 7D94 9120 5AED 5AEE
6462 6EEC 8530 69F4 71A9 9CF8 7C04 9359 569B 9E71
8B77 9CE0 97C4 9800 9C6F 9E0C 82B1 82B2 57D6 5A72
691B 7874 7CC0 8AAE 9335 8624 534E 54D7 59E1 9A85
83EF 94E7 6ED1 733E 5629 64B6 748D 78C6 8550 8796
92D8 8B41 93F5 9A4A 9DE8 5316 5212 6779 753B 8BDD
5D0B 6866 5A73 756B 5B05 7575 89DF 8A71 5283 6466
69EC 6A3A 5AFF 6F85 8AD9 8AE3 9ECA 7E63 8219 8633
6000 5F8A 6DEE 69D0 8922 8E1D 61D0 8931 61F7 7024
6AF0 8032 8639 574F 54B6 58CA 58DE 863E 6B22 6B25
6B53 9D05 61C1 9D4D 9144 56BE 61FD 737E 6B61 8C9B
8B99 9A69 8FD8 73AF 90C7 5CD8 6D39 72DF 8341 6853
8408 8411 581A 5BCF 7D59 96C8 7D84 7FA6 8C86 953E
961B 5BF0 6FB4 7F33 9084 74B0 8C72 9370 956E 9E6E
7CEB 7E6F 8F58 95E4 9436 9B1F 74DB 7746 7F13 8F10
7DE9 650C 5E7B 5942 8092 5950 5BA6 5524 6362 6D63
6DA3 70C9 60A3 6899 7115 902D 559A 559B 5D48 610C
63DB 6E19 75EA 7165 744D 8C62 6F36 7613 69F5 9CA9
64D0 6FA3 77A3 85E7 9BC7 9BF6 9C00 5DDF 8093 8352
8841 671A 5843 614C 7687 505F 51F0 968D 9EC3 9EC4
55A4 582D 5A93 5D32 5FA8 60F6 63D8 6E5F 845F 9051
697B 714C 745D 58B4 6F62 735A 953D 71BF 749C 7BC1
824E 8757 7640 78FA 7A54 8AFB 7C27 87E5 9360 992D
9CC7 8DAA 97F9 9404 9A1C 9C09 9C51 9DEC 6033 604D
70BE 5BBA 6643 6644 595B 8C0E 5E4C 6130 8A64 7180
7E28 8B0A 6ACE 76A9 5164 6EC9 69A5 66C2 769D 93A4
7070 7073 8BD9 54B4 6062 62FB 6325 6D03 867A 8886
6656 70E3 73F2 8C57 5A4E 5A88 63EE 7FDA 8F89 9693
6689 694E 7147 743F 7988 8A7C 5E51 7773 8918 5645
5655 649D 7FEC 8F1D 9EBE 5FBD 96B3 7008 9C34 56D8
56DE 56EC 4F6A 5EFB 5EFD 605B 6D04 8334 8FF4 70E0
8698 9025 75D0 86D4 86D5 8716 9BB0 6094 879D 6BC7
6A93 71EC 8B6D 5349 5C77 6C47 4F1A 8BB3 6CCB 54D5
6D4D 7ED8 8294 835F 8BF2 605A 6075 6867 70E9 70EA
8D3F 5F57 6666 79FD 5599 60E0 6E4F 7D75 7F0B 7FD9
9613 532F 5F59 5F5A 6703 6BC0 6BC1 6ED9 8A6F 8CC4
50E1 5612 7623 8527 8AA8 571A 5BED 6167 6193 66B3
69E5 6F53 8559 5666 5FBB 6A5E 6FAE 7369 74A4 8588
8589 8AF1 982E 6A85 6A9C 71F4 74AF 7BF2 85F1 992F
5696 61F3 77BA 7A62 7E62 87EA 6AD8 7E6A 7FFD 8B53
5136 8B6E 93F8 95E0 5B48 942C 9767 97E2 8B7F 986A
660F 662C 8364 5A5A 60DB 6DBD 960D 60FD 68D4 6B99
8477 7767 776F 95BD 5FF6 6D51 9984 6E3E 9B42 991B
7E49 8F4B 9F32 8BE8 4FD2 5031 5702 638D 6DF7 711D
6EB7 6141 89E8 8AE2 5419 8020 952A 5290 9343 8C41
6509 9A1E 4F78 6D3B 79EE 79F3 706B 4F19 90A9 94AC
9225 6F37 5925 6C8E 6216 8D27 549F 4FF0 6347 7713
83B7 9584 5268 639D 7978 8CA8 60D1 65E4 6E71 798D
55C0 84A6 5684 596F 64ED 6FE9 7372 970D 6AB4 8B0B
96D8 77C6 7A6B 956C 56AF 7016 802F 8267 85FF 8816
56BF 66E4 81DB 7668 77D0 944A 9743 593B 4E0C 8BA5
51FB 5209 53FD 9965 4E69 520F 573E 673A 7391 808C
82A8 77F6 9E21 6785 54AD 8FF9 525E 5527 59EC 5C50
79EF 7B04 98E2 57FA 7EE9 559E 5D46 5D47 6532 6567
7284 7B53 7F09 8D4D 52E3 55D8 7578 7A18 8DFB 9CEE
50DF 6BC4 7B95 9288 5630 6483 69E3 6A2D 757F 7A3D
7DDD 89ED 8CEB 8EB8 9F51 58BC 61BF 6A5F 6FC0 74A3
79A8 7A4D 9324 96AE 64CA 78EF 7C0A 7E3E 7F81 8CF7
913F 6AC5 802D 96DE 8B4F 97F2 9D8F 8B64 9416 9951
766A 8E8B 97BF 9DC4 9F4E 7F87 8640 9447 8989 9459
9F4F 7F88 9E04 898A 4EBC 53CA 4F0B 5409 5C8C 5F76
5FE3 6C72 7EA7 5373 6781 4E9F 4F76 90C6 537D 53DD
59DE 6025 72E4 768D 7B08 7D1A 63E4 75BE 89D9 506E
5359 5EB4 6956 710F 8128 8C3B 6222 68D8 6975 6B9B
6E52 96C6 5849 5AC9 6131 696B 84BA 874D 8D8C 8F91
69C9 8024 818C 92A1 5DAF 6F57 7620 7BBF 8540 857A
8E16 978A 9E61 6A76 6A9D 6FC8 878F 8F2F 894B 8E50
9353 8265 7C4D 8F5A 93F6 9735 9DBA 9DD1 8EA4 96E6
96E7 51E0 5DF1 4E2E 5980 72B1 6CF2 866E 6324 810A
638E 9C7E 5E7E 621F 5D74 9E82 9B62 64A0 64E0 7A56
87E3 5F50 5F51 65E1 8BA1 8BB0 4F0E 7EAA 5756 5993
5FCC 6280 82B0 82B6 9645 5242 5B63 54DC 578D 5CDC
65E2 6D0E 6D4E 7D00 830D 8A08 5264 7D12 7EE7 89CA
8A18 5048 5BC2 5BC4 5F9B 60B8 65E3 689E 6E08 796D
8415 60CE 81EE 846A 8507 517E 75F5 7D99 84DF 88DA
8DE1 969B 588D 66A8 6F03 6F08 799D 7A29 7A4A 8A8B
8DFD 9701 9C9A 66A9 7A37 8AC5 9CAB 5180 5291 66C1
7A44 7E18 858A 8940 9AFB 568C 6A95 6FDF 7E4B 7F7D
89AC 9B86 6AB5 74BE 8E5F 9BFD 9D4B 9F4C 5EED 61FB
7660 7A67 7CED 7E6B 860E 9AA5 9BDA 7031 7E7C 862E
9C40 863B 973D 9C36 9C3F 9C6D 9A65 52A0 5939 593E
5B8A 62B8 4F73 62C1 6CC7 8FE6 67B7 6BE0 6D43 73C8
57C9 5BB6 6D79 75C2 689C 7B33 801E 8888 88B7 50A2
7333 846D 88CC 8DCF 728C 8175 926B 5609 9553 7CD8
8C6D 8C91 93B5 9E9A 573F 6274 5CAC 90CF 835A 90DF
550A 605D 83A2 621B 8125 94D7 621E 86F1 988A 86FA
8DF2 9904 92CF 982C 9830 9D36 9D4A 7532 53DA 73BE
80DB 659A 8D3E 94BE 5A7D 5FA6 659D 6935 8CC8 9240
698E 69DA 7615 6A9F 4EF7 9A7E 67B6 5047 5AC1 5E4F
69A2 50F9 7A3C 99D5 55E7 620B 5978 5C16 5E75 575A
6B7C 95F4 51BF 6214 73AA 80A9 8270 59E6 59E7 517C
76D1 5805 60E4 730F 7B3A 83C5 83FA 8C5C 6E54 724B
728D 7F04 844C 844F 9593 976C 641B 6937 693E 714E
744A 7777 788A 7F23 84B9 8C63 76E3 7B8B 6A2B 719E
7DD8 8551 8573 9CA3 9CFD 9E63 71B8 7BEF 7E11 92FB
8271 97AC 9930 99A2 9E89 7010 97AF 9CD2 6BB1 791B
89B8 9D73 7038 9C14 6AFC 6BB2 8B7C 9C1C 9DBC 7C5B
97C0 9C39 56CF 8643 946F 97C9 56DD 62E3 67A7 4FED
67EC 8327 5039 6338 6361 7B15 51CF 526A 5E34 6898
68C0 6E55 8DBC 63C0 63C3 691C 6E1B 7751 7877 88E5
8A43 950F 5F3F 6695 7450 7B67 7B80 7D78 8C2B 5F45
6229 622C 78B1 5109 7FE6 64BF 6AA2 85C6 8947 8949
8B07 8E47 77BC 7906 7C21 7E6D 8B2D 9B0B 9C0E 9E78
703D 8812 9417 9427 9E7B 8B7E 897A 9E7C 89C1 4EF6
898B 4F9F 5EFA 996F 5251 6D0A 726E 8350 8D31 4FF4
5065 5263 682B 6DA7 73D4 8230 5271 5FA4 6E10 88B8
8C0F 91FC 5BCB 65D4 6957 6BFD 6E85 8171 81F6 8465
8DC8 8DF5 9592 8CCE 9274 952E 50ED 6997 69DB 6F38
528D 528E 58B9 6F97 7BAD 7CCB 8AD3 8CE4 8D9D 8E10
8E3A 5292 5294 6A7A 85A6 8AEB 9375 991E 77AF 77B7
78F5 7900 87B9 9373 64F6 6ABB 6FFA 7E5D 7033 89B5
93E9 807B 8266 8F5E 9451 9452 946C 9473 6C5F 59DC
5C06 8333 6D46 7555 8C47 8441 646A 7FDE 50F5 6F3F
8780 58C3 5F4A 7F30 8591 6A7F 6BAD 87BF 9CC9 7585
7913 7586 7E6E 97C1 9C42 8BB2 5956 6868 508B 848B
52E5 5968 596C 8523 69F3 734E 8029 8199 8B1B 985C
531E 5320 5905 5F1C 6762 964D 6D1A 7EDB 5C07 5F36
88B6 7D73 757A 9171 647E 6EF0 5D79 729F 7CE1 91A4
7CE8 91AC 6AE4 8B3D 827D 8281 4EA4 90CA 59E3 5A07
5CE7 6D47 832D 832E 9A84 80F6 6912 7126 7133 86DF
8DE4 50EC 5604 8660 9C9B 5B0C 5D95 5DA3 618D 6F86
81A0 8549 71CB 81B2 7901 7A5A 9BAB E85D 9D41 9E6A
7C25 87ED 8F47 940E 9A55 9DE6 9DEE 6AF5 81EB 89D2
4F7C 4FA5 6322 72E1 7EDE 997A 6341 6648 70C4 7B05
768E 77EB 811A 94F0 6405 6E6B 7B4A 7D5E 527F 52E6
656B 6E6C 714D 8173 8CCB 50E5 6477 669E 8E0B 9278
9903 510C 528B 649F 64B9 5FBC 657D 657F 7F34 66D2
74AC 77EF 76A6 87DC 9D64 7E73 8B51 5B42 7E90 652A
705A 9C4E 53EB 544C 5CE4 630D 8A06 73D3 7A8C 8F7F
8F83 654E 6559 7A96 6ED8 8F03 5602 5626 65A0 6F16
9175 564D 5DA0 6F50 566D 5B13 7365 85E0 8DAD 8F4E
91AE 8B65 76AD 91C2 9636 7596 7686 63A5 63B2 75CE
79F8 83E8 968E 5588 55DF 5826 5A98 5AC5 63ED 6904
6E5D 813B 8857 716F 7A2D 9782 8754 64D1 7664 9D9B
5369 536A 5B51 5C10 8282 8BA6 5226 5227 52AB 5C8A
6605 523C 52BC 6770 758C 8871 8BD8 62EE 6D01 7ED3
8FFC 5022 6840 685D 83AD 8A10 507C 5A55 5D28 6377
88BA 5091 5AAB 7D50 86E3 9889 5D65 696C 6976 6ED0
776B 7BC0 8710 8A70 9263 9B5D 622A 69A4 78A3 7AED
84F5 9C92 6F54 7FAF 8AB1 8E15 9821 5E6F 64F3 5DBB
64EE 790D 937B 9B9A 5DC0 6AED 881E 8818 883D 5979
59D0 6BD1 5A8E 89E3 89E7 98F7 6A9E 4E2F 4ECB 5424
5C95 5E8E 5FE6 6212 82A5 5C46 5C4A 65BA 73A0 754C
754D 75A5 780E 8878 8BEB 501F 6088 86A7 5FA3 583A
6950 743E 86F6 9AB1 7297 8AA1 892F 9B6A 85C9 7E72
5DFE 4ECA 65A4 9485 5153 91D1 91D2 6D25 77DC 781B
8355 887F 89D4 57D0 73D2 7D1F 60CD 740E 583B 743B
7B4B 5D9C 74A1 9E76 9EC5 895F 4EC5 537A 5DF9 7D27
5807 83EB 50C5 53AA 8C28 9526 5AE4 5ED1 6F0C 76E1
7DCA 84F3 9991 69FF 747E 9326 8B39 9949 4F12 52A4
5C3D 52B2 5997 8FD1 8FDB 4FAD 6783 52C1 6D55 8369
6649 664B 6D78 70EC 8D46 7972 9032 7161 7F19 5BD6
6422 6E8D 7981 9773 5890 616C 7468 50F8 51DA 6B4F
6BA3 89D0 5118 5664 6FC5 7E09 8CEE 568D 58D7 5B27
6FDC 85CE 71FC 74B6 89B2 8D10 9F7D 5755 5759 5DE0
4EAC 6CFE 7ECF 830E 4EB0 79D4 8346 834A 6D87 8396
5A5B 60CA 65CC 65CD 7304 7D4C 83C1 6676 7A09 8148
775B 7CB3 7D93 5162 7CBE 8059 6A78 9CB8 9D5B 9BE8
9D81 E85F 9D84 9E96 9F31 9A5A 9EA0 4E95 4E3C 9631
522D 5753 5B91 6C6B 6C6C 80BC 5244 7A7D 9888 666F
5106 5E5C 61AC 7484 61BC 66BB 71DD 71DE 749F 74A5
9838 87FC 8B66 598C 51C0 5F2A 5F84 8FF3 4FD3 6D44
80EB 501E 51C8 5F33 5F91 75C9 7ADE 9015 5A59 5A67
6871 68B7 6DE8 7ADF 7AEB 811B 656C 75D9 7AE7 9753
50B9 9756 5883 734D 8AA9 8E01 9759 981A 975A 66D4
955C 975C 701E 93E1 7AF6 7AF8 5182 518B 5770 6243
57DB 7D45 99C9 99EB 860F 518F 56E7 6CC2 8FE5 4FB0
70AF 9008 6D7B 70F1 715A 7A98 988E 7D97 50D2 715B
71B2 6F83 71D1 71DB 8927 9848 8614 4E29 52FC 7EA0
673B 725E 7A76 7CFA 9E20 7CFE 8D73 9604 841B 557E
63C2 63EA 63EB 9CE9 644E 6A1B 9B0F 9B2E 4E5D 4E45
4E46 4E63 597A 6C63 6766 7078 7396 820F 97ED 7D24
9152 9579 97EE 531B 65E7 81FC 548E 7542 759A 67E9
67FE 5003 6855 53A9 6551 5C31 5EC4 5313 8205 50E6
5ECF 5ED0 6166 6BA7 820A 9E6B 9BE6 9E94 5336 9F68
9DF2 6B0D 51E5 5727 6285 530A 5C45 62D8 6CC3 72D9
82F4 9A79 5036 6336 6344 75BD 75C0 7717 7820 7F5D
9671 5A35 5A45 5A6E 5D0C 63AC 68AE 6DBA 6910 741A
8152 8D84 8DD4 9514 88FE 96CE 824D 871B 8ACA 8E18
92E6 99D2 9B88 9D21 97A0 97AB 9D8B 5C40 6CE6 4FB7
72CA 6854 6BE9 6DD7 7117 83CA 90F9 6908 6BF1 6E68
7291 8F02 50EA 7CB7 84FB 8DFC 8D9C 8EB9 95B0 6A58
6A8B 99F6 9D59 8E6B 9D74 5DC8 861C E860 9DAA 9F30
9F33 9A67 5480 5F06 6CAE 4E3E 77E9 8392 6319 6907
7B65 6989 6998 849F 9F83 8065 8209 8E3D 64E7 6AF8
9F5F 6B05 8977 53E5 5DE8 8BB5 59D6 5CA0 6007 62D2
6D30 82E3 90AD 5177 601A 62E0 661B 6B6B 70AC 79EC
949C 4FF1 5028 51A3 5267 7C94 801F 86B7 8893 57E7
57FE 60E7 636E 8A4E 8DDD 7123 728B 8DD9 9245 98D3
8661 8C66 952F 5BE0 6133 7AAD 805A 99CF 5287 52EE
5C66 8E1E 9B94 58C9 61C5 64DA 6FBD 7AB6 907D 92F8
5C68 98B6 77BF 8C97 7C34 8E86 91B5 61FC 943B 7220
59E2 5A1F 6350 6D93 8127 88D0 9E43 52EC 92D1 92D7
954C 93B8 9D51 942B 8832 5377 545F 5E23 57CD 6372
83E4 9529 81C7 9308 5946 52B5 5F2E 5026 52CC 684A
72F7 7EE2 96BD 6DC3 74F9 7737 9104 774A 7D6D 7F65
96CB 7760 7D79 98EC 617B 8528 990B 7367 7F82 5658
6485 64A7 5C69 5C6B 4E85 5B52 5B53 51B3 5214 6C12
8BC0 599C 6289 6C7A 82B5 6CEC 73A6 73A8 6317 73CF
7804 7EDD 8673 89C9 5014 6354 6B2E 8697 5D1B 6398
658D 6877 6B8C 7106 8990 89D6 8A23 8D7D 8DB9 902B
5095 53A5 7D55 7D76 899A 8D89 920C 5282 52EA 7474
8C32 99C3 5DA1 5DA5 61B0 6F4F 71A6 7234 7357 761A
855D 8568 9D02 9D03 5671 61A0 6A5B 6A5C 957C 7235
81C4 9562 87E8 87E9 7211 8B4E 8E76 8E77 9D8C 56BC
77CD 89BA 940D 941D 704D 721D 89FC 5F4F 6204 652B
7383 9DE2 6B14 77E1 9FA3 8C9C 8EA9 E84F 9481 519B
541B 5747 6C6E 59F0 8880 8ECD 94A7 8399 8690 687E
76B2 83CC 921E 7885 7B60 76B8 76B9 89A0 9281 929E
9CAA 9E87 9355 9BB6 9E8F 9E95 5441 4FCA 90E1 9656
57C8 5CFB 6343 6659 6D5A 9982 9A8F 710C 73FA 756F
7AE3 7B98 7B9F 8720 5101 5BEF 61CF 9915 71C7 6FEC
99FF 9D54 9D55 9D58 6508 5494 5496 5580 5361 4F67
57B0 80E9 88C3 9272 5F00 5952 63E9 8849 950E 958B
9426 51EF 5240 57B2 607A 95FF 94E0 51F1 5274 6168
8488 584F 6137 6977 8F06 669F 9534 9347 93A7 95D3
98BD 5FFE 708C 708F 6B2C 70D7 52D3 5605 613E 938E
4E6B 520A 681E 52D8 9F9B 582A 5D41 6221 9F95 519A
574E 4F83 780D 83B0 5058 57F3 60C2 583F 6B3F 586A
6B41 8F21 8F41 9851 7AF7 8F57 770B 884E 5D01 5888
961A 77B0 78E1 95DE 77D9 5FFC 780A 7C87 5EB7 5ADD
5D7B 6177 6F2E 69FA 7A45 7CE0 8EBF 93EE 9C47 625B
6443 4EA2 4F09 531F 909F 56E5 6297 72BA 95F6 7095
94AA 9227 958C 5C3B 9ADB 4E02 6537 8003 62F7 6D18
6832 70E4 94D0 7292 92AC 9C93 9760 9BB3 9BCC 533C
5777 82DB 67EF 7241 73C2 79D1 80E2 8F72 75B4 8DB7
94B6 5D59 68F5 75FE 842A 8EFB 988F 6415 7290 7A1E
7AA0 9233 69BC 8596 9897 6A16 778C 78D5 874C 9826
9198 9846 9AC1 791A 58F3 54B3 6BBB 63E2 7FD7 5DB1
53EF 5CA2 70A3 6E07 5D51 6564 6E34 514B 523B 524B
52C0 52CA 5BA2 5CC7 606A 5A14 5C05 8BFE 5801 6C2A
9A92 7F02 55D1 6119 6E98 951E 78A6 7DD9 8AB2 9301
790A 9A0D 808E 80AF 80BB 57A6 6073 5543 8C64 8C87
58BE 9339 61C7 63AF 88C9 8903 52A5 542D 5748 5751
6333 7841 727C 785C 94FF 787B 8A99 92B5 935E 93D7
5DEA 4E6C 551F 53BC 603E 7A7A 5025 57EA 5D06 60BE
6DB3 787F 7B9C 8EBB 8EBC 9313 9D7C 5B54 6050 63A7
979A 5EE4 62A0 82A4 770D 527E 5F44 6473 7798 53E3
52B6 53E9 6263 6010 6542 51A6 5BBC 5BC7 91E6 7A9B
7B58 6EF1 8532 853B 7789 7C06 9DC7 625D 5233 77FB
90C0 67AF 54ED 684D 5800 5D2B 5710 8DCD 7A9F 9AB7
9BAC 72DC 82E6 695B 5E93 4FC8 7ED4 5EAB 79D9 7105
88B4 55BE 7D5D 88E4 7614 9177 8932 56B3 5938 59F1
6647 823F 8A87 4F89 54B5 57AE 9299 630E 80EF 8DE8
9ABB 84AF 64D3 5DDC 51F7 5726 5757 5FEB 4FA9 90D0
54D9 72EF 810D 584A 7B77 E828 9C99 5108 58A4 9136
5672 5EE5 736A 81BE 65DD 7CE9 9C60 5BBD 5BDB 5BEC
9ACB 9467 9AD6 68A1 6B35 6B3E 6B40 7ABD 7ABE 5321
52BB 8BD3 90BC 5329 54D0 6047 6D2D 7B50 7B7A 8A86
8EED 6282 72C2 72C5 8BF3 8ED6 8EE0 8A91 9D5F 593C
5123 61ED 909D 5739 7EA9 51B5 65F7 5CB2 6CC1 77FF
663F 8D36 6846 7716 783F 7736 7D4B 7D56 8CBA 8EE6
9271 92DB 913A 58D9 9ECB 61EC 66E0 720C 77CC 7926
7A6C 7E8A 945B 4E8F 5232 5CBF 609D 76D4 7AA5 8067
7ABA 8667 95DA 985D 5DCB 862C 594E 6646 9035 9108
9804 9997 55B9 63C6 8475 9A99 6223 668C 694F 6951
9B41 777D 8770 982F 6AC6 85C8 9368 9377 9A24 5914
8637 8641 5DD9 8EA8 537C 5080 7143 8DEC 980D 78C8
8E5E 5C2F 532E 6B33 559F 5ABF 6126 6127 6E83 8489
848A 9988 5331 5633 E82E 5B07 6192 6F70 7BD1 8069
E842 806D 8562 6A3B 6BA8 8B09 993D 7C23 E841 8075
7C44 9400 994B 944E 5764 6606 665C 5803 5812 5A6B
5D10 5D11 7311 83CE 88C8 711C 7428 9AE0 88E9 951F
9AE1 9E4D 5C21 6F49 872B 890C 9AE8 71B4 747B 918C
9315 9CB2 81D7 9A09 9BE4 9D7E 9DA4 6083 6346 9603
58F8 68B1 7975 7871 7A07 88CD 58FC 7A1B 7D91 95AB
95B8 56F0 6D83 774F 6269 62E1 62EC 6304 681D 6870
7B48 843F 8440 86DE 9614 5ED3 564B 9822 9AFA 64F4
6FF6 95CA 979F 97D5 61D6 9729 97B9 9B20 7A52 5783
62C9 67C6 5566 7FCB 83C8 641A 908B 65EF 782C 63E6
78D6 5587 85DE 56B9 524C 6E82 814A 63E7 694B 760C
8721 874B 8FA2 8FA3 8772 81C8 650B 7209 81D8 9B0E
6AF4 74CE 9574 9BFB 881F 945E 97A1 6765 4F86 4FEB
5008 5D03 5F95 6D9E 83B1 90F2 5A61 5D0D 5EB2 5FA0
68BE 6DF6 730D 840A 9028 68F6 741C 7B59 94FC 7B82
9338 9A0B 9BE0 9D86 9EB3 553B 8D49 7750 775E 8D56
8CDA 6FD1 8CF4 983C 9842 765E 9D63 7028 702C 7C41
85FE 7669 8970 7C5F 5170 5C9A 62E6 680F 5A6A 5D50
847B 9611 84DD 84DE 8C30 53B1 6F9C 8934 5116 6593
7BEE 61E2 71E3 85CD 8955 9567 95CC 74BC 8964 8B4B
5E71 6514 703E 7046 7C43 7E7F 862D 6595 6B04 7937
8974 56D2 7061 7C63 6B17 8B95 8E9D 897D 946D 9484
97CA 89C8 6D68 63FD 7F06 6984 6F24 7F71 9182 58C8
61D2 89A7 64E5 5B3E 61F6 5B44 89BD 5B4F 652C 6B16
7226 7E9C 70C2 6EE5 71D7 5682 58CF 6FEB 7201 721B
7224 74D3 7060 7CF7 5577 52C6 90CE 90DE 6B34 72FC
83A8 5ACF 5ECA 6879 7405 84C8 6994 746F 7860 7A02
9512 7B64 8246 870B 90D2 8782 8EB4 92C3 93AF 99FA
60A2 6717 9606 6716 70FA 5871 84E2 6A03 8A8F 95AC
6724 57CC 5D00 6D6A 8497 5525 635E 7CA9 6488 52B3
52B4 7262 72EB 7A82 54F0 5520 5D02 6D76 52DE 75E8
94F9 50D7 562E 5D97 61A5 6725 7646 78F1 7C29 87E7
91AA 9412 985F 9ADD 8002 8001 4F6C 54BE 59E5 6045
8356 6833 73EF 7853 94D1 86EF 92A0 6F66 6A51 9BB1
8F51 6D9D 70D9 55E0 8022 916A 5AEA 61A6 6F87 6A6F
802E 8EC2 4EC2 961E 4E50 53FB 5FC7 6250 6C3B 827B
738F 6CD0 7AFB 7833 52D2 697D 97F7 6A02 7C15 9CD3
9C33 9979 990E 96F7 5AD8 7F27 8502 6A0F 757E 6A91
7E32 956D 6AD1 74C3 7FB8 7927 7E8D 7F4D 8632 9433
8F60 58E8 9458 9741 8646 9C69 6B19 7E9D 9F3A 53BD
8012 8BD4 5792 5841 7D6B 50AB 8A84 78CA 854C 78E5
857E 5121 58D8 7657 85DF 6AD0 77CB 7928 7045 881D
863D 8B84 513D 9478 9E13 808B 6CEA 6D21 7C7B 6D99
6DDA 7D2F 9179 9287 981B 982A 64C2 9311 6502 790C
98A3 985E 7E87 8631 79B7 561E 8137 5844 68F1 695E
7890 7A1C 8E1C 8590 51B7 5030 580E 6123 7756 550E
5215 5398 5253 68A8 72F8 79BB 8372 8389 9A8A 60A1
68B8 7281 83DE 55B1 68C3 7282 9E42 527A 6F13 775D
7B63 7F21 8243 84E0 870A 5AE0 5B77 6A06 7483 76E0
7AF0 8C8D 6C02 729B 7CCE 853E 8935 92EB 9CA1 9ECE
7BF1 7E2D 7F79 9305 87CD 8B27 91A8 569F 85DC 908C
91D0 96E2 9BCF 6584 93EB 9BEC 9D79 9EE7 56C4 7055
863A 8821 882B 5B4B 5EF2 5299 9457 7A72 7C6C 9A6A
9C7A 9E1D 793C 674E 91CC 4FDA 5CDB 54E9 5A0C 5CF2
6D6C 9026 7406 88E1 9502 7CB4 88CF 8C4A 92F0 9CA4
6FA7 79AE 9BC9 87F8 91B4 9CE2 9090 9C67 6B1A 529B
5386 5389 5C74 7ACB 540F 6738 4E3D 5229 52B1 5456
575C 6CA5 82C8 4F8B 5CA6 623E 67A5 6CB4 75A0 82D9
96B6 4FD0 4FEA 6803 680E 75AC 7805 8318 8354 8D72
8F79 90E6 5A33 60A7 6817 681B 6835 6D96 7301 73D5
783A 783E 79DD 8385 5533 5A6F 60B7 740D 7B20 7C92
7C9D 86B8 86CE 5088 51D3 53A4 68D9 75E2 86E0 8A48
8DDE 96F3 585B 6144 642E 6EA7 849A 849E 925D 9CE8
53AF 53B2 66A6 6B74 746E 7D9F 8727 52F5 66C6 6B77
7BE5 96B7 9D17 5DC1 6AAA 6FFF 7658 78FF 96B8 9B01
512E 66DE 6AD4 7204 72A1 79B2 8807 56A6 58E2 650A
6ADF 701D 74C5 792A 85F6 9E97 6AEA 720F 74D1 76AA
76ED 792B 7CF2 8823 5137 7667 7930 9148 9DC5 9E9C
56C7 6526 8E92 8F62 6B10 8B88 8F63 652D 74E5 9742
9C71 974B 74C8 4FE9 5006 5AFE 5941 8FDE 5E18 601C
6D9F 83B2 9023 68BF 8054 88E2 4EB7 55F9 5EC9 6169
6E93 6F23 84EE 5332 5969 7191 899D 5286 5333 5652
6190 78CF 8068 806B 8933 9CA2 6FC2 6FD3 7E3A 7FF4
806E 8595 878A 6AE3 71EB 806F 81C1 8E65 8B30 938C
9570 7C3E 880A 8B67 9B11 942E 9C31 7C62 7C68 655B
740F 8138 88E3 6459 69E4 7489 8539 5B1A 6582 6B5B
81C9 913B 895D 7FB7 861D 861E 7EC3 5A08 70BC 604B
6D70 6B93 581C 5AA1 6E45 8430 94FE 50C6 695D 7149
7453 6F4B 7A34 7DF4 6FB0 932C 6BAE 934A 93C8 7032
9C0A 6200 7E9E 7C17 826F 4FCD 51C9 6881 6DBC 690B
8F8C 7CAE 7CB1 589A 7DA1 8E09 6A11 8F2C 7CE7 4E21
4E24 5169 5521 5562 639A 813C 88F2 7DC9 873D 9B49
9B4E 4EAE 54F4 8C05 8F86 55A8 667E 6E78 91CF 7177
8F0C 8AD2 8F1B 9344 8E7D 8FBD 7597 804A 50DA 5BE5
5D7A 5ED6 6180 6F3B 818B 5639 5AFD 5BEE 5D9A 5D9B
61AD 64A9 6579 7360 7F2D 907C 66B8 71CE 7499 7AB7
81AB 7642 7AC2 9563 9E69 5C6A 5EEB 7C1D 7E5A 87DF
8C42 8CFF 8E58 720E 9410 9ACE 98C9 9DEF 948C 91D5
911D 84FC 7212 957D 4E86 5C25 5C26 7093 6599 5C1E
6482 77AD 54A7 6BDF 6318 57D3 5217 52A3 51BD 52BD
59F4 5CE2 6312 6D0C 8322 8FFE 57D2 6D56 70C8 70EE
6369 730E 731F 811F 86DA 88C2 716D 7759 8057 8D94
5DE4 98B2 5120 9BA4 E85E 9D37 64F8 7375 72A3 8E90
9B1B 9B23 9C72 53B8 90BB 6797 4E34 5549 5D0A 60CF
6DCB 667D 7433 7CA6 75F3 7884 7B96 7CBC 9130 96A3
5D99 6F7E 735C 9074 65B4 66BD 71D0 7498 8F9A 9716
77B5 78F7 81E8 7E57 7FF7 9E90 8F54 58E3 7036 93FB
9CDE 9A4E 9E9F 9C57 83FB 4E83 7A1F 50EF 51DB 51DC
649B 5EE9 5EEA 61CD 61D4 6F9F 6A81 6AA9 765B 765D
9872 541D 6061 608B 8D41 711B 8CC3 853A 6A49 7510
81A6 95B5 7584 85FA 8E78 8E8F 8E99 8EAA 8F65 62CE
4F36 5222 7075 56F9 577D 590C 59C8 5CAD 5CBA 5F7E
6CE0 72D1 82D3 6624 670E 67C3 73B2 74F4 51CC 768A
7831 79E2 7ADB 94C3 9675 9E30 5A48 5D1A 6395 68C2
6DE9 740C 7B2D 7D37 7EEB 7F9A 7FCE 8046 8232 83F1
86C9 8851 797E 8A45 8DC9 8EE8 84E4 88EC 9234 959D
96F6 9F84 7DBE 8506 8F18 970A 99D6 6FAA 8576 9302
9717 9B7F 9CAE 9D12 9E77 71EF 971B 971D 9F62 702E
9143 9BEA 5B41 8626 9F61 6AFA 91BD 9748 6B1E 7227
9EA2 9F97 963E 888A 9886 9818 5DBA 4EE4 53E6 5464
70A9 6E9C 7198 5218 6CA0 7544 6D4F 6D41 7559 65C8
7409 7571 786B 88D7 5AB9 5D67 65D2 84A5 84C5 905B
998F 9A9D 69B4 7460 98D7 5289 746C 7624 78C2 954F
99E0 9E60 6A4A 74A2 7581 9560 7645 87C9 99F5 56A0
61F0 700F 85F0 938F 93A6 993E 9E8D 93D0 98C0 9402
9A2E 98C5 9C21 9DB9 9A51 67F3 6801 685E 73CB 687A
7EFA 950D 7DB9 71AE 7F76 92F6 6A6E 7F80 5B3C 516D
7FCF 586F 5EC7 6F91 78DF 9E68 8E53 9724 96E1 98C2
9B38 9DDA 74FC 7505 56D6 54AF 9F99 5C78 5499 6CF7
830F 663D 680A 73D1 80E7 772C 783B 7ADC 7B3C 804B
9686 6E70 5D90 69DE 6F0B 856F 7643 7ABF 7BED 9F8D
56A8 5DC3 5DC4 7027 8622 93E7 9733 66E8 6727 6AF3
7216 74CF 77D3 7931 7932 8971 9F92 7C60 807E 882A
882C 9F93 8C45 8E98 9468 9747 9A61 9E17 9647 5784
5785 62E2 5131 96B4 58DF 58E0 650F 7AC9 54E2 6887
7866 8856 5FBF 8D1A 5A04 507B 5A41 55BD 6E87 848C
50C2 697C 560D 5ED4 617A 851E 9071 6A13 71A1 8027
877C 779C 802C 825B 87BB 8B31 8EC1 9AC5 97BB 9ACF
5D5D 6402 587F 5D81 645F 6F0A 750A 7BD3 7C0D 964B
5C5A 6F0F 7618 9542 E834 763A 763B 93E4 9732 565C
64B8 5695 64FC 5362 5E90 82A6 5786 67A6 6CF8 7089
680C 80EA 8F73 822E 9E2C 7388 823B 9885 9229 9C88
9B72 76E7 56A7 58DA 5EEC 650E 7018 7379 74B7 8606
6AE8 7210 74D0 81DA 77D1 7C5A 7E91 7F4F 826B 8826
8F64 946A 9871 9AD7 9C78 9E15 9EF8 5364 864F 6314
635B 63B3 9E75 7875 9C81 865C 5877 6EF7 84FE 6A10
6F9B 9B6F 64C4 6A79 78E0 9565 7002 6AD3 6C0C 8263
93C0 826A 942A 9465 5725 752A 9646 4F93 5774 5F54
5F55 5CCD 52CE 8D42 8F82 9678 5A3D 6DD5 6DE5 6E0C
7849 83C9 902F 9E7F 6902 742D 797F 7984 50C7 5279
52E0 6EE4 76DD 7769 788C 7A11 8CC2 8DEF 8F05 5876
5ED8 645D 6F09 7B93 7CB6 850D 622E 6A1A 719D 8194
819F 89EE 8DA2 8E1B 8F98 9181 6F5E 7A4B 8557 9304
9332 9334 7490 7C0F 87B0 9D3C 6FFE 7C36 8E57 8F46
9A04 9E6D 7C2C 7C35 93D5 9BE5 9D66 9D71 9E93 93F4
9A3C 7C59 89FB 8642 9DFA 6C07 9A74 95FE 6988 95AD
99BF 6C00 81A2 6ADA 85D8 66E5 9DDC 9A62 5415 5442
4FA3 90D8 4FB6 65C5 68A0 7112 7963 7A06 94DD 5C61
7D7D 7F15 5C62 8182 8190 891B 92C1 5C65 8938 5122
7A5E 7E37 7A6D 5BFD 578F 5F8B 54F7 8651 7387 7EFF
5D42 6C2F 844E 7DA0 7DD1 616E 7BBB 52F4 7E42 6AD6
7208 9462 5B6A 5CE6 631B 683E 9E3E 8114 6EE6 92AE
9D49 571D 5971 5B4C 5B7F 5DD2 6523 66EB 6B12 7053
7F89 81E0 571E 7064 864A 947E 7674 7675 9E1E 5375
4E71 91E0 4E82 63A0 7565 7567 950A 7A24 7A25 5719
92DD 92E2 64FD 62A1 6384 4ED1 4F26 56F5 6CA6 7EB6
4F96 8F6E 502B 966F 5707 5A68 5D18 5D19 60C0 6DEA
83D5 68C6 8140 7896 7DB8 8726 8E1A 8F2A 78EE 9300
9BE9 7A10 8023 8BBA 57E8 6EA3 8AD6 634B 9831 56C9
7F57 5570 7321 8136 841D 903B 6924 8161 9523 7BA9
9AA1 9559 87BA 7F85 89B6 93CD 5138 89BC 9A3E 7380
863F 908F 6B0F 9E01 7C6E 947C 9960 9A58 5246 502E
7822 84CF 88F8 8EB6 7630 8803 81DD 651E 66EA 7673
6CFA 5CC8 6D1B 7EDC 8366 9A86 6D1C 73DE 7B3F 7D61
843D 645E 6F2F 7296 96D2 99F1 9BA5 9D45 6FFC 7E99
9C73 5638 5463 5988 5ABD 5B24 5B37 9EBB 75F2 5AF2
8534 7298 87C6 87C7 9A6C 72B8 739B 7801 8682 99AC
6EA4 7341 9064 746A 78BC 879E 93B7 9DCC 9C22 4E87
6769 7943 9581 9A82 551B 508C 7770 561C 69AA 79A1
7F75 99E1 7923 9B15 5417 55CE 561B 57CB 85B6 973E
4E70 836C 8CB7 562A 8552 9DF6 52A2 8FC8 4F45 58F2
9EA6 5356 8109 8108 9EA5 8847 52F1 8CE3 9081 9721
9722 989F 9862 59CF 6097 86EE 6172 6471 9992 69FE
6A20 7792 779E 9794 9945 9CD7 9B17 9B18 9C3B 883B
5C58 6E80 774C 6EE1 6EFF 87A8 8954 87CE 93CB 77D5
66FC 50C8 8C29 9124 5881 5ADA 5E54 6162 6F2B 734C
7F26 8504 8513 71B3 6FAB 6FB7 9558 7E35 87C3 8B3E
93DD 8630 7264 9099 5402 5FD9 6C52 8292 5C28 6757
6767 76F2 5396 607E 7B00 832B 54E4 5A0F 6D5D 72F5
727B 786D 91EF 94D3 75DD 86D6 92E9 99F9 8609 83BD
83BE 833B 58FE 6F2D 87D2 880E 5301 732B 8C93 6BDB
77DB 6BDC 6BDD 6786 7266 8305 65C4 6E35 8EDE 9155
5825 86D1 951A 7DE2 9AE6 8765 9AF3 9328 87CA 9D9C
5187 536F 5918 623C 5CC1 6CD6 8306 6634 94C6 7B37
84E9 925A 5183 7683 82BC 5190 8302 5192 770A 8D38
8004 88A4 8992 5AA2 5E3D 8CBF 911A 6117 6693 6959
6BF7 7441 7780 8C8C 912E 8750 61CB 551C 5E85 5692
6FF9 56B0 4E48 7666 5445 6C92 6CA1 679A 73AB 82FA
6802 7709 8104 8393 6885 73FB 8122 90FF 5833 5A92
5D4B 6E44 6E48 7338 7742 847F 6963 6973 7164 7442
7996 815C 587A 69D1 9176 9545 9E5B 92C2 9709 5FBE
9387 77C0 6517 862A 9DA5 651F 9EF4 6BCE 6BCF 51C2
7F8E 6334 6D7C 5A84 5D44 6E3C 5ABA 9541 5B0D 71D8
8EBE 9382 9EE3 59B9 62BA 6CAC 6627 7959 8882 771B
5A9A 5BD0 75D7 8DCA 9B3D 715D 7778 97CE 9B45 7BC3
875E 569C 691A 95E8 626A 73A3 9494 9580 9585 636B
83DB 748A 7A48 9346 864B 95F7 7116 60B6 66AA 71DC
61D1 61E3 4EEC 5011 63B9 64DD 6C13 753F 867B 51A1
5EAC 7F5E 8394 840C 8420 5922 6E95 76DF 96FA 750D
511A 6A57 77A2 8544 8771 9133 9138 5E6A 61DE 6FDB
7374 66DA 6726 6AAC 6C0B 77C7 791E 9BCD 8268 9E72
77D2 9740 995B 986D 9E0F 52D0 731B 74FE 8499 9530
824B 8722 9333 61F5 8813 9BED 9F06 5B5F 68A6 5923
61DC 9725 973F 8E0E 54AA 7787 519E 5F25 7962 8FF7
88AE 7315 8C1C 84BE 8A78 8B0E 919A 5F4C 64DF 7CDC
7E3B 9E8A 9E8B 79B0 9761 737C 9E9B 7222 6202 6520
74D5 863C 957E 91BE 91BF 9E0D 91C4 7C73 7F8B 8288
4F8E 6CB5 5F2D 6D23 6549 772B 7C8E 8112 772F 6E33
845E 8746 851D 92A4 6FD4 7030 5B4A 7056 5196 7CF8
6C68 6C95 5B93 6CCC 89C5 5CDA 7955 5BBB 79D8 5BC6
6DE7 8993 8994 5E42 8C27 5853 5E4E 899B 5627 6993
6EF5 6F1E 7190 8524 871C 9F0F 51AA 6A12 5E66 6FD7
85CC 8B10 6AC1 7C1A 7F83 5B80 8287 7720 5A42 7EF5
5A94 68C9 7DBF 7DDC 81F1 8752 5B35 6AB0 6ACB 77C8
77CA 77CF 4E0F 6C45 514D 6C94 9EFE 4FDB 52C9 7704
5A29 506D 5195 52D4 5595 6110 6E4E 7F05 8442 817C
7DEC 9BB8 9763 9762 7CC6 9EAA 9EAB 9EBA 9EB5 55B5
82D7 5A8C 63CF 7784 9E4B 5AF9 9D93 9C59 676A 7707
79D2 6DFC 6E3A 7F08 7BCE 7DF2 85D0 9088 5999 5E99
7385 7AD7 5EBF 5EDF 4E5C 5400 54A9 54F6 5B6D 706D
8995 6423 6EC5 8511 858E 9D13 5E6D 61F1 700E 7BFE
6AD7 881B 884A 9456 9C74 74F1 6C11 578A 59C4 5CB7
5FDE 600B 65FB 65FC 739F 82E0 73C9 76FF 51A7 7F60
5D0F 636A 7418 741D 7F17 668B 7449 75FB 7888 9231
7DCD 7DE1 8CEF 9309 9D16 9372 76BF 51BA 5221 95F5
62BF 6CEF 52C4 6543 95FD 60AF 654F 7B22 7B3D 6E63
9594 610D 656F 9EFD 95A9 50F6 615C 61AB 6F63 7C22
9CD8 8820 9C35 540D 660E 9E23 6D3A 7700 8317 51A5
6719 7733 94ED 910D 5AC7 6E9F 733D 84C2 669D 69A0
9298 9CF4 7791 879F 89AD 4F72 59F3 51D5 614F 9169
547D 63B5 8A7A 8C2C 7F2A 7E46 8B2C 6478 56A4 5C1B
8C1F 5AEB 998D 6479 6A21 819C 9EBC 9EBD 6469 9B79
6A45 78E8 7CE2 8B28 8B29 64F5 9943 56A9 8611 8B55
9ACD 9B54 5298 995D 62B9 61E1 9EBF 672B 52B0 573D
59BA 5E13 603D 6B7E 6B7F 6B81 6CAB 8309 964C 5E1E
6629 67BA 768C 771C 773F 781E 79E3 8388 83AB 773D
7C96 7D48 88B9 86E8 8C83 55FC 587B 5BDE 6F20 84E6
8C8A 9286 977A 58A8 5AFC 66AF 763C 7790 7799 9546
9B69 9ED9 7E38 9ED8 8C98 85E6 87D4 93CC 7205 9A40
7933 7E86 8031 4E6E 54DE 725F 4F94 52BA 6048 6D20
6859 7738 8C0B 927E 8B00 77B4 936A 9D3E 9EB0 87F1
67D0 6BCD 6BEA 734F 58B2 6C01 4EA9 7261 59C6 62C7
5CD4 7273 7546 7552 80DF 5A12 755D 755E 782A 756E
9267 8E07 6728 4EEB 76EE 51E9 6C90 72C7 5776 7091
7267 82DC 6BE3 83AF 869E 94BC 52DF 843A 96EE 5893
5E55 5E59 6154 6958 7766 926C 6155 66AE 6A22 8252
9702 7A46 97AA 65C0 4E06 6927 62CF 62FF 6310 8ABD
954E 93BF 4E78 54EA 96EB 5185 90A3 5436 59A0 7EB3
80AD 5A1C 8872 94A0 7D0D 88A6 637A 7B1D 8C7D 8EDC
8C80 9209 55F1 84B3 9779 9B76 8149 718B 6468 5B7B
4E43 5976 827F 6C16 7593 59B3 5EFC 8FFA 5037 91E2
5B2D 5948 67F0 8010 8418 6E3F 9F10 8926 879A 933C
56E1 7537 62A9 678F 67AC 4FBD 5357 67DF 5A1A 7558
83AE 96BE 5583 9056 6694 6960 7175 8AF5 96E3 8D67
63C7 6E73 8433 8169 877B 6201 5A7B 56D4 4E6A 56A2
56CA 8830 9B1E 9995 6B1C 9962 64C3 66E9 652E 7062
513E 9F49 5B6C 6A82 5476 6013 6320 5CF1 7847 94D9
7331 86F2 8A49 7899 6493 5DA9 7376 87EF 5912 8B4A
9403 5DCE 737F 57B4 607C 60A9 8111 5318 8133 5816
60F1 5AD0 7459 8166 78AF 95F9 5A65 6DD6 9599 9B27
81D1 7592 8BB7 5450 6290 7732 8A25 5462 5A1E 9981
8147 9912 9BBE 9BD8 5167 6C1D 713E 5AE9 80FD 83BB
55EF 922A 92B0 5571 59AE 5C3C 576D 6029 6CE5 7C7E
502A 5C54 79DC 90F3 94CC 57FF 5A57 6DE3 730A 86AD
68FF 8DDC 922E 873A 89EC 8C8E 8F17 9713 9CB5 9BE2
9E91 9F6F 81E1 4F31 4F32 4F60 62DF 62B3 72D4 82E8
67C5 639C 65CE 6672 5B74 9268 999C 5117 511E 96AC
64EC 85BF 6AB7 9448 5C70 6C3C 8FE1 6635 80D2 9006
533F 75C6 7724 5804 60C4 5ADF 6135 6EBA 7768 817B
66B1 7E0C 81A9 5B3A 62C8 5E74 79CA 54D6 79E5 9C87
9B8E 9CB6 9D47 9ECF 9BF0 6D8A 637B 6DF0 8F87 649A
64B5 78BE 8F26 7C10 6506 8E68 8E8E 5344 5EFF 5FF5
59E9 57DD 824C 5A18 5B22 5B43 917F 91B8 91C0 9E1F
8311 8312 8885 9CE5 5ACB 88CA 8526 5B1D 892D 5B32
5C3F 8132 634F 63D1 8080 5E07 573C 82F6 67BF 9667
6D85 8042 81EC 556E 60D7 83CD 9689 55A6 655C 55EB
5D72 9269 8E02 565B 6470 69F7 8E17 8E19 954A 954D
5DAD 7BDE 81F2 931C 989E 8E51 5699 8076 93B3 95D1
5B7C 5B7D 6AF1 7C4B 8616 56C1 9F67 5DD5 7CF1 7CF5
8825 56D3 8B98 8EA1 9477 9873 9480 810C 56DC 60A8
62F0 5B81 549B 62E7 72DE 67E0 804D 5BCD 5BD5 5BDC
5BE7 511C 51DD 5680 5B23 64F0 7370 85B4 6AB8 8079
944F 9B21 9E0B 6A63 77C3 4F5E 4FAB 6CDE 752F 5BD7
6F9D 6FD8 599E 725B 725C 6C7C 5FF8 626D 6C91 72C3
7EBD 677B 7084 94AE 7D10 83A5 9215 9775 519C 4FAC
54DD 6D53 8113 79FE 8FB2 5102 8FB3 5665 61B9 6FC3
857D 79AF 81BF 7A60 895B 91B2 6B01 7E77 5F04 630A
6335 7651 9F48 7FBA 8B68 5542 69C8 8028 7373 6ABD
9392 941E 8B73 5974 4F16 5B65 9A7D 7B2F 99D1 4F2E
52AA 5F29 782E 80EC 6012 5089 6419 5973 9495 7C79
91F9 8842 6067 6712 8844 759F 8650 7627 597B 6E1C
6696 7156 7157 992A 7878 9EC1 71F6 90CD 632A 689B
50A9 643B 513A 6A60 8BFA 558F 63BF 903D 6426 9518
6992 7A2C 8AFE 8E43 7CD1 9369 61E6 61E7 7CE5 7A64
7CEF 685B 5662 54E6 7B7D 6BEE 591E 4E6F 97B0 8BB4
6B27 6BB4 74EF 9E25 5878 6B50 6BC6 71B0 750C 8192
9D0E 6AD9 85F2 8B33 93C2 9DD7 9F75 5418 5455 5076
8162 5614 8026 8545 85D5 6004 6CA4 616A 6F1A 5991
7685 8DB4 8225 556A 8469 6777 722C 8019 63B1 7436
7B62 6F56 5E0A 5E15 6015 8899 62CD 4FF3 5F98 6392
7305 68D1 724C 7B84 8F2B 7C30 72A4 54CC 6D3E 6E43
848E 9383 78D7 7705 7568 6F58 6500 723F 67C8 76D8
8DD8 5ABB 5E4B 84B0 642B 69C3 76E4 78D0 7E0F 8E52
700A 87E0 8E63 939C 97B6 5762 51B8 5224 6C9C 6CEE
708D 53DB 7249 76FC 7554 88A2 8A4A 6EBF 9816 92EC
9D65 897B 947B 4E53 6C78 6C97 80A8 80EE 96F1 6EC2
8196 9736 5390 5E9E 9004 65C1 823D 5ACE 7BE3 8783
9CD1 9F8E 9F90 9C1F 882D 55D9 802A 89AB 9AC8 7090
80D6 629B 62CB 812C 5228 5486 5789 5E96 72CD 70AE
70B0 722E 888D 530F 86AB 8EF3 9784 891C 9E85 8DD1
5945 6CE1 75B1 76B0 7832 8422 9EAD 791F 792E 5478
600C 80A7 67F8 80DA 8843 9185 962B 966A 966B 57F9
6BF0 8D54 952B 88F4 88F5 8CE0 9307 4FD6 4F02 6C9B
4F69 5E14 59F5 65BE 65C6 6D7F 73EE 914D 7B29 84DC
8F94 99B7 5D8F 9708 8F61 55B7 5674 6FC6 6B55 74EB
76C6 6E53 8450 5460 7FC9 7FF8 55AF 5309 6026 62A8
6CD9 6072 80D3 7830 6888 70F9 7851 8EEF 959B 6F30
562D 99CD 78DE 8283 670B 6337 7AFC 5017 8391 580B
5F38 5F6D 68DA 6916 50B0 585C 5873 6412 6F28 787C
7A1D 84EC 9E4F 69F0 6A25 71A2 6189 6F8E 8F23 7BF7
81A8 930B 97F8 9AFC 87DA 87DB 9B05 7E84 97FC 9D6C
9A2F 9B14 945D 6367 6DCE 768F 527B 63BD 692A 78B0
8E2B 6D4C 5DFC 95CF 4E76 55B8 4E15 4F13 4F3E 6279
7EB0 90B3 576F 6036 62AB 62B7 708B 72C9 72D3 7812
6082 79DB 79E0 7D15 94CD 65C7 7FCD 801A 8C7E 91FD
921A 9239 925F 9294 5288 78C7 99D3 9AEC 567C 9303
930D 9B7E 61B5 7914 7915 95E2 9739 76AE 9630 8298
5CAF 6787 6BDE 80B6 6BD7 6BD8 75B2 7B13 868D 90EB
9674 5564 57E4 5D25 86BD 8C7C 6911 7137 7435 813E
8157 698C 9C8F 7F74 818D 8731 96A6 9B6E 58C0 9B8D
7BFA 87B7 8C94 7C32 7F86 9D67 6707 9F19 882F 5339
5E80 4EF3 572E 82C9 8134 75DE 92A2 8AC0 9D04 64D7
567D 7656 56AD 5C41 6DE0 6E12 63CA 5AB2 5AD3 7764
7765 6F4E 50FB 6FBC 568A 7513 7588 8B6C 9DFF E862
9E0A 7247 56E8 504F 5AA5 728F 7BC7 7FE9 9DA3 9A88
80FC 8141 6944 6969 8CC6 8ADA 9ABF 8E41 99E2 9A08
8991 8C1D 8CB5 8ADE 9A97 9B78 9A17 9A19 527D 5F6F
6F02 7F25 98D8 78E6 65DA 7E39 7FF2 87B5 72A5 98C3
98C4 9B52 74E2 85B8 95DD 6B8D 779F 7BFB 91A5 76AB
9860 7968 50C4 52E1 560C 5AD6 5FB1 6153 6C15 6487
6486 66BC 77A5 4E3F 82E4 9405 5AF3 59D8 62FC 7917
7A66 99AA 9A5E 73AD 8D2B 8CA7 7415 5AD4 9891 983B
5B2A 85B2 56AC 77C9 98A6 9870 54C1 6980 6729 725D
6C56 5A09 8058 4E52 7539 4FDC 6D84 782F 8275 7AEE
9829 5E73 8BC4 51ED 546F 576A 5CBC 82F9 90F1 5C4F
5E21 67B0 6D34 73B6 8353 5A26 74F6 5C5B 5E32 6DDC
840D 86B2 5840 5E48 7129 7501 7F3E 8060 84F1 86E2
8A55 8EFF 9C86 51F4 617F 7BB3 8F27 6191 9B83 6A98
7C08 860B 948B 5761 5CA5 6CFC 5A1D 91D9 9887 6E8C
9166 6F51 91B1 93FA 5A46 5619 8522 9131 76A4 8B08
6AC7 53F5 5C00 94B7 7B38 9255 99CA 5EF9 5CB6 8FEB
6540 6622 6D26 73C0 54F1 70DE 7834 7836 7C95 5964
84AA 9817 9B44 5256 9892 6294 6299 634A 638A 88D2
7B81 5485 54E3 5A44 5837 7283 517A 54DB 4EC6 6534
6535 6251 62AA 7087 5DEC 5DED 67E8 9660 75E1 94FA
99C7 5657 64B2 92EA 64C8 9BC6 5724 530D 8386 83E9
83D0 8461 84B1 84B2 50D5 917A 58A3 749E 6FEE 77A8
7A59 9564 8D0C 7E80 93F7 6734 5703 57D4 6D66 70F3
666E 5711 6EA5 669C 8C31 6F7D 6A38 6C06 8AE9 6A8F
9568 8B5C 8E7C 9420 8216 8217 7011 66DD 4E03 8FC9
6C8F 59BB 6053 67D2 501B 51C4 6816 6864 7F3C 90EA
5A38 60BD 621A 637F 687C 6DD2 840B 671E 671F 68F2
6B3A 7D2A 8904 50DB 5601 617D 69BF 69ED 6F06 7DC0
617C 78CE 8AC6 8E26 8AFF 970B 8E4A 9B4C 93DA 9D88
4E93 7941 9F50 573B 5C90 5C93 5FEF 82AA 4E9D 5176
5947 6589 6B67 7947 7948 80B5 75A7 7AD2 5258 658A
65C2 8006 8110 8691 8694 869A 9880 57FC 5D0E 5E3A
6391 6DC7 6E0F 7309 7566 8401 8DC2 8EDD 91EE 9A90
9A91 5D5C 68CA 68CB 7426 742A 797A 86F4 612D 7881
7895 8900 951C 980E 9B3E 9B3F 65D7 7CB8 7DA5 7DA6
7DA8 7DD5 871D 871E 9F4A 7482 79A5 8572 8E11 87A7
9321 9CAF 61E0 6FDD 85C4 6AB1 6AC0 7C31 81CD 9A0E
9A0F 9CCD 8604 9BD5 9D78 9D80 9E92 7C4F 7E83 8269
8810 9B10 9A39 9B55 9C2D 7382 9EA1 4E5E 9094 4F01
5C7A 5C82 8291 542F 5447 675E 7398 76C0 5518 8C48
8D77 5553 5554 555F 5A4D 7EEE 6675 68E8 88FF 7DAE
7DBA 8AEC 7C2F 95D9 6C14 8BAB 6C17 6C54 8FC4 5F03
6C7D 77F5 829E 546E 6CE3 7081 76F5 54A0 5951 780C
8360 6814 6C23 8A16 552D 6B2B 5921 6112 68C4 6E46
6E47 847A 789B 6456 66A3 7508 78B6 5650 6187 5668
61A9 78DC 78E7 78E9 85BA 7918 7F4A 87FF 6390 845C
62E4 8DD2 9160 9790 5736 51BE 5E22 6070 6D3D 6B8E
7848 6118 9AC2 5343 4EDF 9621 5731 5732 5977 6266
6C58 828A 8FC1 4F65 5C8D 6744 6C67 833E 6B26 7ACF
81E4 948E 62EA 7275 7C81 60AD 8688 8C38 94C5 5A5C
5B6F 727D 91FA 6394 8C26 9206 96C3 50C9 6106 7B7E
925B 9A9E 9E50 6173 6434 647C 6481 7B9E 8AD0 9077
8930 8B19 9845 6AB6 6510 6511 6ACF 7C3D 93F2 9D6E
6513 9A2B 9431 9B1C 9B1D 7C64 97C6 4EF1 5C92 5FF4
6272 62D1 4E79 524D 8368 94A4 6B6C 8654 94B1 94B3
4E7E 5042 63AE 63F5 8EE1 5A8A 9210 9246 9257 5898
69A9 7B9D 92AD 6F5C 7FAC 8541 6A6C 9322 9ED4 9386
9EDA 9A1D 6FF3 9A1A 704A 7C56 9C2C 51F5 6D45 80B7
6DFA 5D70 614A 9063 69CF 8181 8738 6F5B 8C34 7F31
7E7E 8B74 9453 6B20 520B 4F23 82A1 4FD4 831C 5029
6093 5811 5094 5D4C 68C8 6920 55DB 7698 84A8 5879
6B49 7DAA 8533 5119 69E7 7BCF 8F24 7BDF 58CD 5B31
7E34 545B 7F8C 6215 6217 65A8 67AA 73B1 7F97 7310
7437 8DC4 55F4 690C 7347 8154 55C6 6EAC 8723 9516
5D88 6227 69CD 7244 7472 7FAB 9535 7BEC 9306 8E4C
956A 8E61 9397 93D8 93F9 5F37 5F3A 5899 5AF1 8537
6A2F 6F12 8503 58BB 5B19 5EE7 8594 6AA3 7246 8B12
8262 8620 62A2 7F9F 6436 7FA5 588F 6464 7E48 8941
7E66 709D 5534 7197 7FBB 515B 74E9 6084 7857 90FB
9117 5D6A 8DF7 9121 9125 5281 6572 8E0D 9539 589D
78BB 981D 9AB9 58BD 5E67 6A47 71C6 7F32 78FD 936B
936C 7E51 7E70 8DAC 8E7A 8E7B 9430 4E54 4FA8 834D
835E 6865 785A 83EC 55AC 7744 50D1 69D7 8C2F 563A
5AF6 6194 854E 9792 6A35 6A4B 729E 7644 77A7 7904
85EE 8B59 8DAB 9408 97BD 9866 5DE7 91E5 6100 9ADC
4FCF 8BEE 9657 5CED 5E29 7A8D 6BBC 7FD8 8A9A 9ADA
50FA 64AC 64BD 9798 97D2 7AC5 7FF9 97A9 8E88 5207
82C6 767F 8304 807A 4E14 5392 59BE 602F 5327 7A83
503F 608F 6308 6D2F 60EC 6DC1 7B21 611C 86EA 6705
7BA7 7DC1 9532 9B65 7BCB 8E25 7A55 85D2 9365 9BDC
9411 7ACA 7C61 4EB2 4FB5 94A6 887E 9A8E 83F3 5A87
5D5A 7D85 8A9B 5D94 89AA 9849 99F8 9BBC 5BF4 5E88
82A9 82B9 57C1 73E1 77DD 79E6 8039 83E6 8699 6366
7434 7439 79BD 9219 96C2 52E4 55EA 5AC0 6EB1 9772
5659 64D2 65B3 9CF9 61C4 6A8E 6FBF 763D 8793 61C3
8804 5745 6611 7B09 68AB 8D7A 8D7E 5BD1 9513 5BDD
5BE2 92DF 87BC 5422 5423 628B 6C81 551A 83E3 63FF
6B3D 6407 64B3 7019 85FD 9751 9752 6C22 8F7B 503E
537F 90EC 570A 57E5 6C2B 6DF8 6E05 8EFD 50BE 5ECE
873B 8F15 9CAD 9BD6 944B 591D 7520 5260 52CD 60C5
6B91 7858 6674 E847 68FE 6C30 845D 6692 64CF 6A08
64CE 6AA0 9EE5 82D8 9877 8BF7 5EBC 9803 6F00 8ACB
E846 6ABE 8B26 5E86 51CA 6385 6BB8 7883 7B90 9758
6176 78EC 512C 6FEA 7F44 6AE6 5B86 8DEB 928E 536D
909B 7A77 E840 7A79 8315 684F 7B3B 7B47 8D79 60F8
712A 712D 743C 823C 86E9 86EC 7162 718D 7758 778F
7AAE 511D 618C 6A69 749A 85D1 74CA 7AC6 85ED 74D7
4E18 4E20 90B1 5775 6058 79CB 79CC 5BC8 86AF 5A9D
8429 6978 84F2 9E59 7BCD 7DE7 8775 7A50 8DA5 9CC5
87D7 97A6 97A7 8612 E859 9C0C 9C0D 9D96 9C43 9F9D
53F4 56DA 624F 72B0 738C 6739 6C53 808D 6C42 866C
6CC5 866F 4FC5 89D3 8A04 8A05 914B 5512 6D57 7D0C
838D 900E 9011 91DA 6882 6B8F 6BEC 7403 8D47 91FB
5D37 5DEF 6E1E 6E6D 76B3 76DA 9052 716A 7D7F 86F7
88D8 5DF0 89E9 8CD5 7486 8764 92B6 9194 9B82 9F3D
9BC4 9D6D 8824 9C3D 641D 7CD7 533A 66F2 4F39 4F49
5324 5C96 8BCE 9639 9A71 5765 5C48 5CA8 5CB4 62BE
6D40 795B 80E0 88AA 5340 86C6 8EAF 7B41 7CAC 86D0
8A58 8D8B 5D87 99C6 6188 657A 8AB3 99C8 9EB9 9AF7
9B7C 8DA8 9EAF 8EC0 9EB4 9EE2 9A45 9C38 9C4B 4F62
52AC 65AA 6710 6711 80CA 83C3 8850 9E32 6DED 6E20
7D47 7FD1 844B 8EE5 8556 7496 78F2 87B6 9D1D 74A9
7FF5 87DD 9F29 8627 5337 5FC2 7048 6235 6B0B 6C0D
7C67 81DE 766F 8837 8862 8EA3 883C 947A 9E1C 53D6
7AD8 5A36 7D36 8A53 7AEC 877A 9F8B 9F72 53BA 53BB
521E 547F 8FF2 90E5 801D 9612 89D1 8DA3 95B4 9EAE
95C3 89B0 89B7 9F01 89BB 5CD1 606E 609B 5708 570F
68EC 99E9 9A21 9409 5168 6743 4F7A 8BE0 59FE 6CC9
6D24 8343 62F3 7277 8F81 5573 57E2 5A58 60D3 75CA
7842 94E8 6E76 7288 7B4C 7D5F 8143 8472 643C 697E
7454 89E0 8A6E 8DE7 8F07 8737 9293 6A29 8E21 7E13
919B 95CE 9CC8 9B08 5B49 5DCF 9C01 6B0A 9F64 8838
98A7 9874 72AD 72AC 6C71 753D 754E 70C7 7EFB 7DA3
8647 529D 5238 5DFB 7276 6926 52E7 97CF 52F8 7094
7F3A 849B 7638 5374 537B 57C6 5D05 60AB 96C0 785E
786E 8203 9615 5859 6409 76B5 788F 9619 9E4A 6128
69B7 58A7 6164 6BC3 78BA 8D9E 71E9 95CB 7910 95D5
9D72 792D 590B 56F7 5CEE 9021 8F11 5BAD 5E2C 88D9
7FA3 7FA4 88E0 4EBD 7F56 56D5 5465 80B0 887B 8887
86A6 88A1 86BA 7136 9AE5 562B 9AEF 71C3 7E4E 5184
5189 59CC 82D2 67D3 73C3 5AA3 8485 6A6A 7A63 5134
52F7 703C 737D 8618 79B3 74E4 7A70 8E9F 9B24 58CC
56B7 58E4 6518 7219 7E95 8BA9 61F9 8B72 8B93 835B
9976 6861 8558 6A48 8953 9952 72AA 6270 5A06 96A2
5B08 64FE 7ED5 9076 7E5E 60F9 70ED 71B1 4EBA 4EBB
4EC1 58EC 5FC8 6732 5FCE 79C2 82A2 9213 9B5C 928B
9D40 5FCD 834F 6820 6823 8375 79F9 68EF 7A14 7D9B
8EB5 5203 5204 8BA4 4EDE 4EED 8BB1 4EFB 5C7B 6268
7EAB 598A 6752 7263 7EB4 8095 8F6B 97E7 996A 59D9
7D09 887D 6041 7D1D 8A12 8ED4 6895 88B5 7D4D 814D
845A 976D 9771 97CC 98EA 8A8D 9901 6254 4ECD 8FB8
793D 82BF 967E 65E5 9A72 56F8 91F0 9224 99B9 620E
809C 6804 72E8 7ED2 8319 8338 8363 5BB9 5CF5 6BE7
70FF 5AB6 5D58 7D68 7FA2 5AC6 5D64 6408 6411 6449
69B5 6EB6 84C9 6995 69AE 7194 7462 7A41 877E 8923
9555 6C04 7E19 878D 878E 99E5 9AF6 5B2B 5DB8 7203
9394 701C 66E7 8811 5197 5B82 5087 8EF5 7A43 53B9
79B8 67D4 7C88 5A83 63C9 6E18 8447 7448 816C 7CC5
875A 8E42 8F2E 9352 97A3 74C7 9A25 9C07 9D94 697A
7163 97D6 8089 5B8D 5DBF 909A 5982 4F9E 5E24 8339
6847 88BD 94F7 6E2A 7B4E 8498 92A3 8560 5112 9D11
5685 5B2C 5B7A 6FE1 85B7 9D3D 66D8 71F8 8966 7E7B
8815 98A5 91B9 986C 9C6C 6C5D 8097 4E73 8FB1 910F
5165 6256 8FBC 6741 6D33 55D5 5AB7 6EBD 7F1B 84D0
9CF0 8925 7E1F 64E9 5827 648B 58D6 962E 670A 8F6F
800E 5044 8EDF 5A86 611E 744C 815D 5AF0 789D 7DDB
8761 8F2D 74C0 791D 6875 7524 7DCC 8564 854A 854B
6A64 7E60 8602 8603 6C6D 82AE 6798 868B 9510 745E
8739 777F 92B3 92ED 53E1 58E1 95F0 6DA6 958F 95A0
6F64 6A4D 633C 637C 53D2 82E5 504C 5F31 9100 5A7C
6E03 712B 6949 5D76 84BB 7BAC 7BDB 7207 9C19 9C2F
9DB8 4EE8 686C 6492 6D12 8A2F 9778 6F75 7051 5345
9491 98D2 810E 8428 9212 644B 96A1 99BA 98AF 85A9
6AD2 680D 6BE2 6122 63CC 585E 6BF8 816E 5625 567B
9CC3 984B 9C13 55EE 8D5B 50FF 8CFD 7C3A 8644 4E09
5F0E 53C1 6BF5 6BF6 5381 6BFF 7299 9B16 58ED 4F1E
5098 6563 7CC1 7CC2 9993 6A75 7CDD 7CE3 7CE4 7E56
93D2 994A 4FD5 9590 6852 6851 69E1 55D3 6421 78C9
892C 98A1 939F 9859 4E27 55AA 63BB 6145 6414 6E9E
9A9A 7F2B 7E45 81CA 9CCB 98BE 9A12 9A37 9C20 9C62
626B 6383 5AC2 57FD 7619 6C09 77C2 9ADE 8272 6D13
681C 6DA9 556C 6E0B 94EF 96ED 6B6E 7417 55C7 745F
6B70 92AB 6F81 61CE 64CC 6FC7 6FCF 7637 7A51 6F80
74B1 7012 7A61 7E6C 7A6F 8F56 93FC 8B45 98CB 88C7
8053 68EE 69EE 8942 7BF8 50E7 9B19 95AA 7E07 6740
6C99 7EB1 4E77 5239 7802 5526 6331 6BBA 7300 7C86
7D17 838E 94E9 75E7 7870 8531 88DF 699D 6A27 9B66
9CA8 95B7 93A9 9BCA 9BCB 7E7A 50BB 510D 7E4C 503D
553C 5565 5E39 8410 55A2 6B43 715E 7FDC 7B91 7FE3
95AF 970E 7B5B 7BE9 7C01 7C1B 6652 66EC 5C71 5F61
9096 5738 5220 522A 6749 6763 829F 59CD 59D7 886B
9490 57CF 633B 72E6 73CA 8222 70FB 75C1 8120 8ED5
7B18 91E4 958A 8DDA 527C 6427 5607 5E53 717D 852A
6F78 6F98 66D1 6A86 7E3F 81BB 9BC5 7FB4 7FB6 95EA
9655 70B6 965D 9583 6671 7154 7752 718C 89A2 8BAA
6C55 759D 82EB 5261 6247 8A15 8D78 5093 5584 692B
928F 9A9F 50D0 912F 58A0 58A1 7F2E 5B17 64C5 657E
6A3F 81B3 78F0 8B06 E84C E849 8D61 7E55 87EE 87FA
8B71 8D0D E84E 9425 994D 9A38 9CDD 7057 9C53 9C54
4F24 6B87 5546 6113 89DE 50B7 5892 616F 6EF3 6F21
850F 6BA4 71B5 87AA 89F4 8B2A 9B3A 88F3 57A7 6244
664C 8D4F 8CDE 945C 4E04 4E0A 4EE9 5C19 5C1A 6066
7EF1 7DD4 5F30 634E 68A2 70E7 83A6 713C 713D 7A0D
65D3 7B72 8244 86F8 8F0E 8571 71D2 98B5 9AFE 9BB9
52FA 828D 82D5 67D6 73BF 8414 97F6 5C11 52AD 5372
90B5 7ECD 54E8 5A0B 8891 7D39 7DA4 6F72 5962 731E
8D4A 7572 8F0B 8CD2 8CD6 6AA8 820C 4F58 8675 86C7
86E5 820D 6368 538D 8BBE 793E 820E 5399 5C04 6D89
6DBB 6E09 8A2D 8D66 5F3D 6151 6442 6444 6EE0 6174
6475 850E 8802 97D8 9A07 61FE 651D 7044 9E9D 6B07
7533 5C7E 625F 4F38 8EAB 4F81 547B 59BD 7C76 7EC5
7F59 8BDC 59FA 67DB 6C20 73C5 7A7C 7C78 5A20 5CF7
7521 7712 7837 5814 6DF1 7D33 515F 692E 8460 88D1
8A37 7F67 84E1 8A75 7527 8518 71CA 8593 99EA 9CB9
9BD3 9D62 9BF5 9C3A 795E 698A 926E 9C30 90A5 5F1E
628C 6C88 5BA1 77E4 54C2 77E7 5BB7 8C02 8C09 5A76
6E16 8A20 5BE9 8AD7 9823 9B6B 66CB 77AB 5B38 700B
89BE 8B85 80BE 4FBA 661A 751A 80C2 6D81 7718 6E17
7973 8124 814E 613C 614E 6939 7606 8703 6EF2 92E0
762E 5347 751F 9629 544F 58F0 6598 6607 67A1 6CE9
82FC 6B85 7272 73C4 7AD4 80DC 965E 66FB 9679 7B19
6E66 713A 7525 924E 8072 935F 9F2A 9D7F 6E11 7EF3
7E04 61B4 6FA0 7E69 8B5D 9C66 7701 771A 5057 6E3B
5723 665F 6660 5270 76DB 5269 52DD 741E 8CB9 5D4A
8056 58AD 69BA 8542 6A73 8CF8 5C38 5931 5E08 545E
8671 8672 8BD7 90BF 9E24 5C4D 65BD 6D49 72EE 5E2B
7D41 91F6 6E64 6E7F 8479 6EAE 6EBC 7345 8492 84CD
8A69 9247 7461 917E 9CF2 7BB7 8768 E85C 9CFE 8937
9CBA 6FD5 9366 9BF4 9C24 9DB3 8979 7C6D 91C3 5341
9963 4EC0 77F3 8FBB 4F66 65F6 7ACD 8BC6 5B9E 5B9F
65F9 98E0 59FC 5CD5 62FE 70BB 794F 8680 98DF 57D8
6642 83B3 5BD4 6E5C 9048 5852 5D75 6EA1 8494 9250
5BE6 69AF 8755 927D 7BD2 9CA5 9B96 9F2B 8B58 9F2D
9C23 53F2 77E2 4E68 8C55 4F7F 59CB 9A76 5158 5BA9
5C4E 7B36 6981 9242 99DB 58EB 6C0F 793B 4E16 4E17
4ED5 5E02 793A 534B 5F0F 5FD5 4E8A 53D3 623A 4E8B
4F8D 52BF 5469 67F9 89C6 8BD5 9970 519F 5BA4 6040
6043 62ED 662F 6630 67BE 67FF 7702 8D33 9002 683B
70D2 770E 7721 8210 8F7C 901D 94C8 7B39 8996 91C8
5D3C 5F11 5FA5 63D3 8C25 8CB0 91CA 52E2 55DC 5F12
7176 7757 7B6E 89E2 8A66 8EFE 9230 9243 98FE 8213
8906 8A93 9069 596D 92B4 566C 5B15 6FA8 8ADF 8AE1
907E 991D 6AA1 87AB 8B1A 7C2D 7C42 896B 91CB 9C18
9F5B 5159 74E7 53CE 6536 624B 624C 5B88 57A8 9996
824F 5BFF 53D7 72E9 517D 552E 6388 6DAD 7EF6 75E9
8184 58FD 7626 7DAC 5900 7363 7378 93C9 4E66 6BB3
6292 7EBE 53D4 6778 67A2 964E 59DD 67D5 500F 5010
66F8 6B8A 7D13 6393 68B3 6DD1 7102 83FD 8ED7 9103
758E 758F 8212 6445 6BF9 6BFA 7D80 8F93 7479 8DFE
8E08 6A1E 852C 8F38 6A7E 9B9B 5135 6504 702D 9D68
5C17 79EB 5A4C 5B70 8D4E 587E 719F 74B9 8D16 668F
6691 9ECD 7F72 9F20 5C5E 9F21 8700 6F7B 85A5 85AF
66D9 7659 85F7 8961 7CEC 8969 7C54 8834 9C6A 9E00
9C70 672E 672F 620D 675F 6CAD 8FF0 4FB8 54B0 6037
6811 7AD6 8357 6055 5EB6 5EBB 7D49 8481 8853 5C0C
88CB 6570 7AEA 8167 9265 5885 6F31 6F44 6578 6F8D
8C4E 6A39 6FD6 9330 93E3 9D90 9D91 866A 5237 5530
800D 8A9C 8870 6454 7529 5E05 5E25 87C0 535B 95E9
62F4 9582 6813 6DAE 8168 53CC 6EDD 971C 96D9 5B40
9AA6 5B47 9A3B 6B06 7935 9DDE 9E74 826D 9A66 9E18
723D 587D 6161 6A09 7E14 93EF 7040 8C01 813D 8AB0
6C35 6C34 6C3A 9596 5E28 6D97 6D9A 7971 7A05 7A0E
88DE 7761 542E 696F 987A 821C 9806 8563 6A53 779A
77A4 77AC 9B0A 8BF4 54FE 8AAA 8AAC 5981 70C1 6714
94C4 6B36 7855 77DF 55CD 6420 84B4 55FD 69CA 78A9
7361 7BBE 9399 720D 9460 53B6 7E9F 4E1D 53F8 7CF9
79C1 549D 6CC0 4FEC 601D 6056 8652 9E36 5AA4 65AF
7D72 7F0C 86F3 6952 7997 9270 98D4 51D8 53AE 69B9
79A0 7F73 8724 92AF 9536 5636 565D 5EDD 6495 6F8C
78C3 7DE6 856C 92D6 71CD 8784 9376 87D6 87F4 98B8
9A26 9401 9DE5 9F36 6B7B 5DF3 4E96 56DB 7F52 5BFA
6C5C 4F3A 4F3C 4F40 5155 59D2 6CE4 7940 4FA1 5B60
6CD7 9972 9A77 4FDF 5A30 67B1 67F6 726D 68A9 6D0D
6D98 8082 98E4 7B25 801C 91F2 7AE2 8997 55E3 8086
8C84 923B 98FC 79A9 99DF 857C 5129 9A03 7003 87A6
4E7A 5FEA 677E 6780 67A9 5A00 67D7 502F 51C7 68A5
5D27 5EBA 6DDE 83D8 5D69 7879 8719 61BD 6AA7 6FCD
9B06 6002 609A 6352 8038 7AE6 50B1 612F 5D77 616B
8073 99F7 8BBC 5B8B E82A 8BF5 9001 9882 8A1F 980C
8AA6 9938 93B9 E829 51C1 635C 910B 55D6 5EC0 5ECB
641C 6EB2 7340 8490 84C3 998A 98D5 6457 953C 8258
878B 9199 93AA 993F 98BC 9A2A 53DC 53DF 5081 55FE
778D 64DE 85AE 64FB 85EA 6AE2 7636 82CF 7526 9165
7A23 7AA3 7A4C 9BC2 8607 8613 6AEF 56CC 4FD7 738A
5919 8BC9 6CDD 8083 6D2C 6D91 73DF 7D20 901F 5BBF
6880 6B90 7C9B 9A95 5083 7C9F 8A34 8C21 55C9 5850
5851 5ACA 612B 6EAF 6EB8 8085 9061 9E54 50F3 612C
69A1 8186 850C 89EB 8D9A 906C 619F 6A0E 6A15 6F65
78BF 92C9 9917 6F5A 7E24 6A5A 749B 7C0C 85D7 8B16
8E5C 9A4C 9C50 9DEB 72FB 75E0 9178 5334 7958 7B07
7B6D 849C 7B97 590A 8295 867D 5020 54F8 6D7D 837D
837E 772D 8470 6ED6 7762 71A3 6FC9 9796 96D6 7EE5
968B 968F 9040 7D8F 96A8 74CD 81B8 7021 9AC4 9AD3
4E97 5C81 7815 795F 7C8B 8C07 57E3 5D57 813A 9042
6B72 6B73 716B 775F 788E 96A7 5B18 6FBB 7A42 8AB6
8CE5 6A96 71E7 74B2 79AD 7A57 7A5F 7E40 895A 9083
65DE 7E50 7E78 9406 8B62 9429 5B59 72F2 836A 5B6B
98E7 640E 733B 84C0 98F1 69C2 8575 859E 7543 635F
7B0B 96BC 7B4D 640D 69AB 7BB0 7C28 93A8 9DBD 5DFA
6F60 5506 5A11 838F 509E 6332 686B 68AD 7743 55E6
7FA7 84D1 644D 7F29 8D96 7C11 7C14 7E2E 9AFF 9BBB
6240 5522 7D22 7410 7411 60E2 9501 55E9 669B 6E91
7463 938D 9396 93BB 93BC 93C1 9024 6EB9 8736 4ED6
5B83 7260 7942 549C 8DBF 94CA 584C 6999 6EBB 9248
891F 8E79 4FA4 5854 5896 736D 9B99 9CCE 737A 9C28
E82F 631E 72E7 95FC 5D09 6DBE 6428 905D 9062 9618
69BB 6BFE 79A2 64BB 6FBE 8ABB 8E0F 5683 9314 56BA
6FCC 8E4B 979C 95D2 97B3 95E5 8B76 8EA2 8968 56FC
5B61 9A80 73C6 80CE 99D8 53F0 65F2 90B0 576E 62AC
82D4 70B1 70B2 83ED 8DC6 9C90 7B88 81FA 98B1 5113
9B90 5B2F 64E1 85B9 6AAF 7C49 592A 51AD 5933 5FF2
6C70 6001 80BD 949B 6CF0 7C8F 8226 915E 9226 6E99
614B 71E4 574D 8D2A 6039 5574 75D1 8211 8CAA 644A
6EE9 563D 6F6C 762B 64F9 6524 7058 7671 575B 6619
5013 8C08 90EF 5A52 60D4 5F3E 8983 6983 75F0 952C
8C2D 58B0 58B5 619B 6F6D 8AC7 9188 58C7 66C7 6A5D
931F 6A80 9843 7F48 85EB 58DC 8B5A 8C9A 91B0 8B60
7F4E 9DE4 5FD0 5766 8892 94BD 83FC 6BEF 926D 55FF
61B3 61BB 66BA 9193 74AE 8962 53F9 70AD 57EE 63A2
509D 6E60 8D55 50CB 5606 78B3 8215 64A2 6B4E 8CE7
6C64 94F4 6E6F 5621 528F 7FB0 876A 859A 9557 8E5A
93DC 940B 97BA 9F1E 9967 5763 5510 5802 508F 557A
68E0 910C 5858 5D63 642A 6E8F 84CE 969A 69B6 6F1F
717B 746D 799F 8185 6A18 78C4 7CC3 819B 6A56 7BD6
7CD6 8797 8E3C 7CDB 87B3 8D6F 91A3 9933 9395 9939
95DB 9944 9DB6 5E11 5018 5052 6DCC 50A5 8025 8EBA
954B 93B2 513B 6203 66ED 7223 77D8 9482 70EB 6465
8D9F 71D9 4ED0 5932 5F22 6D9B 7EE6 638F 7D5B 8A5C
5ACD 5E4D 6146 642F 6ED4 69C4 746B 97EC 98F8 7E1A
7E27 6FE4 8B1F 97B1 97DC 9955 530B 8FEF 54B7 6D2E
9003 6843 9676 5555 68BC 6DD8 7EF9 8404 7979 88EA
7DAF 872A 9780 9184 9789 92FE 932D 99E3 6AAE 9940
9A0A 9F17 8BA8 5957 8A0E 7553 5FD1 5FD2 7279 8CA3
8126 7286 94FD 615D 92F1 87D8 81AF 9F1F 75BC 75CB
5E50 817E 8A8A 6F1B 6ED5 9086 7E22 87A3 99E6 8B04
512F 85E4 9A30 7C50 E85B 9C27 7C58 8645 9A63 972F
551E 6730 5254 68AF 9511 8E22 92BB 9DC8 E861 9DC9
5397 7EE8 504D 73F6 557C 5A9E 5D39 60FF 63D0 6E27
7A0A 7F07 7F64 9046 9E48 55C1 7445 7D88 78AE 5FB2
6F3D 7DF9 855B 876D 9898 8DA7 8E44 918D 8B15 8E4F
9357 9CC0 9D3A 984C 9BB7 9D5C 9A20 9BF7 9D97 9D99
4F53 632E 8EB0 9AB5 8EC6 9AD4 623B 5C49 5243 6D1F
501C 608C 6D95 9016 5C5C 6090 60D5 63A6 9037 60D6
63E5 66FF 6974 88FC 8905 6B52 6BA2 9AF0 8599 568F
9B00 5694 74CB 9B04 7C4A 941F 8DAF 5929 5172 5A56
6DFB 915F 9754 9EC7 975D 7530 5C47 6CBA 606C 754B
7551 76F7 80CB 7560 751B 751C 83FE 6E49 5861 586B
6437 9617 78B5 7DC2 78CC 7AB4 9D2B 74B3 95D0 9DC6
9DCF 5FDD 6B84 500E 553A 60BF 6375 6DDF 666A 7420
8146 89CD 75F6 7753 8214 9902 89A5 8CDF 932A 9766
63AD 7471 777C 821A 65EB 4F7B 5EA3 604C 6311 7967
804E 8280 6761 5CA7 5CB9 8FE2 7952 689D 7B24 84DA
84E8 9F86 6A24 8729 92DA 9797 9AEB 9CA6 87A9 9BC8
93A5 9F60 9C37 5BA8 6640 6713 8101 7A95 8A82 7AB1
5B25 773A 7C9C 94EB 7D69 899C 8D92 8DF3 982B 7CF6
6017 8D34 841C 8051 8CBC 8DD5 94C1 86C8 9244 50E3
9295 9D29 9421 9422 9435 9A56 546B 5E16 98FB 992E
5385 5E81 6C40 827C 542C 753A 8013 539B 70C3 686F
70F4 7D8E 9793 8074 807C 5EF0 807D 5EF3 9092 5EF7
4EAD 5EAD 839B 505C 5A77 5D49 6E1F 7B73 8476 8713
695F 69B3 95AE 9706 8064 874F 8AEA 9F2E 5722 4FB9
5A17 633A 6D8F 6883 70F6 73FD 8121 94E4 8247 988B
8248 8A94 92CC 9832 6FCE 4E6D 56F2 70B5 901A 75CC
55F5 84EA 6A0B 71A5 4EDD 540C 4F5F 5F64 5CC2 5E9D
54C3 72EA 833C 664D 6850 6D75 70D4 783C 8692 772E
79F1 94DC 7AE5 7CA1 7D67 8855 8D68 916E 9256 50EE
52ED 9275 9285 9907 9C96 6F7C 735E 66C8 6723 6A66
6C03 729D 81A7 77B3 7A5C 9BA6 7EDF 6345 6876 7B52
7D71 7B69 7D82 6078 75DB 615F 6185 5077 5078 5A7E
5AAE 936E 4EA0 5934 6295 9AB0 7DF0 982D 59B5 7D0F
6568 6B95 65A2 9EC8 8623 900F 51F8 79BF 79C3 6022
7A81 550B 6D8B 6378 5817 6E65 75DC 8456 5D80 92F5
9D5A 9F35 56F3 56FE 51C3 5CF9 5EA9 5F92 6087 6348
6D82 837C 9014 5C60 688C 63EC 7A0C 5857 5D5E 760F
7B61 816F 84A4 922F 5716 5717 5EDC 6F73 8DFF 9174
999F 934E 99FC 9D4C 9D9F 9DCB 9DF5 571F 5721 5410
6C62 948D 91F7 514E 8FCC 5154 83B5 580D 83DF 9D75
6E4D 732F 7153 8C92 56E2 56E3 629F 526C 5278 5718
587C 6171 6476 69EB 6F19 7BFF 6AB2 93C4 7CF0 9DD2
9DFB 5715 7583 5F56 6E6A 8916 63A8 84F7 85EC 9893
96A4 5C35 9839 983A 983D 9B4B 7A68 8608 8E6A 4FC0
812E 817F 50D3 8E46 9ABD 9000 5A27 717A 86FB 8715
892A 99FE 541E 5451 65FD 6D92 554D 671C 711E 66BE
9ED7 5C6F 5FF3 829A 9968 8C58 8C5A 8ED8 98E9 9C80
9B68 9715 81C0 81CB 6C3D 757D 5749 4E47 8BAC 6258
6261 6C51 9966 6754 4F82 5483 62D5 62D6 6CB0 4FBB
6329 635D 838C 88A5 8A17 6DB6 812B 8131 98E5 99B2
9B60 9A5D 9A6E 4F57 9640 9641 5768 5CAE 6CB1 72CF
8FF1 9A7C 67C1 7823 7824 8889 9E35 7D3D 5836 8A51
8DCE 9161 78A2 99B1 69D6 99C4 8E3B 99DD 99DE 6A50
9B80 9D15 9F27 9A28 9F0D 9A52 9F09 5F75 59A5 6BE4
5EB9 5AA0 692D 6955 5AF7 64B1 6A62 9D4E 9C16 62D3
67DD 553E 841A 8DC5 6BFB 7BA8 8600 7C5C 5C72 52B8
54C7 5A03 5F8D 6316 6D3C 5A32 7556 7A8A 5558 5AA7
55D7 86D9 6432 6E9B 6F25 7AAA 9F03 6528 74E6 4F64
90B7 5493 74F2 7819 889C 8049 55E2 817D 8183 896A
97C8 97E4 6B6A E81F 558E 7AF5 5D34 5916 9861 4E5B
5F2F 525C 5A60 5E35 5846 6E7E 7755 873F 6F6B 8C4C
5F4E 58EA 7063 4E38 5213 6C4D 7EA8 8284 5B8C 5C8F
5FE8 628F 73A9 7B02 7D08 6356 987D 70F7 7413 8CA6
9811 909C 5B9B 5007 550D 633D 665A 76CC 839E 57E6
5A49 60CB 6669 689A 6DB4 7EFE 8118 83C0 667C 6900
742C 7696 7579 7897 7DA9 7DB0 8F13 8E20 92D4 9350
4E07 534D 5350 59A7 6764 6365 8155 842C 7FEB 92C4
858D 933D 8D03 93AB 8D0E 5C23 5C29 5C2A 5C2B 6C6A
4EA1 4EBE 5166 738B 4EFC 5F7A 83A3 869F 7F51 5FF9
5F80 5F83 6789 7F54 60D8 83F5 6680 68E2 7139 86E7
8F8B 7DB2 8744 8AB7 8F1E 7007 9B4D 5984 5FD8 8FCB
65FA 76F3 671B 6722 5371 5A01 70D3 504E 9036 9687
9688 55B4 5A81 5A99 6104 63CB 63FB 6E28 7140 8468
8473 5FAE 6933 6972 6EA6 7168 8A74 7E05 875B 89A3
5DB6 8587 71F0 9CC2 7650 5DCD 9C03 9C04 56D7 4E3A
97E6 5729 56F4 5E0F 6CA9 8FDD 95F1 5CD7 5CDE 6D08
70BA 97CB 6845 6DA0 552F 5E37 60DF 7EF4 55A1 570D
5D6C 5E43 6E4B 6E88 7232 741F 9055 6F4D 7DAD 84F6
912C 6F59 6F7F 9180 6FF0 934F 95C8 9BA0 7653 89B9
72A9 973A 973B 5383 4F1F 4F2A 5C3E 7EAC 829B 82C7
59D4 709C 73AE 6D27 5A13 6364 6D58 8371 8BFF 5049
507D 5D23 68B6 75CF 784A 840E 9697 9AA9 5D54 5EC6
5FAB 6107 7325 8466 848D 9AAA 9AAB 6690 6932 7152
744B 75FF 8172 8249 97EA 50DE 78A8 8732 873C 9C94
5BEA 7DEF 853F 8AC9 8E13 97D1 9820 85B3 5130 6FFB
9361 9BAA 58DD 97D9 98B9 7022 97E1 4EB9 6596 536B
672A 4F4D 5473 82FF 754F 80C3 53DE 8ECE 5C09 7859
83CB 8C13 5582 5AA6 6E2D 732C 715F 589B 851A 6170
71AD 729A 78D1 7DED 875F 885B 61C0 6FCA 748F 7F7B
885E 8B02 9317 9927 9B87 87B1 893D 9935 9B4F 85EF
8F4A 93CF 9728 9CDA 8636 9956 8B86 8E97 8B8F 8E9B
6637 586D 6E29 6985 6B9F 6EAB 7465 8F92 69B2 761F
8C71 8F3C 8F40 9CC1 93BE 9942 9C1B 9C2E 6587 5F63
7EB9 82A0 7086 7807 95FB 7D0B 8689 868A 73F3 960C
922B 96EF 7612 805E 99BC 9B70 9CFC 9D0D 87A1 95BA
95BF 87C1 95C5 9F24 95E6 95E7 520E 543B 545A 5FDF
6286 5461 80B3 7D0A 687D 8117 7A33 7A4F 7A69 95EE
598F 6C76 83AC 554F 6E02 8115 63FE 6435 7D7B 9850
74BA 7FC1 55E1 9E5F 8789 9393 9DB2 52DC 5963 5855
5D61 6EC3 84CA 66A1 7788 806C 74EE 8579 7515 7F4B
9F46 631D 502D 6DA1 83B4 5529 6DB9 6E26 7327 8435
5594 7A9D 7AA9 8717 64BE 8778 8E12 6211 5A50 5A51
6370 4EF4 6C83 809F 5367 81E5 5053 637E 5A89 5E44
63E1 6E25 7125 786A 6943 815B 65A1 7783 6FE3 74C1
81D2 9F8C 9F77 4E4C 572C 5F19 6C59 6C5A 6C61 90AC
545C 6747 5DEB 5C4B 6D3F 8BEC 94A8 70CF 8DB6 526D
7A8F 91EB 9114 55DA 8A88 6B4D 8AA3 7BBC 8790 9D2E
93A2 9C1E 65E0 6BCB 5433 5434 543E 5449 829C 90DA
5514 5A2A 68A7 6D16 6D6F 8323 8381 73F8 7966 9E40
7121 7991 8708 856A 7491 9D50 9BC3 9F2F 9DE1 4E44
4E94 5348 4EF5 4F0D 575E 59A9 5E91 5FE4 6003 8FD5
65FF 6B66 739D 4FAE 4FC9 5035 6342 554E 5A2C 727E
73F7 5862 6440 7193 7894 9E49 7466 821E 5AF5 5EE1
61AE 6F55 933B 511B 6A46 7512 9D61 8E8C 5140 52FF
52A1 620A 9622 4F06 5C7C 6264 5C89 674C 82B4 5FE2
7269 77F9 6544 8BEF 52D9 609E 609F 60AE 7C85 901C
6664 7110 7111 5A7A 5D4D 75E6 9696 9770 9A9B 5966
5D68 6EA9 96FE 5BE4 7183 8AA4 9E5C 92C8 7AB9 971A
9F3F 9727 9F40 8601 9A16 9DA9 5915 516E 5FDA 6C50
897F 8980 5438 5E0C 6278 5365 6614 6790 77FD 7A78
80B8 80B9 4FD9 5F86 6038 8BF6 90D7 997B 550F 595A
5A2D 5C56 5C57 606F 6095 665E 6C25 6D60 727A 72F6
8383 553D 6089 60DC 6878 6B37 6DC5 6E13 70EF 7101
7108 740B 7852 83E5 8D65 91F8 5092 60C1 6670 6673
711F 712C 7280 774E 7A00 7C9E 7FD5 7FD6 823E 910E
5380 5D60 5FAF 6EAA 7155 7699 84A0 9521 50D6 69BD
7184 7188 7199 7DC6 8725 8A92 8C68 990F 563B 564F
5B06 5B09 761C 819D 9919 51DE 6A28 6A40 6B59 6B5A
71B9 71BA 71BB 7AB8 7FB2 8785 8787 932B 71E8 72A0
77A6 7902 87CB 8C3F 8C40 8C6F 8C95 7E65 96DF 9BD1
9D57 89F9 8B46 91AF 93ED 96B5 5DC7 66E6 7214 72A7
9145 89FD 9F37 8835 9E02 89FF 9474 4E60 90CB 5E2D
7FD2 88AD 89CB 5AB3 693A 84B5 84C6 5D8D 6F1D 89A1
8D98 69E2 8777 8582 96B0 6A84 8B35 93B4 972B 9CDB
98C1 9A31 9A3D 8972 9C3C 9A68 676B 67B2 6D17 73BA
5F99 94E3 559C 5F9A 8448 8478 9222 5C63 6F07 84F0
9291 6198 6199 66BF 6A72 6B56 79A7 8AF0 58D0 7E30
8B11 87E2 8E5D 74BD 9C5A 77D6 7E9A 8EA7 5338 534C
620F 5C43 7CFB 9969 546C 5FE5 602C 7EC6 90C4 4FC2
54A5 6044 76FB 90E4 6B2F 7EE4 7D30 91F3 960B 5848
691E 8204 8D87 9699 6140 6ECA 798A 7D8C 8D69 969F
7182 7294 7A27 622F 6F5D 6F5F 6F99 856E 89A4 6231
9ED6 6232 78F6 8669 993C 9B29 56B1 95DF 973C 884B
5477 75A8 867E 8C3A 5084 9595 656E 7146 98AC 778E
8766 9C15 5323 4FA0 72CE 4FE0 5CE1 67D9 70A0 72ED
965C 5CFD 70DA 72F9 73E8 796B 7856 7B1A 7FC8 823A
967F 6E8A 7864 9050 6433 6687 7455 7B6A 78AC 821D
8F96 78CD 7E00 8578 7E16 8D6E 9B7B 8F44 935C 971E
938B 9EE0 9A22 9DB7 959C 4E05 4E0B 5413 5737 759C
590F 68BA 53A6 5EC8 7771 8AD5 5687 61D7 7F45 5913
93EC 4ED9 4EDA 5C73 5148 597E 7EA4 4F61 5FFA 6C19
6774 7946 79C8 82EE 67AE 7C7C 73D7 83B6 6380 94E6
8DF9 9170 9528 50CA 50F2 5615 929B 9C9C 66B9 97EF
5B10 61B8 859F 9341 9342 7E4A 893C 97F1 9BAE 99A6
8E6E 5B45 5EEF 6515 8B63 7E8E 9DB1 8973 8E9A 7E96
9C7B 4F2D 549E 95F2 59B6 5F26 8D24 54B8 550C 6326
6D8E 80D8 5A34 5A39 5A71 7D43 8237 86BF 8854 5563
6E7A 75EB 86DD 9591 9E47 5ACC 8858 7509 929C 5AFA
5AFB 61AA 648F 6F96 8AB8 8CE2 8AF4 8F31 918E 7647
764E 85D6 9E79 7925 8D12 9466 9DF3 9DF4 9DFC 51BC
72DD 663E 9669 5D04 6BE8 70CD 7303 86AC 967A 8D7B
7B45 5C1F 5C20 641F 7992 8706 8DE3 7BB2 85D4 96AA
5DAE 736B 736E 85D3 934C 71F9 9855 5E70 6507 6AF6
861A 7381 97C5 986F 7066 53BF 5C98 82CB 73B0 7EBF
81FD 9650 59ED 5BAA 770C 9665 54EF 57B7 5A0A 5A28
5CF4 665B 6D80 83A7 9677 73FE 784D 9985 774D 7D64
7F10 7FA1 732E 7CAF 7FA8 817A 50E9 50F4 7DAB 8AA2
648A 7DDA 92E7 61B2 6A4C 7E23 930E 9921 8C4F 9EB2
7017 81D4 737B 7CEE 93FE 9730 9F38 4E61 8297 76F8
9999 90F7 53A2 554C 9109 910A 5EC2 6E58 7F03 8459
9115 697F 858C 7BB1 7DD7 81B7 8944 5FC0 9AA7 9E98
6B00 74D6 9576 9C5C 9472 9A64 74E8 4F6D 8BE6 5EA0
6819 7965 7D74 7FD4 8A73 8DED 4EAB 4EAF 54CD 8683
9977 6651 98E8 60F3 9284 9909 9C9E 56AE 8801 9B9D
9BD7 97FF 9957 995F 9C76 5411 59E0 5DF7 9879 73E6
8C61 7F3F 842B 9805 50CF 52E8 5D91 66CF 6A61 8950
87D3 940C 9C4C 7071 7072 547A 67AD 4FBE 524A 54D3
67B5 9A81 5BAF 5BB5 5EA8 6077 6D88 7EE1 8653 900D
9E2E 554B 5A4B 689F 7107 7307 8427 75DA 75DF 785D
7863 7A99 7FDB 8437 9500 63F1 7D83 5610 6B4A 6F47
7BAB 8E03 5635 61A2 64A8 7362 92B7 9704 5F47 81AE
856D 9B48 9D1E 7A58 7C18 85C3 87C2 87CF 8B1E 9D35
56A3 701F 7C2B 87F0 9AC7 56BB 56C2 6AF9 9AD0 9DCD
8828 9A4D 6BCA 8648 6D28 90E9 5D24 6DC6 8A24 8AB5
5C0F 6653 6681 7B71 7B7F 769B 66C9 7BE0 8B0F 76A2
5B5D 8096 52B9 54B2 6054 4FF2 54EE 6548 6821 6D8D
7B11 5578 509A 6569 6EE7 8A68 560B 5628 8A9F 562F
6B57 71BD 6585 6586 4E9B 6954 6B47 874E 880D 52A6
534F 65EA 90AA 5354 80C1 57A5 594A 5CEB 604A 62F9
631F 633E 8105 8107 810B 887A 5055 659C 8C10 7332
7D5C 7FD3 55CB 6136 643A 744E 7D8A 7181 818E 52F0
64B7 64D5 7DF3 7F2C 8762 978B 8AE7 71F2 64F7 97B5
896D 9437 651C 7E88 8B97 9FA4 5199 51A9 5BEB 85DB
4F33 707A 6CC4 6CFB 7944 7EC1 7F37 5378 6D29 70A7
70A8 5368 5A0E 5C51 5C53 5070 5FA2 68B0 710E 79BC
7D32 4EB5 5A9F 5C5F 63F3 6E2B 7D4F 7D6C 8C22 50C1
586E 698D 69AD 8909 5667 5C67 66AC 7DE4 97F0 5DB0
5EE8 61C8 6FA5 736C 7CCF 85A2 85A4 9082 71EE 893B
8B1D 5911 7009 97A2 7023 7215 87F9 880F 9F58 9F65
9F42 8EA0 5C6D 8E9E 5FC4 5FC3 90A4 59A1 5FFB E844
82AF 8F9B 6615 677A 6B23 76FA 4FFD 8398 60DE 8A22
920A 950C 65B0 6B46 5EDE 92C5 5677 567A 5B1C 85AA
99A8 946B 99AB 6794 9B35 9414 4F08 6F43 9620 4F29
56DF 5B5E 7098 4FE1 8ED0 812A 8845 8A2B 712E 99B8
820B 9856 91C1 5174 72CC 661F 57B6 9A8D 60FA 7329
714B 7446 8165 86F5 89EA 7BB5 7BC2 8208 8B03 9B8F
66D0 89F2 9A02 76A8 9BF9 5211 884C 90A2 5F62 9649
4F80 90C9 54D8 578B 6D10 9498 9658 5A19 784E 88C4
94CF 9203 9276 9292 92DE 7772 9192 64E4 674F 59D3
5E78 6027 8347 5016 8395 5A5E 60BB 6DAC 5842 7DC8
5B39 81D6 51F6 5144 5147 5308 828E 8BBB 5FF7 6C79
54C5 605F 6D36 80F7 80F8 8A29 8A7E 96C4 718A 8BC7
8A57 5910 657B 4F11 4FE2 4FEE 54BB 5EA5 70CB 70CC
7F9E 8129 8119 9E3A 81F9 8C85 9990 6A07 929D 9AE4
9AF9 9380 9BB4 9D42 9948 93C5 98CD 82EC 673D 7D87
6EEB 7CD4 79C0 5CAB 73DB 7EE3 8896 7407 9508 6EB4
7D89 7493 88E6 890E 890F 92B9 8791 7E4D 7E61 93E5
93FD 9F45 620C 65F4 759E 76F1 6B28 7809 80E5 987B
8A0F 987C 8657 865A 8C1E 5AAD 5E41 63DF 6B3B 865B
9808 6948 7AA2 980A 5618 7A30 9700 9B46 5653 589F
5B03 6B54 7E03 8566 8751 6B58 8ADD 8B43 9B56 9A49
9450 9B1A 4FC6 5F90 84A3 8BB8 5474 59C1 8BE9 5194
6829 73DD 5066 8A31 6E51 668A 8A61 9126 7CC8 9191
76E8 65ED 4F35 5E8F 6C7F 4F90 5379 6C80 53D9 6064
662B 6D2B 57BF 6B30 6B88 70C5 73EC 755C 52D6 52D7
654D 6558 70FC 7EEA 7EED 9157 55A3 58FB 5A7F 6702
6E86 7D6E 8A39 55C5 6149 7166 7D9A 84C4 8CC9 69D2
6F35 6F4A 76E2 7781 7DD2 805F 928A 735D 7A38 7DD6
85C7 77B2 85DA 7E8C 9C6E 84FF 5405 8F69 660D 54BA
5BA3 6645 8ED2 688B 8C16 55A7 5847 5A97 6103 610B
63CE 8431 8432 6684 714A 7444 84D2 777B 5107 79A4
7BAE 7FE7 8756 5B1B 857F 8AE0 8AFC 9379 99FD 77CE
7FFE 85FC 8610 8809 8B5E 9C1A 8B82 7384 73B9 75C3
60AC 65CB 7401 8701 5AD9 6F29 66B6 7487 6A88 74BF
61F8 9009 70DC 6685 9078 7663 766C 6030 6CEB 6621
70AB 7EDA 7729 88A8 94C9 7404 7734 8852 6E32 7D62
6965 6966 9249 78B9 8519 955F 9799 98B4 7E3C 7E4F
93C7 8D19 75B6 8486 9774 859B 8FA5 97BE 7A74 6588
4E74 5779 5B66 5CA4 5CC3 8313 6CF6 8895 9E34 8E05
5B78 5DA8 6FA9 71E2 89F7 96E4 9DFD 96EA 6A30 81A4
825D 8F4C 9CD5 9C48 8840 5437 6034 6CE7 72D8 75A6
6856 70D5 8C11 8D90 8B14 7025 5743 52CB 57D9 7104
52DB 5864 718F 7AA8 8512 52F2 52F3 85AB 99E8 5691
58CE 736F 85B0 66DB 71FB 81D0 77C4 860D 58E6 720B
7E81 91BA 5EF5 5BFB 5DE1 65EC 9A6F 674A 8BE2 5CCB
6042 6D35 6D54 7D03 8340 6812 686A 6BE5 73E3 5071
5C0B 5FAA 63D7 8A62 99B4 9129 9C9F 565A 6F6F 6533
6A33 71C2 71C5 71D6 7495 8951 87F3 9C4F 9C58 7065
5342 8BAD 8BAF 4F28 6C5B 8FC5 4F9A 5F87 72E5 8FFF
900A 6B89 8A0A 8A13 8A19 595E 5DFD 6BBE 905C 613B
8CD0 5640 8548 9868 9442 4E2B 538B 5440 5E98 62BC
9E26 6860 9E2D 5B72 94D4 690F 9D09 930F 9D28 58D3
9D76 941A 7259 4F22 5C88 82BD 5393 6792 740A 7B0C
869C 5810 5D15 5D16 6DAF 731A 7458 775A 8859 6F04
9F56 758B 538A 5E8C 54D1 5516 555E 75D6 96C5 7602
8565 529C 5720 4E9A 7A75 897E 8BB6 4E9C 72BD 8FD3
4E9E 73A1 57AD 5A05 631C 7811 4FF9 6C29 57E1 5A6D
6397 8A1D 63E0 6C2C 7330 8050 5714 7A0F 7AAB 9F7E
54BD 6079 5266 70DF 73DA 80ED 5063 5D26 6DF9 7109
83F8 9609 6E6E 814C 50BF 7159 9122 5AE3 6F39 852B
5D96 6A2E 9183 95B9 5B2E 7BF6 61D5 81D9 9EEB 8BA0
5186 5EF6 95EB 4E25 598D 82AB 8A00 8A01 5CA9 6616
6CBF 708E 90D4 59F8 5A2B 72FF 7814 839A 5A2E 76D0
7402 784F 8A2E 9586 960E 5D52 5D53 E863 7B75 7D96
8712 5869 63C5 694C 8A7D 789E 8505 989C 8664 95BB
53B3 6A90 984F 9854 56B4 58DB 5DCC 7C37 6AE9 9E99
58E7 5B4D 5DD6 5DD7 5DDA 6B15 7939 9E7D 9EA3 5935
6281 6C87 4E75 5156 5944 4FE8 5157 533D 5F07 884D
5043 53A3 63A9 773C 8412 90FE 9153 5D43 611D 624A
63DC 68EA 6E30 6E37 7430 9043 9692 693C 787D 7F68
88FA 6F14 8917 622D 8758 9B47 565E 8EBD 7E2F 6ABF
9EE1 53B4 7517 9C0B 9DA0 9EE4 9F5E 9F91 513C 9EEC
9EED 9869 9F34 5DD8 66EE 9B58 9F39 9F74 9EF6 538C
599F 89C3 726A 59F2 5F65 5F66 781A 5501 5BB4 664F
8273 898E 9A8C 5050 639E 7114 8C1A 9681 55AD 5830
6565 7130 7131 7312 786F 8455 96C1 693B 6EDF 9CEB
53AD 5895 66A5 7196 917D 5B0A 8C33 990D 9D08 71C4
71D5 8AFA 8D5D 9B33 66D5 9D33 9140 9A10 9A13 56A5
E845 5B3F 8276 8D0B 8EC5 7213 91B6 9A34 9DC3 7054
8D17 8D18 89FE 8B8C 91BC 995C 9A57 9DF0 8277 704E
91C5 9A60 7067 8B9E 8C53 8C54 7069 592E 5489 59CE
62B0 6CF1 6B83 80E6 770F 79E7 9E2F 9260 96F5 9785
9348 9D26 626C 7F8A 9626 9633 65F8 6768 7080 4F6F
52B7 6C1C 75A1 9496 98CF 579F 5F89 661C 6D0B 7F8F
70CA 73DC 773B 967D 5D35 5D38 63DA 86D8 656D 6698
694A 716C 7993 760D 8AF9 8F30 935A 9D39 98BA 940A
9C11 9737 9E09 536C 4EF0 4F52 5771 594D 5C9F 517B
70B4 6C27 75D2 7D3B 509F 6967 8EEE 6143 6C31 7FAA
990A 99DA 61E9 6501 7001 7662 7922 600F 67CD 6059
6837 7F95 8A47 69D8 6F3E 6A23 5E7A 592D 5406 5996
6796 7945 8A1E 5593 847D 6946 8170 9D01 9080 723B
5C27 5C2D 80B4 579A 59DA 5CE3 8F7A 5004 70D1 73E7
7A91 509C 582F 63FA 6BBD 8C23 8EFA 55C2 5AB1 5FAD
612E 6416 6447 733A 9059 9065 647F 669A 69A3 7464
7476 929A 98D6 9906 5DA2 5DA4 5FBA 78D8 7AAF 7AB0
991A 7E47 8B20 8B21 9390 9CD0 98BB 8628 9864 9C29
4EF8 5B8E 5C86 62AD 6773 6B80 72D5 82ED 54AC 67FC
7711 7A85 7A88 8200 5060 5A79 5D3E 6E94 84D4 699A
9D22 95C4 9A15 9F69 9DD5 7A7E 836F 8981 888E 7A94
7B44 846F 8A4F 718E 899E 977F 735F 9E5E 85AC 9F3C
66DC 71FF 825E 85E5 77C5 66E3 8000 7E85 9DC2 8B91
9470 503B 6930 668D 564E 6F71 882E 7237 8036 6353
63F6 94D8 723A E84A 91FE 92E3 9381 64E8 4E5F 5414
4EAA 51B6 57DC 91CE 5622 6F1C 58C4 4E1A 53F6 66F3
9875 90BA 591C 62B4 4EB1 67BC 6D02 9801 6359 6654
67BD 70E8 505E 6396 6DB2 8C12 5828 6B97 814B 8449
9113 58B7 696A 696D 998C 50F7 66C4 66C5 6B4B 71C1
64D6 64DB 76A3 77B1 9134 9765 5DAA 5DAB 6FB2 8B01
9923 5688 64EB 66D7 77B8 9371 64EA 7217 790F 9391
9941 9D7A 9768 9A5C 9E08 81B6 5C83 4E00 5F0C 8FB7
8864 4F0A 8863 533B 541A 58F1 4F9D 794E 54BF 6D22
7317 7569 90FC 94F1 58F9 63D6 6B39 86DC 7995 5ADB
6F2A 7A26 92A5 5B04 566B 5901 747F 9E65 7E44 6AB9
6BC9 91AB 9EDF 8B69 9DD6 9EF3 4E41 4EEA 531C 572F
5937 519D 5B90 675D 6C82 8BD2 4F87 5B9C 6021 6CB6
72CB 886A 8FE4 9974 54A6 59E8 5CD3 5F2C 605E 67C2
74F5 8351 8D3B 8FFB 5BA7 5DF8 6245 6818 684B 7719
80F0 8898 914F 75CD 79FB 8413 5A90 692C 7FA0 86E6
8A52 8CBD 9057 6686 6938 8A83 8DE0 9809 9890 98F4
7591 5100 71AA 907A 5DAC 5F5B 5F5C 8794 9824 9825
5BF2 5DB7 7C03 984A 9BA7 5F5D 5F5E 8B3B 93D4 7C4E
89FA 8B89 9E03 4E59 5DF2 4EE5 8FC6 9487 4F41 653A
77E3 82E1 82E2 5EA1 8223 8681 91D4 501A 6246 7B16
9018 506F 5D3A 65D1 6905 9218 926F 9CE6 65D6 8F22
657C 8798 6AA5 7912 8264 87FB 9857 8F59 9F6E 4E42
4E49 4EBF 5F0B 5208 5FC6 827A 4EE1 5307 808A 8BAE
9623 4EA6 4F07 5C79 5F02 5FD4 8285 4F3F 4F5A 52AE
5453 5744 5F79 6291 66F5 6759 8034 82C5 8BD1 9091
4F7E 546D 5479 59B7 5CC4 6008 603F 6613 678D 6CC6
7088 79C7 7ECE 8BE3 9A7F 4FCB 5955 5E1F 5E20 5F08
67BB 6D42 73B4 75AB 7FBF 8875 8F76 5508 57BC 6092
6339 6827 683A 6B2D 6D65 6D73 76CA 88A3 8C0A 8CA4
966D 52DA 57F6 57F8 6098 60A5 6BB9 7570 7F9B 7FCA
7FCC 841F 8A32 8A33 8C59 8C5B 9038 91F4 96BF 5E46
6561 6679 68ED 6B94 6E59 7132 86E1 8A4D 8DC7 8EFC
9220 9AAE 4E84 517F 610F 6EA2 7348 75EC 7AE9 7F22
7FA9 8084 88D4 88DB 8A63 52E9 5AD5 5ED9 698F 6F69
7617 8189 84FA 8734 977E 99C5 5104 648E 69F8 6BC5
71A0 71A4 71BC 761E 8ABC 9552 9E5D 9E62 9ED3 5293
571B 58BF 5B11 5B1F 5DA7 61B6 61CC 66C0 6BAA 6FBA
71DA 7631 7796 7A53 7E0A 8257 858F 87A0 8939 5BF1
6581 66CE 6A8D 6B5D 71E1 71F1 7FF3 7FFC 81C6 8C96
9BA8 7654 85D9 85DD 8D00 93B0 9571 7E76 7E79 8C77
972C 9BE3 9D82 9D83 9D8D 7037 8619 8B6F 8B70 91B3
91B7 9950 56C8 943F 9DC1 9DCA 61FF 897C 9A5B 9DE7
8649 9DFE 8B9B 9F78 4E5A 56D9 56E0 9625 9634 4F8C
5794 59FB 6D07 8335 836B 97F3 9A83 6836 6BB7 6C24
9670 51D0 79F5 88C0 94DF 967B 9682 5591 5819 5A63
6114 7B43 7D6A 6B45 6EB5 798B 8491 852D 6147 7616
92A6 78E4 7DF8 9787 8AF2 9712 99F0 567E 6FE6 95C9
9720 97FE 5198 4E51 541F 72BE 82C2 65A6 57A0 6CFF
5701 5CFE 70CE 72FA 73E2 7C8C 8376 8A14 552B 5A6C
5BC5 5D1F 5D2F 6DEB 8A21 94F6 921D 9F82 6EDB 7892
911E 5924 8529 8A1A 8ABE 9280 9F88 5656 6BA5 748C
569A 6AAD 87EB 972A 9F57 9F66 9DE3 5EF4 5C39 5F15
5432 996E 8693 9690 6DFE 91FF 920F 98F2 96A0 9777
98EE 6704 8D9B 6A83 763E 96B1 5DBE 6FE5 87BE 861F
6AFD 766E 8B94 5370 831A 6D15 80E4 57BD 6E5A 730C
5ED5 9173 616D 764A 6196 6197 E857 9BA3 61DA 6ABC
5E94 5FDC 82F1 5040 685C 73F1 83BA 5568 5A74 5A96
6125 6E36 7EEC 6720 7150 745B 5AC8 78A4 9533 5624
6484 6ECE 7507 7DD3 7F28 7F42 8767 8CCF 6A31 748E
565F 7F43 892E 9719 9D2C 9E66 5B30 61C9 81BA 97FA
7516 93A3 9E70 9DA7 56B6 5B46 5B7E 6516 7034 7F4C
8621 6AFB 74D4 792F 8B7B 9DAF 944D 7E93 8833 9DEA
8EC8 9DF9 9E0E 9E1A 76C1 8FCE 8314 76C8 8365 8367
83B9 8424 8425 8426 86CD 55B6 6E81 6E8B 843E 50CC
584B 6979 6EE2 84E5 6F46 7192 8747 7469 877F 5B34
71DF 7E08 87A2 6FD9 6FDA 6FF4 85C0 89AE 8B0D 8D62
5DC6 650D 651A 701B 7020 702F 8805 6AFF 7050 7C5D
705C 8D0F 7C6F 77E8 90E2 6D67 68AC 988D 9895 9896
646C 5F71 6F41 763F 7A4E 9834 5DCA 5EEE 941B 766D
6620 668E 786C 5AB5 81A1 9795 7005 8B4D 54DF 5537
55B2 4F63 62E5 75C8 9095 5EB8 50AD 55C8 9118 96CD
5889 5ADE 6175 6EFD 69E6 7245 92BF 5670 58C5 64C1
6FAD 90FA 955B 81C3 7655 96DD 93DE 9CD9 5EF1 7049
9954 9C45 9DDB 7670 5581 9899 9852 9C2B 6C38 752C
548F 603A 6CF3 4FD1 52C7 52C8 6810 57C7 6080 67E1
6D8C 607F 509B 60E5 6111 6E67 7867 8A60 584E 5D71
5F6E 6139 86F9 6142 8E0A 799C 9CAC 8E34 9BD2 7528
82DA 783D 919F 4F18 5FE7 6538 5466 602E 6CD1 5E7D
60A0 9E80 6EFA 6182 512A 913E 5698 61EE 7000 6ACC
7E8B 8030 5C22 5C24 7531 6C8B 72B9 90AE 601E 6CB9
80AC 6023 65BF 67DA 75A3 5CF3 6D5F 79DE 839C 83A4
83B8 900C 90F5 94C0 5064 86B0 8A27 9030 6E38 7336
904A 9C7F 6962 7337 923E 9C89 8F0F 99C0 8555 8763
9B77 8F36 9B8B 6AFE 908E 53CB 6709 4E23 5363 82C3
9149 7F91 5EAE 7F90 83A0 6884 8048 811C 94D5 6E75
848F 7989 870F 92AA 69F1 7256 7257 9EDD 53C8 53F3
5E7C 4F51 4F91 5B67 72D6 7CFF 54CA 56FF 59F7 5BA5
5CDF 7270 7950 8BF1 8FF6 5500 688E 86B4 4EB4 8C81
91C9 916D 8A98 9F2C 625C 7EA1 8FC2 8FC3 7A7B 9653
7D06 8676 5539 6DE4 76D3 6E1D 7600 7B8A 4E8E 4E90
4E88 9098 4F03 4F59 59A4 6275 6745 6B24 7397 7399
65BC 76C2 81FE 8867 9C7C 4FDE 516A 79BA 7AFD 8201
8330 8362 5A1B 5A2F 5A31 72F3 8C00 9151 9980 6E14
8438 91EA 9683 9685 96E9 9B5A 5823 582C 5D33 5D4E
5D5B 6109 63C4 6970 6E61 756C 756D 7862 8174 903E
9AAC 611A 6961 6986 6B48 724F 745C 8245 865E 89CE
6F01 E856 E85A 776E 7AAC 8206 8915 6B76 7FAD 854D
8753 8ADB 96D3 9918 9B63 5B29 61D9 6F9E 89A6 8E30
6B5F 74B5 87B8 8F3F 935D 7916 8B23 9AC3 9BBD 65DF
7C45 9A1F 9BF2 9C05 9DE0 9E06 4E0E 4F1B 5B87 5C7F
7FBD 96E8 4FC1 4FE3 6327 79B9 8BED 5704 5CFF 7964
504A 532C 5709 5EBE 6554 9105 842D 842E 94FB 50B4
5BD9 659E 6940 7440 7610 8207 8A9E 7AB3 92D9 9F89
5673 5DBC 8C90 6594 9E8C 860C 9F6C 7389 9A6D 5401
572B 807F 828B 828C 59AA 5FEC 996B 80B2 90C1 5F67
6631 72F1 79D7 831F 4FFC 5CEA 682F 6D74 7821 94B0
9884 5590 57DF 5809 6086 60D0 6B32 6DE2 6DEF 88AC
8C15 9033 9608 5585 55A9 55BB 5A80 5BD3 5EBD 5FA1
68DB 68DC 68EB 7134 7419 77DE 88D5 9047 98EB 99AD
9E46 6108 6EEA 715C 7A22 7F6D 84AE 84E3 8A89 923A
9810 5AD7 5D8E 622B 6BD3 7344 7609 7DCE 871F 872E
8F0D 9289 96A9 564A 617E 7A36 84F9 8581 8C6B 9079
92CA 9CFF 6FA6 71CF 71E0 8577 8AED 9325 95BE 9D25
9D27 9D2A 5125 7907 79A6 9B4A 9E6C 7652 791C 7A65
7BFD 7E58 91A7 9D52 6AF2 9947 861B 8B7D 8F5D 942D
9731 6B0E 9A48 9B3B 7C5E 9C4A 9DF8 9E12 6B1D 8EC9
9B30 9B31 706A 7C72 7229 56E6 9E22 5248 51A4 5F32
6081 7722 9E33 5BC3 6E01 6E06 6E0A 6E15 60CC 6DF5
847E 68E9 84AC 870E 9E53 7BA2 9CF6 8735 99CC 92FA
9D1B 5B3D 9D77 7041 9F18 9F1D 5143 8D20 90A7 5458
56ED 6C85 676C 57A3 7230 8C9F 539F 54E1 5706 7B0E
8696 8881 53A1 915B 570E 63F4 6E72 7328 7F18 9228
9F0B 5712 5713 586C 5AB4 5AC4 6E90 6E92 733F 7342
849D 699E 69AC 8F95 7DE3 7E01 875D 876F 9B6D 571C
6A7C 7FB1 8597 8788 8B1C 8F45 9EFF 93B1 6ADE 908D
9A35 9DA2 9DB0 53B5 8FDC 76F6 903A 9060 5917 8099
59B4 82D1 6028 9662 57B8 884F 5086 5A9B 63BE 7457
7990 613F 88EB 8911 8924 566E 9858 66F0 66F1 7EA6
7D04 7BB9 77F1 5F5F 5F60 6708 6209 5216 6C4B 5C84
6288 793F 5CB3 6782 73A5 6071 94A5 6085 60A6 868E
868F 8ECF 94BA 9605 6373 8DC0 8DC3 7CA4 8D8A 9205
7CB5 925E 95B1 95B2 5B33 6A3E 7BD7 5DBD 9FA0 7C46
7039 8625 9EE6 721A 79B4 8E8D 7C65 9E11 7C70 9FA5
9E19 8480 7174 8495 7185 596B 8779 8D5F 9835 99A7
8D07 4E91 52FB 5300 4F1D 544D 56E9 5998 62A3 6C84
7EAD 82B8 6600 7547 7703 79D0 90E7 6DA2 7D1C 8018
803A 9116 96F2 612A 6C32 6EB3 7B7C 84B7 6C33 7189
6F90 8553 92C6 6A52 7BD4 7E1C 7E67 5141 962D 593D
628E 72C1 73A7 9668 837A 6B92 5597 9217 9695 6B9E
891E 99BB 78D2 9723 9F6B 9F73 5B55 8FD0 679F 90D3
607D 6655 9106 915D 508A 60F2 6120 7F0A 904B 614D
6688 816A 97EB 97F5 71A8 7DFC 8570 8574 7E15 8580
8CF1 9196 919E 992B 85F4 97D7 97DE 860A 97FB 5E00
531D 6C9E 5482 62F6 6CAF 685A 7D25 7D2E 9254 9B73
81DC 81E2 6742 7838 97F4 96D1 78FC 894D 96DC 56D0
96E5 707D 707E 753E 54C9 683D 70D6 83D1 6E3D 6EA8
7775 8CF3 5BB0 8F7D 5D3D 8F09 518D 5728 6257 6D05
50A4 9168 510E 7E21 5142 7CCC 7C2A 7C2E 9415 54B1
507A 5592 661D 5BC1 648D 5127 6512 5139 6522 8DB1
8DB2 6682 66AB 8CDB 8D5E 933E 913C 6FFD 8E54 9142
74C9 8D0A 93E8 74D2 9147 56CB 7052 8B83 74DA 79B6
8978 8B9A 9961 7242 7F98 8D43 8CCD 81E7 8CD8 8D13
9AD2 8D1C 9A75 99D4 5958 5F09 810F 585F 846C 92BA
81D3 81DF 50AE 906D 7CDF 8E67 91A9 51FF 947F 65E9
67A3 6806 86A4 68D7 7485 6FA1 74AA 85BB 85FB 7076
7681 7682 5515 5523 9020 688D 55BF 6165 7170 8241
566A 7C09 71E5 7AC3 8B5F 8DAE 8E81 7AC8 556B 4F2C
5219 629E 6CA2 62E9 6CCE 6CFD 8D23 8FEE 5247 5536
5567 5E3B 7B2E 8234 8CAC 6EAD 77E0 5616 5AE7 5E58
7BA6 8536 6A0D 6B75 8ACE 8D5C 64C7 6FA4 769F 7794
7C00 802B 790B 8957 8B2E 8CFE 880C 9F5A 9F70 9E05
4EC4 5928 5E82 6C44 6603 6617 6351 5D31 7A04 8D3C
8CCA 9C97 8808 9C02 9C61 600E 8C2E 8B56 8B5B 56CE
66FD 66FE 5897 912B 589E 618E 7F2F 6A67 71B7 7494
77F0 78F3 7F7E 7E52 8B44 9C5B 9503 92E5 7511 8D60
8D08 5412 8FCA 548B 62AF 6313 67E4 54F3 5067 55B3
63F8 6E23 6EA0 6942 5284 76B6 7B9A 6A1D 89F0 76BB
76BC 8B47 9F44 9F47 624E 672D 7534 8F67 8ECB 95F8
86BB 94E1 7160 7250 9598 9705 9358 8B57 538F 82F2
7728 781F 6429 9C8A 9C9D 8E37 9B93 9BBA 4E4D 7079
8BC8 54A4 5953 67F5 6805 70B8 5BB1 75C4 86B1 8A50
643E 6463 69A8 81AA 91A1 5908 7C82 635A 658B 658E
658F 6458 69B8 9F4B 5B85 7FDF 7A84 9259 503A 7826
50B5 5BE8 7635 6CBE 6BE1 65C3 6834 7C98 86C5 98E6
60C9 8A40 8D88 8A79 959A 8C35 5661 5DA6 6FB6 859D
9085 9711 6C08 6C0A 77BB 9E6F 65DC 8B6B 9958 9CE3
9A59 9B59 9C63 9E07 8B9D 62C3 65A9 98D0 5C55 76CF
5D2D 65AC 7416 640C 76DE 5D83 5D84 6990 8F97 98AD
5AF8 9186 6A4F 8E4D 8F3E 76BD 9EF5 5360 4F54 6218
6808 685F 7AD9 5061 7EFD 83DA 68E7 6E5B 6226 7DBB
5D98 8F1A 9AA3 6230 8665 8666 89B1 8F4F 8638 9A4F
5F20 5F21 5F35 7AE0 50BD 9123 5ADC 5F70 615E 6F33
7350 7CBB 8501 9067 66B2 6A1F 748B 9926 87D1 93F1
9A3F 9C46 9E9E 4EC9 6DA8 6DB1 638C 6F32 5E65 7903
979D 9423 4E08 4ED7 6259 5E10 6756 80C0 8D26 7C80
5E33 8139 75EE 969C 5887 5D82 5E5B 8CEC 762C 7634
7795 4F4B 948A 59B1 5DF6 62DB 662D 70A4 76C4 91D7
5541 924A 99CB 7ABC 9363 722B 627E 6CBC 7475 53EC
5146 8BCF 679B 5797 72E3 8D75 7B0A 8081 65D0 68F9
7F40 8A54 7167 7F69 7B8C 8087 8088 8D99 66CC 71F3
9BA1 6AC2 77BE 7F84 8707 5AEC 906E 5387 6298 6B7D
77FA 7813 7C77 8674 54F2 57D1 7C8D 88A9 5560 608A
6662 6663 8F84 5586 68CF 86F0 8A5F 8C2A 647A 8F12
6A00 78D4 8F19 92B8 8F99 87C4 569E 8B2B 8B3A 9BBF
8F4D 8B81 8975 8B8B 8005 9517 8D6D 8936 937A 8FD9
67D8 6D59 9019 6DDB 55FB 8517 6A1C 9E67 87C5 9DD3
8D1E 9488 4FA6 6D48 73CD 73CE 8C9E 5E2A 6815 6862
771E 771F 7827 796F 91DD 5075 6552 686D 9159 5BCA
6E5E 8474 9049 6438 659F 6968 7349 7504 798E 8496
84C1 9241 9755 699B 69C7 6B9D 7467 78AA 799B 6F67
7BB4 6A3C 6FB5 81FB 85BD 9331 8F43 9356 937C 7C48
9C75 5C52 8BCA 62AE 6795 59EB 5F2B 6623 80D7 8F78
755B 75B9 7715 8897 7D3E 8044 8419 88D6 8999 8A3A
8EEB 5AC3 7F1C 7A39 99D7 7E1D 7E25 8FB4 9B12 9EF0
5733 9635 7EBC 4FB2 630B 9663 9E29 632F 6715 681A
7D16 7739 8D48 5866 63D5 7D7C 8704 6576 8AAB 8CD1
92F4 9547 9707 9D06 93AD 93AE 9EEE 51E7 4E89 4F42
59C3 5F81 6014 722D 5CE5 6323 70A1 72F0 E853 70DD
7710 94B2 57E9 5D1D 5D22 6399 7319 7741 8047 94EE
5A9C 63C1 7B5D 5FB0 775C 84B8 9266 5FB4 7B8F 5FB5
E851 8E2D 7BDC 931A 9B07 7665 6C36 628D 7CFD 62EF
639F 5863 6678 6138 649C 6574 6B63 32A3 8BC1 8BE4
90D1 5E27 653F 75C7 5E40 8A3C 912D 8ACD 9D0A 8B49
4E4B 652F 536E 6C41 829D 5431 5DF5 6C65 679D 77E5
7EC7 80A2 5F94 6800 7957 79D3 79D6 80D1 80DD 8879
887C 5001 75B7 796C 79EA 8102 96BB 6894 6220 6925
81F8 6418 7994 7D95 69B0 8718 99B6 9CF7 8B22 9D32
7E54 8635 9F05 79B5 6267 4F84 5767 76F4 59EA 5024
503C 8040 8041 91DE 57F4 57F7 804C 690D 6B96 7983
7D77 8DD6 74E1 588C 646D 99BD 5B02 6179 6F10 8E2F
6A34 81B1 7E36 8077 87D9 8E60 8E62 8EC4 8E91 5902
6B62 53EA 51EA 52A7 65E8 962F 5740 5741 5E0B 627A
6C66 6C9A 7EB8 82B7 62A7 7949 830B 54AB 6049 6307
67B3 6D14 780B 8F75 6DFD 75BB 7D19 8A28 8DBE 8EF9
9EF9 916F 85E2 8967 9624 81F3 8296 5FD7 5FEE 627B
8C78 5236 5394 5781 5E19 5E1C 6CBB 7099 8D28 90C5
4FE7 5CD9 5EA2 5EA4 6303 67E3 6809 6D37 7951 965F
5A21 5F8F 631A 664A 684E 72FE 79E9 81F4 889F 8D3D
8F7E 4E7F 506B 5F9D 63B7 68BD 7318 7564 75D4 79F2
79F7 7A92 7D29 7FD0 88A0 89D7 8CAD 94DA 9E37 5082
5D3B 5F58 667A 6EDE 75E3 86ED 9A98 5BD8 5ECC 6431
6ECD 7A19 7A1A 7B6B 7F6E 8DF1 8F0A 9527 96C9 5886
69DC 6EEF 6F4C 7590 7608 88FD 899F 8A8C 928D 5E5F
6184 646F 6F6A 71AB 7A3A 81A3 89EF 8CEA 8E2C 92D5
65D8 7004 7DFB 96B2 99E4 9D19 5128 5295 61E5 64F2
64FF 6ADB 7A49 87B2 61EB 8D04 6ACD 74C6 89F6 9A2D
9BEF 7929 8C51 9A3A 9A47 8E93 9DD9 9455 8C52 4E2D
4F00 6C77 5223 5990 5F78 8FDA 5FE0 6CC8 7082 7EC8
67CA 76C5 8873 949F 822F 8877 7D42 9221 5E52 8520
953A 87A4 9D24 87BD 937E 9F28 8E71 9418 7C66 80BF
79CD 51A2 55A0 5C30 585A 6B71 7144 816B 7607 7A2E
8E35 4EF2 4F17 5995 72C6 794C 794D 833D 8876 91CD
869B 5045 773E 5839 5A91 7B57 8846 8AE5 5DDE 821F
8BCC E81B E81A 4F9C 5468 6D00 6D32 70BF 8BEA 70D0
73D8 8F80 90EE 5A64 5F9F 6DCD 77EA 9031 9E3C 558C
7CA5 8D52 8F08 9282 8CD9 8F16 970C 99F2 568B 76E9
8B05 9D43 9A06 8B78 59AF 8F74 8EF8 78A1 8098 5E1A
759B 83F7 666D 776D 7B92 9BDE 7EA3 4F37 546A 5492
5B99 7EC9 E825 E824 5191 54AE 663C 7D02 80C4 836E
665D 76B1 914E 7C99 8464 8A4B 7503 50FD 76BA 99CE
5663 7E10 9AA4 7C40 7C55 7C52 9A5F 6731 52AF 4F8F
8BDB 90BE 6D19 8331 682A 3231 73E0 8BF8 732A 7843
88BE 94E2 7D51 86DB 8A85 8DE6 69E0 6F74 876B 9296
6A65 8AF8 8C6C 99EF 9BA2 9D38 7026 85F8 6AE7 6AEB
9F04 9BFA 8829 7AF9 6CCF 7AFA 70A2 7B01 833F 70DB
7A8B 9010 7B1C 8233 7603 84EB 71ED 880B 8E85 9C41
529A 5B4E 705F 65B8 66EF 6B18 7225 883E 9483 4E36
4E3B 5B94 62C4 782B 7F5C 967C 6E1A 7151 716E 8A5D
5631 6FD0 9E88 77A9 5C6C 56D1 77DA 4F2B 4F47 4F4F
52A9 7EBB 82A7 82CE 577E 677C 6CE8 82E7 8D2E 8FEC
9A7B 58F4 67F1 67F7 6BB6 70B7 795D 75B0 771D 7969
7ADA 8387 79FC 7D35 7D38 7F9C 8457 86C0 5D40 7B51
8A3B 8CAF 8DD3 8EF4 94F8 7B6F 9252 98F3 99B5 58B8
7BB8 7FE5 6A26 92F3 99D0 7BC9 7BEB 9714 9E86 9444
6293 6A9B 81BC 7C3B 9AFD 722A 62FD 8DE9 4E13 53C0
5C02 7816 5C08 911F 5AE5 747C 750E 819E 989B 78DA
8AEF 87E4 9853 9C44 8F6C E83F 5B68 8EE2 7AF1 8F49
7077 556D 581F 8483 7451 50CE 8D5A 64B0 7BC6 9994
7E33 8948 8CFA 8B54 994C 56C0 7C51 5986 5E84 599D
5E92 8358 5A24 6869 838A 6E77 7CA7 88C5 88DD 6A01
7CDA 4E2C 58EE 58EF 72B6 72C0 58F5 6889 710B 5E62
649E 6205 96B9 8FFD 9A93 690E 9525 9310 9A05 9D7B
6C9D 5760 7B0D 5A37 7F00 60F4 7500 7F12 7577 787E
8187 589C 7DB4 8D58 7E0B 8AC8 918A 9323 991F 7908
8D05 8F5B 9446 5B92 8FCD 80AB 7A80 8C06 8AC4 8860
51C6 57FB 51D6 6E96 7DA7 8A30 7A15 5353 62D9 70AA
502C 6349 684C 68C1 6DBF 68F3 7438 7AA7 69D5 7A5B
7A71 883F 5734 5F74 72B3 707C 53D5 59B0 8301 65AB
6D4A 4E35 6D5E 70F5 8BFC 914C 5544 5545 5A3A 68B2
7740 65AE 666B 6913 7422 65B1 787A 7AA1 7F6C 64AF
64C6 65B2 799A 5285 8AC1 8AD1 92DC 6FC1 7BE7 64E2
6580 65B5 6FEF 6AE1 8B36 956F E84D 942F 9D6B 7042
8817 9432 7C57 9DDF 7C71 4ED4 5B56 5B5C 830A 5179
54A8 59D5 59FF 8332 6825 7386 7D0E 8D40 8D44 5D30
6DC4 79F6 7F01 8C18 8D7C 55DE 5B73 5D6B 6914 6E7D
6ECB 7CA2 8458 8F8E 9111 5B76 798C 89DC 8CB2 8CC7
8D91 9531 7A35 7DC7 922D 9543 9F87 8F1C 9F12 6FAC
8AEE 8DA6 8F3A 9319 9AED 9CBB 937F 93A1 983E 983F
9BD4 9D85 9F4D 9C26 9F5C 7C7D 5B50 5407 59C9 59CA
674D 77F7 79C4 80CF 5470 79ED 8014 8678 7B2B 6893
91E8 5559 7D2B 6ED3 8A3E 8A3F 699F 6A74 5B57 81EA
8293 8321 5033 525A 6063 7278 6E0D 7725 7726 80D4
80FE 6F2C 5528 5B97 5027 7EFC 9A94 582B 5D4F 5D55
60FE 68D5 7323 8159 847C 6721 6936 5D78 7A2F 7D9C
7DC3 71A7 7DF5 7FEA 8250 876C 8E28 8E2A 78EB 8C75
8E64 9A0C 9B03 9A23 9B09 9B37 9BEE 9BFC 9441 603B
506C 6374 60E3 6121 63D4 6403 50AF 84D7 6460 7DCF
7E02 71EA 7E3D 936F 93D3 7EB5 662E 75AD 500A 7314
7882 7CBD 7CC9 7632 7E26 931D 7E31 90B9 9A7A 8BF9
90F0 966C 63AB 83C6 68F7 68F8 9112 7B83 7DC5 8ACF
9139 9CB0 9BEB 9EC0 9A36 9F7A 8D71 8D70 9BD0 594F
63CD 5AB0 79DF 83F9 8445 84A9 5346 8DB3 5352 54EB
5D12 5D2A 65CF 50B6 7A21 7BA4 8E24 8E3F 955E 93C3
8BC5 963B 7EC4 4FCE 723C 73C7 7956 7D44 8A5B 977B
93BA 8B2F 5297 8E9C 945A 8EA6 947D 7E64 7F35 7E82
7E89 7C6B 7E98 94BB 63DD 6525 539C 6718 55FA 6A36
87D5 7E97 5D8A 5634 567F 6FE2 74BB 67A0 682C 7D4A
9154 666C 6700 797D 7F6A 8FA0 917B 855E 9189 5DB5
6A87 92F7 930A 6A8C 7A5D 6B08 5C0A 5D9F 9075 6A3D
7E5C 7F47 9D8E 940F 9CDF 9C52 9DF7 50D4 5642 6499
8B50 6358 928C 6628 79E8 838B 637D 690A 8443 7A13
7B70 923C 5DE6 4F50 7E53 4F5C 5750 963C 5C9D 5C9E
600D 4FB3 67DE 795A 80D9 5511 5EA7 888F 505A 8444
84D9 98F5 7CF3 5497
__END__

=head1 NAME

Unicode::Collate::CJK::Pinyin - weighting CJK Unified Ideographs
for Unicode::Collate

=head1 SYNOPSIS

    use Unicode::Collate;
    use Unicode::Collate::CJK::Pinyin;

    my $collator = Unicode::Collate->new(
        overrideCJK => \&Unicode::Collate::CJK::Pinyin::weightPinyin
    );

=head1 DESCRIPTION

C<Unicode::Collate::CJK::Pinyin> provides C<weightPinyin()>,
that is adequate for C<overrideCJK> of C<Unicode::Collate>
and makes tailoring of CJK Unified Ideographs in the order
of CLDR's pinyin ordering.

=head1 CAVEAT

The pinyin ordering includes some characters that are not
CJK Unified Ideographs and can't utilize C<weightPinyin()>
for collation. For them, use C<entry> instead.

=head1 SEE ALSO

=over 4

=item CLDR - Unicode Common Locale Data Repository

L<http://cldr.unicode.org/>

=item Unicode Locale Data Markup Language (LDML) - UTS #35

L<http://www.unicode.org/reports/tr35/>

=item L<Unicode::Collate>

=item L<Unicode::Collate::Locale>

=back

=cut
