/*
 *  audiopreview.c
 *
 *  Copyright 2009 Arnaud Soyez <weboide@codealpha.net>
 *
 *  This file is part of AudioPreview.
 *
 *  AudioPreview is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  AudioPreview is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with AudioPreview.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gio/gio.h>
#include <glib/gi18n-lib.h>
#include <gst/gst.h>
#include "ap_config.h"
#include "audiopreview.h"
#include "player.h"
#include "log.h"
#include "stream.h"
#include "playlist.h"
#include "functions.h"



int main (int argc, char *argv[])
{
  gchar           *tmp = NULL;
  GError          *error = NULL;
  GOptionContext  *context;
  GOptionGroup    *tmp_option_group;
  
  exit_status_code = EXIT_SUCCESS;
  
  /* Set prog name and init Gstreamer */
  g_set_prgname (PACKAGE);
  gst_init(NULL, NULL);
  
  /* Default Configuration */
  ap_config_init (&ap_config);
  
  /* set up the verbose/debug log handler */
  (void) g_log_set_handler (NULL,
                     G_LOG_LEVEL_INFO
                      | G_LOG_LEVEL_MESSAGE
                      | G_LOG_LEVEL_DEBUG
                      | G_LOG_LEVEL_WARNING,
                     log_handler,
                     NULL);
  
  /* Set the function to call in case of receiving signals:
   * we need to stop the main loop to exit correctly
   */
  (void) signal (SIGALRM,  error_quit);
  (void) signal (SIGHUP,   error_quit);
  (void) signal (SIGINT,   error_quit);
  (void) signal (SIGPIPE,  error_quit);
  (void) signal (SIGQUIT,  error_quit);
  (void) signal (SIGTERM,  error_quit);
  (void) signal (SIGTTIN,  SIG_IGN);
  (void) signal (SIGTTOU,  SIG_IGN);
  
  /* Handles any change of size for the terminal */
  (void) signal (SIGWINCH, terminal_size_changed);
  
  /* Define localization */
  ap_config.locale = setlocale (LC_ALL, "");
  if (!ap_config.locale)
  {
    g_error (_("Error during setting current locale. "
               "Falling back to default locale."));
    if (ap_config.locale)
      g_free (ap_config.locale);
    
    ap_config.locale = setlocale (LC_ALL, "C");
    if (!ap_config.locale)
    {
      g_critical (_("Couldn't set any locale. Exiting..."));
      exit (EXIT_FAILURE);
    }
  }
  
  /* i18n */
  bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);
  
            /* -------------- Options parsing ------------- */
  
  context = g_option_context_new (SHORTDESCRIPTION);
  g_option_context_set_summary (context, SUMMARY);
  g_option_context_set_description (context, BUG_REPORT_TEXT);
  g_option_context_add_main_entries (context, entries, GETTEXT_PACKAGE);
  g_option_context_add_group (context, gst_init_get_option_group ());
  
  tmp_option_group = g_option_group_new ("debug", _("Debugging Options:"),
                              _("Options to be used only for debugging or bug"
                              " reporting"), NULL, NULL);
  g_option_group_add_entries (tmp_option_group, debug_entries);
  
  g_option_context_add_group (context, tmp_option_group);
  
  // Let's parse args!
  if (!g_option_context_parse (context, &argc, &argv, &error))
  {
    g_printerr ( _("Error while parsing options: %s.\n"), error->message);
    g_printerr ( _("Run '%s --help' to see a full list of available command\
 line options.\n"), argv[0]);
    g_error_free (error);
    g_strfreev (remaining_args);
    g_option_context_free (context);
    exit (EXIT_FAILURE);
  }
  
  g_option_context_free (context);
  
  
  /* Colors */
  ap_config.want_colors = need_colors (ap_config.colors);
  
  /* Showing version */
  
  if (ap_config.show_version)
  {
    g_print ("%s\n\n%s\n\n%s\n", PACKAGE_STRING, COPYRIGHT, COPYRIGHT_TXT);
    g_strfreev (remaining_args);
    exit (EXIT_SUCCESS);
  }
  
  
  // Deactivate buffering for debug.
  if (ap_config.debug)
  {
    g_debug ("%s: Deactivating buffered output for stderr and stdout...",
             __FUNCTION__);
    
    setbuf (stdout, NULL);
    setbuf (stderr, NULL);
  }
  
  if (isatty (STDOUT_FILENO) == 1)
    g_debug ("\033[34mSTDOUT is a TTY.\033[m");
  else
    g_debug ("STDOUT is not a TTY.");
  
  if (ap_config.entirely)
  {
    ap_config.duration = -1;
    ap_config.position = START_PLAYING_START;
  }

  /* Arguments parsing */
  
  if (remaining_args)
  {
    int i = 0;
    while (remaining_args[i] != NULL)
    {
      playlist_add (remaining_args[i]);
      i++;
    }
  }
  
  g_strfreev (remaining_args);
  
  /* Parse playlist from FILE if any  (-f FILE)*/
  
  if (ap_config.playlist_from_file)
  {
    playlist_add_from_file (ap_config.playlist_from_file);
    g_free (ap_config.playlist_from_file);
  }
  
  /* -- Some configuration checks -- */
  
  if (ap_config.duration < 0 && ap_config.maxduration == 0)
    g_warning (_("Infinite streams (like internet radio) will play continuously."));
  
  if (ap_config.use_playbin2)
  {
    g_debug ("%s: Playbin2 Mode", __FUNCTION__);
  }
  else
  {
    g_debug ("%s: Playbin Mode", __FUNCTION__);
  }
  
  config_check_volume ();
  config_check_start_position ();
  
            /* ------------ Options parsing DONE ------------- */
  
  /* assign playlist to the config handler */
  ap_config.playlist = g_list_reverse (ap_config.playlist);
  
  if (!g_list_length (ap_config.playlist))
  {
    g_message ( _("No stream found.\n"));
    g_message ( _("Run '%s --help' to see the usage help.\n"), argv[0]);
  }
  
  /* Shuffle the playlist if requested */
  if (ap_config.shuffle)
    playlist_shuffle ();
  
  /* Prepare main loop */
  loop = g_main_loop_new (NULL, FALSE);
  player_init (done_playing, bus_call);
  g_idle_add ( (GSourceFunc) start_thread_wait_for_key, NULL);
  
  /* Main Loop */
  g_main_loop_run (loop);
  
  /* free memory */
  g_debug ("%s: Freeing memory", __FUNCTION__);
  playlist_free ();
  return exit_status_code;
}

/*============================================================================*/

/**
 * Requests to quit with the given error exit code
 */
void quit (gint error_status)
{
  g_print ("\n");
  g_debug ("%s: Stopping Main Loop (error code: %i)...",
           __FUNCTION__, error_status);
  
  // We set the exit status code
  exit_status_code = error_status;
  if (loop && g_main_loop_is_running(loop))
  {
    player_clear_pipeline ();
    g_main_loop_quit (loop);
  }
  else
    exit (error_status);
}

/**
 * Requests to quit with an error exit code
 */
void error_quit ()
{
  quit (EXIT_FAILURE);
}

/**
 * Called when done playing, this quits with a success exit code
 */
void done_playing ()
{
  g_debug ("%s: invoked.", __FUNCTION__);
  quit (EXIT_SUCCESS);
}


gboolean key_dispatcher (gint *car)
{
  //~ g_debug ("%s: Dispatching: '%c' (%d)", __FUNCTION__, car, car);
  
  if (car == NULL)
    return FALSE; // to get removed from the main loop
  
  switch (*car)
  {
    // PAUSE
    case ' ':
    {
      player_pause ();
      break;
    }
    
    
    // PREVIOUS stream
    case 'P':
    case 'p':
    {
      play_previous ();
      break;
    }
    
    // NEXT stream
    case 'N':
    case 'n':
    {
      play_next ();
      break;
    }
    
    // QUIT
    case 'Q':
    case 'q':
    {
      player_clear_pipeline ();
      done_playing_callback ();
      break;
    }
    
    // RESTART current stream
    case 'R':
    case 'r':
    {
      player_set_position (ap_config.position);
      break;
    }
  }
  
  g_free (car);
  return FALSE; // to get removed from the main loop
}

/**
 * Starts the function wait_for_key in a thread.
 */
gboolean start_thread_wait_for_key ()
{
  GError  *error = NULL;

  g_debug("Starting thread wait for key");
  if (!g_thread_create ((GThreadFunc) wait_for_key, &key_dispatcher, FALSE, &error))
  {
    g_error (_("Thread failed: %s"), error->message);
    return FALSE;
  }

  return FALSE; // to get removed from the main loop
}
