
# LIBTBX_SET_DISPATCHER_NAME cctbx.xfel.powder_from_spots
from __future__ import division
import logging

from iotbx.phil import parse
from dials.util import log
from dials.util import show_mail_on_error
from dials.util.options import OptionParser
from xfel.small_cell.spotfinder_radial_average import Spotfinder_radial_average

logger = logging.getLogger("dials.command_line.powder_from_spots")

help_message = """
Script to synthesize a powder pattern from DIALS spotfinding output
Example usage:
cctbx.xfel.powder_from_spots all.expt all.refl

This computes d-spacings of peak maxima in a reflections file as generated by
dials.find_spots. The results are binned and plotted as a histogram. Plotting
only maxima (and thus discarding the rest of the peak profile) has a large
sharpening effect; the resulting patterns are better than those obtained by
synchrotron powder diffraction.

The input is a single combined .refl file and the corresponding .expt file.
For small-molecule data, 10k to 20k shots are a minimum for good results.
Consider filtering the input by number of spots using the min_... and
max_reflections_per_experiment options of dials.combine_experiments. Min=3
and max=15 are a good starting point for small-molecule samples.

An excellent reference geometry (rmsd <<1 px) is important. A current detector
metrology refined from a protein sample is probably the best approach. Try a
plot with split_panels=True to confirm that the patterns on each panel agree.
In a data set from the MPCCD detector at SACLA we found that the Tau2 and Tau3
tilts (the tilts around the detector fast and slow axes) had to be refined for
each panel.
"""

phil_scope = parse(
    """
  file_path = None
    .type = str
    .multiple = True
    .help = Files to read
  n_bins = 3000
    .type = int
    .help = Number of bins in the radial average
  d_max = 20
    .type = float
  d_min = 1.4
    .type = float
  verbose = True
    .type = bool
    .help = Extra logging information
  panel = None
    .type = int
    .help = Only use data from the specified panel
  reference_geometry = None
    .type = path
    .help = Apply this geometry before creating average. Not implemented.
  unit_cell = None
    .type = unit_cell
    .help = Show positions of miller indices from this unit_cell and space \
            group. Not implemented.
  space_group = None
    .type = space_group
    .help = Show positions of miller indices from this unit_cell and space \
            group. Not implemented.
  peak_position = *xyzobs shoebox
    .type = choice
    .help = By default, use the d-spacing of the peak maximum. Shoebox: Use the \
            coordinates of every pixel in the reflection shoebox. This entails \
            intensity-weighted peaks.
  peak_weighting = *unit intensity
    .type = choice
    .help = The histogram may be intensity-weighted, but the results are \
            typically not very good.
  downweight_weak = 0
    .type = float
    .help = Subtract a constant from every intensity. May help filter out \
            impurity peaks.
  split_panels = False
    .type = bool
    .help = Plot a pattern for each detector panel.
  xyz_offset = 0. 0. 0.
    .type = floats
    .help = origin offset in millimeters
output {
  log = dials.powder_from_spots.log
    .type = str
  xy_file = None
    .type = str
}
"""
)





class Script(object):
  def __init__(self):
    usage = "$ cctbx.xfel.powder_from_spots EXPERIMENTS REFLECTIONS [options]"
    self.parser = OptionParser(
        usage=usage,
        phil=phil_scope,
        epilog=help_message,
        check_format=False,
        read_reflections=True,
        read_experiments=True,
        )

  def run(self):
    params, options = self.parser.parse_args(show_diff_phil=False)
    averager = Spotfinder_radial_average(params)
    averager.calculate()
    averager.plot()


if __name__ == "__main__":
  with show_mail_on_error():
    script = Script()
    script.run()
