/* $Id: kmo_copy-test.c,v 1.4 2013-10-08 11:18:56 aagudo Exp $
 *
 * This file is part of the KMOS Library
 * Copyright (C) 2002-2006 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: aagudo $
 * $Date: 2013-10-08 11:18:56 $
 * $Revision: 1.4 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <sys/stat.h>

#include <cpl.h>

#include "kmo_dfs.h"

const char  *test_global_path_test_data;
float   test_global_seed_data, test_global_seed_noise;
int     test_global_size_x, test_global_size_y, test_global_size_z, test_global_nr_frames;
void kmo_get_pipe_command(char*, const char*, const char*, int);
int kmo_test_file_exists(const char*);
void kmo_test_verbose_off();
void kmo_test_verbose_on();
double kmo_test_esorex_data(const char*, int);
const char* kmo_test_cat_strings(char*, const char*, const char*);
int  kmo_test_create_RAW_data(int, int, const char*, char**, char**, int*, int, char**, char**, int*, int, float, float);
int  kmo_test_create_RAW_data_zero(int, int, const char*);
int  kmo_test_create_F2D_data(int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2D_data_noise(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2I_data(int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2I_data_noise(int, int, int, const char*, char *, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data(int, int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data_infinite(int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data2(int, int, int, int, int*, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data_noise(int, int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F1I_data(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F1I_data_noise(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_B2D_data(int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);

const char      *path_recipe       = "kmo_copy/";

const char      *valid_files[]     = {"v_data_F3I.fits",
                                      "v_data_noise_F3I.fits",
                                      "v_data_F2I.fits",
                                      "v_data_noise_F2I.fits",
                                      "v_data_F1I.fits",
                                      "v_data_noise_F1I.fits",
                                      "v_data_F3I_2_false_true.fits"};

const char      *invalid_files[]   = {"i_data_RAW.fits",
                                      "i_data_F2D.fits",
                                      "i_data_F3I_false.fits"};

/**
    @defgroup kmo_copy_test  kmo_copy_test unit tests

    @{
*/

/**
  @brief   test
  @param   ifu     If equal to INT_MIN then --ifu isn't specified
  @param   x       If equal to INT_MIN then --x1 isn't specified
  @param   y       If equal to INT_MIN then --y1 isn't specified
  @param   z       If equal to INT_MIN then --z1 isn't specified
  @param   xsize   If equal to INT_MIN then --x2 isn't specified
  @param   ysize   If equal to INT_MIN then --y2 isn't specified
  @param   zsize   If equal to INT_MIN then --z2 isn't specified
 */
static void
test_copy(const char *path_op1,
          int ifu,
          int x, int xsize,
          int y, int ysize,
          int z, int zsize,
          int ret_val)
{
    // create sof-file
    const char *sof_path = "test_copy.sof";
    FILE *fh = fopen( sof_path, "w" );
    fprintf (fh, "%s\n", path_op1);
    fclose(fh);

    // create esorex-command
    char tmp[512];
    char esorex_command[512];
    sprintf(esorex_command,
            "esorex --suppress-prefix=TRUE kmo_copy ");

    if (ifu != INT_MIN) {
        sprintf(tmp, "--ifu=%d ", ifu);
        strcat(esorex_command, tmp);
    }
    if (x != INT_MIN) {
        sprintf(tmp, "--x=%d ", x);
        strcat(esorex_command, tmp);
    }
    if (y != INT_MIN) {
        sprintf(tmp, "--y=%d ", y);
        strcat(esorex_command, tmp);
    }
    if (z != INT_MIN) {
        sprintf(tmp, "--z=%d ", z);
        strcat(esorex_command, tmp);
    }
    if (xsize != INT_MIN) {
        sprintf(tmp, "--xsize=%d ", xsize);
        strcat(esorex_command, tmp);
    }
    if (ysize != INT_MIN) {
        sprintf(tmp, "--ysize=%d ", ysize);
        strcat(esorex_command, tmp);
    }
    if (zsize != INT_MIN) {
        sprintf(tmp, "--zsize=%d ", zsize);
        strcat(esorex_command, tmp);
    }

    strcat(esorex_command, sof_path);

    kmo_get_pipe_command(esorex_command,
                         "log_kmo_copy.txt", sof_path, TRUE);

    // execute esorex-command
    if (ret_val == 0) {
        cpl_test_eq(0, system(esorex_command));
    } else {
        cpl_test_noneq(0, system(esorex_command));
    }
}

/**
    @brief
        Generates test data for kmo_copy.

    @param path      Path where the generated test data should be saved to.
*/
static int kmo_generate_test_data_copy(const char *path)
{
    char        file_path[256];

    /* ----- valid test data ----- */
    /* F3I */
    cpl_test_eq(0,
            kmo_test_create_F3I_data(test_global_size_x, test_global_size_y, test_global_size_z,  FALSE, test_global_nr_frames,
                          kmo_test_cat_strings(file_path, path, valid_files[0]),
                          NULL, NULL, NULL, 0,
                          NULL, NULL, NULL, 0));

    cpl_test_eq(0,
            kmo_test_create_F3I_data_noise(test_global_size_x, test_global_size_y, test_global_size_z,  FALSE,
                          test_global_nr_frames,
                          kmo_test_cat_strings(file_path, path, valid_files[1]),
                          NULL, NULL, NULL, 0,
                          NULL, NULL, NULL, 0));

    /* F2I */
    cpl_test_eq(0,
            kmo_test_create_F2I_data(test_global_size_x, test_global_size_y, FALSE, test_global_size_z,
                          kmo_test_cat_strings(file_path, path, valid_files[2]),
                          NULL, NULL, NULL, 0,
                          NULL, NULL, NULL, 0));

    cpl_test_eq(0,
            kmo_test_create_F2I_data_noise(test_global_size_x, test_global_size_y, test_global_size_z,
                          kmo_test_cat_strings(file_path, path, valid_files[3]),
                          NULL, NULL, NULL, 0,
                          NULL, NULL, NULL, 0));

    /* F1I */
    cpl_test_eq(0,
            kmo_test_create_F1I_data(test_global_size_z, test_global_nr_frames,
                          kmo_test_cat_strings(file_path, path, valid_files[4]),
                          NULL, NULL, NULL, 0,
                          NULL, NULL, NULL, 0));

    cpl_test_eq(0,
            kmo_test_create_F1I_data_noise(test_global_size_z, test_global_nr_frames,
                          kmo_test_cat_strings(file_path, path, valid_files[5]),
                          NULL, NULL, NULL, 0,
                          NULL, NULL, NULL, 0));

    int val1[] = {FALSE, TRUE};

    cpl_test_eq(0,
            kmo_test_create_F3I_data2(test_global_size_x, test_global_size_y, test_global_size_z, 2, val1,
                          kmo_test_cat_strings(file_path, path, valid_files[6]),
                          NULL, NULL, NULL, 0,
                          NULL, NULL, NULL, 0));

    /* ----- invalid test data ----- */
    /* RAW */
    cpl_test_eq(0,
            kmo_test_create_RAW_data(test_global_size_x, test_global_size_y,
                        kmo_test_cat_strings(file_path, path, invalid_files[0]),
                        NULL, NULL, NULL, 0,
                        NULL, NULL, NULL, 0, test_global_seed_data, 0.0));

    /* F2D */
    cpl_test_eq(0,
            kmo_test_create_F2D_data(test_global_size_x, test_global_size_y, FALSE,
                        kmo_test_cat_strings(file_path, path, invalid_files[1]),
                        NULL, NULL, NULL, 0,
                        NULL, NULL, NULL, 0));

    int val2[] = {FALSE};

    cpl_test_eq(0,
            kmo_test_create_F3I_data2(test_global_size_x, test_global_size_y, test_global_size_z, 1, val2,
                         kmo_test_cat_strings(file_path, path, invalid_files[2]),
                         NULL, NULL, NULL, 0,
                         NULL, NULL, NULL, 0));

    return 0;
}

/**
  @brief    Test of kmo_copy recipe.
  @param    argc   the number of parameters
  @param    argv   the parameter list

  Call @c kmo_copy-test @c --generate when only the test data for this
  recipe should be generated.
  Call @c kmo_copy-test @c --no-gen when only the tests should be
  executed and no data should be generated.

 */
int main(int argc, char *argv[])
{
    float   tol = 0.01;

    char    test_path[256],
            in_file[256];

    cpl_test_init("<kmos-spark@mpe.mpg.de>", CPL_MSG_WARNING);

    strcpy(test_path, test_global_path_test_data);
    strcat(test_path, path_recipe);

    if (kmo_test_file_exists(test_path) == FALSE) {
        mkdir(test_path, 0777);
    }

    if ((argc == 1) || ((argc > 1) && (strcmp(argv[1], "--no-gen") != 0))) {
        cpl_test_zero(kmo_generate_test_data_copy(test_path));
    }

    if ((argc > 1) && (strcmp(argv[1], "--generate") == 0)) {
        /* just generate the data */
        return cpl_test_end(0);
    }

    /* ----- run the tests ----- */
    FILE *fd = fopen("log_kmo_copy.txt", "w");
    fprintf(fd, "%s\n", " ");
    fclose(fd);

    /* ----- valid tests ----- */
    //
    // in-file: F3I, data only
    //
    kmo_test_verbose_off();

    kmo_test_cat_strings(in_file, test_path, valid_files[0]);

    // copy scalar (invalid IFU selected)
    test_copy(in_file, 2, 1, INT_MIN, 3, INT_MIN, 1, INT_MIN, 0);
    cpl_test_abs(kmo_test_esorex_data(COPY, 0), 0.0, tol);

    kmo_test_verbose_on();

    // copy scalar
    test_copy(in_file, 3, 1, INT_MIN, 3, INT_MIN, 1, INT_MIN, 0);
    cpl_test_abs(kmo_test_esorex_data(COPY, 0), 18.0, tol);

    //   copy x-vector
    test_copy(in_file, INT_MIN, 1, 3, 3, INT_MIN, 1, INT_MIN, 0);
    cpl_test_abs(kmo_test_esorex_data(COPY, 0), 57.3, tol);

    //
    // in-file: F3I, data and noise
    //
    kmo_test_cat_strings(in_file, test_path, valid_files[1]);

    // copy scalar
    test_copy(in_file, INT_MIN, 1, INT_MIN, 3, INT_MIN, 1, INT_MIN, 0);
    cpl_test_abs(kmo_test_esorex_data(COPY, 0), 78.0, tol);

    // copy x-vector
    test_copy(in_file, 1, 1, 3, 3, INT_MIN, 1, INT_MIN, 0);
    cpl_test_abs(kmo_test_esorex_data(COPY, 0), 13.2, tol);

    // copy y-vector
    test_copy(in_file, INT_MIN, 4, INT_MIN, 1, 6, 3, INT_MIN, 0);
    cpl_test_abs(kmo_test_esorex_data(COPY, 0), 109.8, tol);

    test_copy(in_file, 4, 5, INT_MIN, 5, 2, 7, INT_MIN, 0);
    cpl_test_abs(kmo_test_esorex_data(COPY, 0), 71.8, tol);

    // copy z-vector
    test_copy(in_file, INT_MIN, 1, INT_MIN, 1, INT_MIN, 1, 7, 0);
    cpl_test_abs(kmo_test_esorex_data(COPY, 0), 60.0, tol);

    test_copy(in_file, 4, 5, INT_MIN, 6, INT_MIN, 4, 2, 0);
    cpl_test_abs(kmo_test_esorex_data(COPY, 0), 69.8, tol);

    // copy z-image
    test_copy(in_file, INT_MIN, 1, 5, 1, 6, 1, INT_MIN, 0);
    cpl_test_abs(kmo_test_esorex_data(COPY, 0), 94.2, tol);

    test_copy(in_file, 4, 4, 2, 5, 2, 7, INT_MIN, 0);
    cpl_test_abs(kmo_test_esorex_data(COPY, 0), 71.2, tol);

    // copy x-image
    test_copy(in_file, 4, 5, INT_MIN, 1, 6, 1, 7, 0);
    cpl_test_abs(kmo_test_esorex_data(COPY, 0), 53.8, tol);

    test_copy(in_file, INT_MIN, 1, INT_MIN, 1, 2, 1, 2, 0);
    cpl_test_abs(kmo_test_esorex_data(COPY, 0), 54, tol);

    // copy y-image
    test_copy(in_file, 4, 3, 3, 6, INT_MIN, 6, 2, 0);
    cpl_test_abs(kmo_test_esorex_data(COPY, 0), 72.6, tol);

    test_copy(in_file, INT_MIN, 1, 5, 1, INT_MIN, 1, 7, 0);
    cpl_test_abs(kmo_test_esorex_data(COPY, 0), 67.1999, tol);

    // copy whole cube
    test_copy(in_file, 1, 1, test_global_size_x, 1, test_global_size_y, 1, test_global_size_z, 0);
    cpl_test_abs(kmo_test_esorex_data(COPY, 0), 23.4, tol);

    // copy sub cube
    test_copy(in_file, INT_MIN, 1, 2, 4, 3, 4, 4, 0);
    cpl_test_abs(kmo_test_esorex_data(COPY, 0), 142.8, tol);

    // test --x1=5 --xsize=1 equals copy x-image
    test_copy(in_file, 4, 5, 1, 1, 6, 1, 7, 0);
    cpl_test_abs(kmo_test_esorex_data(COPY, 0), 53.8, tol);

    //
    // in-file: F2I, data and noise
    //
    kmo_test_cat_strings(in_file, test_path, valid_files[3]);

    // copy scalar
    test_copy(in_file, INT_MIN, 5, INT_MIN, 6, INT_MIN, INT_MIN, INT_MIN, 0);
    cpl_test_abs(kmo_test_esorex_data(COPY, 0), 151.2, tol);

    // copy x-vector
    test_copy(in_file, INT_MIN, 1, 5, 1, INT_MIN, INT_MIN, INT_MIN, 0);
    cpl_test_abs(kmo_test_esorex_data(COPY, 0), 21.6, tol);

    // copy y-vector
    test_copy(in_file, INT_MIN, 4, INT_MIN, 1, 6, INT_MIN, INT_MIN, 0);
    cpl_test_abs(kmo_test_esorex_data(COPY, 0), 86.4, tol);

    // copy whole image
    test_copy(in_file, 1, 1, test_global_size_x, 1, test_global_size_y, INT_MIN, INT_MIN, 0);
    cpl_test_abs(kmo_test_esorex_data(COPY, 0), 17.4, tol);

    // copy sub image
    test_copy(in_file, INT_MIN, 2, 2, 2, 4, INT_MIN, INT_MIN, 0);
    cpl_test_abs(kmo_test_esorex_data(COPY, 0), 79.2, tol);

    //
    // in-file: F1I, data and noise
    //
    kmo_test_cat_strings(in_file, test_path, valid_files[5]);

    // copy scalar
    test_copy(in_file, INT_MIN, 7, INT_MIN, INT_MIN, INT_MIN, INT_MIN,
              INT_MIN, 0);
    cpl_test_abs(kmo_test_esorex_data(COPY, 0), 31.6, tol);

    // copy vector
    test_copy(in_file, 3, 1, 7, INT_MIN, INT_MIN, INT_MIN, INT_MIN, 0);
    cpl_test_abs(kmo_test_esorex_data(COPY, 0), 7.6, tol);

    // copy vector
    test_copy(in_file, INT_MIN, 2, 2, INT_MIN, INT_MIN, INT_MIN, INT_MIN, 0);
    cpl_test_abs(kmo_test_esorex_data(COPY, 0), 15.3999, tol);

    // copy sub cube, first IFU is invalid
    kmo_test_cat_strings(in_file, test_path, valid_files[6]);
    test_copy(in_file, INT_MIN, 1, 2, 4, 3, 4, 4, 0);
    cpl_test_abs(kmo_test_esorex_data(COPY, 0), 27.05, tol);

    /* ----- invalid data ----- */
    // RAW
    kmo_test_cat_strings(in_file, test_path, invalid_files[0]);
    test_copy(in_file, 100, 100, 100, 100, 100, 100, 100, -1);

    // F2D
    kmo_test_cat_strings(in_file, test_path, invalid_files[1]);
    test_copy(in_file, 100, 100, 100, 100, 100, 100, 100, -1);

    kmo_test_cat_strings(in_file, test_path, valid_files[0]);

    // F3I too large x1
    test_copy(in_file, 1, 100, 1, 1, test_global_size_y, 1, test_global_size_z, -1);

    // F3I too large y1
    test_copy(in_file, 1, 1, 2, 100, 1, 1, test_global_size_z, -1);

    // F3I too large z1
    test_copy(in_file, 1, 1, 2, 1, 3, 100, 1, -1);

    // F3I too large xsize
    test_copy(in_file, 1, 1, 100, 1, test_global_size_y, 1, test_global_size_z, -1);

    // F3I too large ysize
    test_copy(in_file, 1, 1, 2, 1, 100, 1, test_global_size_z, -1);

    // F3I too large zsize
    test_copy(in_file, 1, 1, 2, 1, 3, 1, 100, -1);

    // F3I too small x1
    test_copy(in_file, 1, 0, 1, 1, test_global_size_y, 1, test_global_size_z, -1);

    // F3I too small y1
    test_copy(in_file, 1, 1, 3, 0, test_global_size_y, 1, test_global_size_z, -1);

    // F3I too small z1
    test_copy(in_file, 1, 1, 4, 1, test_global_size_y, 0, test_global_size_z, -1);

    // copy sub cube, IFU is invalid
    kmo_test_cat_strings(in_file, test_path, invalid_files[2]);
    test_copy(in_file, INT_MIN, 1, 2, 4, 3, 4, 7, -1);

    return cpl_test_end(0);
}

/** @} */
