package Lire::DlfAnalyserProcess;

use strict;

use Lire::PluginManager;
use Lire::DlfQuery;
use Lire::DlfSchema;
use Lire::FilterSpec;
use Lire::FilterExpr;
use Lire::Utils qw/ check_param check_object_param period_range /;

use POSIX qw/ strftime /;

use Carp;

=pod

=head1 NAME

Lire::DlfAnalyseProcess - Object that controls the analysis process.

=head1 SYNOPSIS

  use Lire::DlfStore;
  use Lire::DlfAnalyserProcess;

  my $store = Lire::DlfStore->open( "store" );

  my $process = new Lire::DlfAnalyserProcess( $store, $analyser_name,
                                              $analyser_config, $dlf_source );

  $process->run_analysis_job();

  print "DLF records created: ", $process->dlf_count(),    "\n";
  print "Errors encountered: ",  $process->errors_count(), "\n";

=head1 DESCRIPTION

This object encapsulates the Lire DLF analysis process. It takes as a
Lire::DlfStore, the name of the analyser, its configuration and
optionally, a dlf_source ID. When a dlf_source is used, the analysis
will only be done using the DLF coming from that source.

The object provides the API to the converter. Methods are also
available to query information on the conversion process.

=head1 new( $store, $analyser_name, $analyser_config, [ $dlf_source ] );

Create a Lire::DlfAnalysisProcess.

=cut

sub new {
    my ( $class, $store, $analyser, $config, $dlf_source ) = @_;

    check_object_param( $store, 'store', 'Lire::DlfStore' );
    check_param( $analyser, 'analyser_name' );
    check_param( $config, 'analyser_config' );

    croak "no analyser '$analyser' was registered"
      unless Lire::PluginManager->has_plugin( 'dlf_analyser', $analyser );

    return bless { '_store' => $store,
                   '_dlf_source' => $dlf_source,
                   '_analyser' => $analyser,
                   '_analyser_config' => $config,
                 }, $class;
}


=pod

=head2 run_analysis_job()

Import the log data from ImportJob as DLF. This method will throw an
exception if it is called more than once.

=cut

sub run_analysis_job {
    my ( $self, $time ) = @_;

    $time ||= time;
    $self->_init_process( $time );

    my $analyser = Lire::PluginManager->get_plugin( 'dlf_analyser',
                                                    $self->{'_analyser'} );

    eval {
        $analyser->analyse( $self, $self->{'_analyser_config'} );
    };
    $self->error( $@ ) if $@;

    $self->_save_analysis_stats();

    $self->{'_log_stream'}->close();
    delete $self->{'_log_stream'};
    $self->{'_dlf_stream'}->close();
    delete $self->{'_dlf_stream'};

    return;
}

sub _init_process {
    my ($self, $time ) = @_;

    my $analyser = Lire::PluginManager->get_plugin( 'dlf_analyser',
                                                    $self->{'_analyser'} );

    $self->{'_time'} = $time;
    $self->{'_dlf_count'} = 0;
    $self->{'_error_count'} = 0;
    $self->{'_job_id'} = strftime( "$self->{'_analyser'}-%Y%m%d_%H%M%S",
                                   localtime( $time ) );

    $self->{'_dlf_stream'} =
      $self->{'_store'}->open_dlf_stream( $analyser->dst_schema(), 'w' );
    $self->{'_log_stream'} =
      $self->{'_store'}->open_dlf_stream( 'lire_import_log', 'w' );

    return;
}

sub _save_analysis_stats {
    my $self = $_[0];

    my $stream = $self->{'_store'}->open_dlf_stream( 'lire_import_stats', 'w');
    $stream->write_dlf( { 'time_start' => $self->{'_time'},
                          'elapsed' => time() - $self->{'_time'},
                          'error_count' => $self->{'_error_count'},
                          'dlf_count' => $self->{'_dlf_count'},
                          'job_name' => $self->{'_analyser'},
                          'job_id' => $self->{'_job_id'},
                        } );
    $stream->close();

    return;
}

=pod

=head2 job_id()

Returns the job identifier associated to this process.

=cut

sub job_id {
    return $_[0]{'_job_id'};
}

=pod

=head2 dlf_store()

Returns the Lire::DlfStore in which this conversion process is storing
the DLF records.

=cut

sub dlf_store {
    return $_[0]{'_store'};
}

=pod

=head2 dlf_source()

Returns the source from which the DLF should come in the src_schema to
be analysed.

=cut

sub dlf_source {
    return $_[0]{'_dlf_source'};
}

=pod

=head2 dlf_analyser()

Returns the name of the analyser which will be run.

=cut

sub dlf_analyser {
    return $_[0]{'_analyser'};
}

=pod

=head2 dlf_analyser_config()

Returns the analysis configuration data that should be used by the
converter.

=cut

sub dlf_analyser_config {
    return $_[0]{'_analyser_config'};
}

=pod

=head2 dlf_count()

Returns the number of DLF records created.

=cut

sub dlf_count { 
    return $_[0]{'_dlf_count'};
}

=pod

=head2 error_count()

Returns the number of errors encountered in the conversion process.

=cut

sub error_count { $_[0]{'_error_count'} }

=pod

=head2 source_filter()

Returns a Lire::FilterExpr which should be used to limit the DLF
records to analyse.

This method returns null if the whole DLF stream should be analysed.

=cut

sub source_filter {
    my $self = $_[0];

    return undef
      unless defined $self->{'_dlf_source'};
    my $plugin = Lire::PluginManager->get_plugin( 'dlf_analyser',
                                                  $self->{'_analyser'} );
    my $schema = Lire::DlfSchema::load_schema( $plugin->src_schema() );

    my $spec = new Lire::FilterSpec();
    $spec->superservice( $schema->superservice() );
    $spec->schema( $schema->id() )
      if $schema->id() ne $schema->superservice();

    return new Lire::FilterExpr::DlfSource( 'container' => $spec,
                                            'value' => $self->{'_dlf_source'} );
}

=pod

=head1 API FOR THE DLF ANALYSERS

This is the object that encapsulates the DLF implementation and hides
the complexitity of the storage framework from the DLF analysers. It
offers the following methods to the DLf converter.

=head2 write_dlf(  $dlf, [$related_to] )

This writes the $dlf DLF record conforming the $schema's schema in the
Lire::DlfStore. $Dlf is an hash reference. Keys are the schema's field
name. Undefined value means that this field isn't available in that
record.

When writing to a derived schema, the $related_to parameter can be
an arrayr reference containing the dlf_ids of the related record.

=cut

sub write_dlf {
    my ( $self, $dlf, $dlf_ids ) = @_;

    check_object_param( $dlf, 'dlf', 'HASH' );
    check_object_param( $dlf_ids, 'related_to', 'ARRAY' )
      if defined $dlf_ids;

    $dlf->{'dlf_source'} = $self->{'_job_id'};
    $self->{'_dlf_stream'}->write_dlf( $dlf, $dlf_ids );
    $self->{'_dlf_count'}++;

    return;
}

=pod

=head2 error( $error_msg );

Method that should be used by the Lire::Analyser to report that an
error was encountered during the analysis process. $error_msg should
be used to report the nature of the error. 

=cut

sub error {
    my ( $self, $error_msg ) = @_;

    check_param( $error_msg, 'error_msg' );

    $self->{'_log_stream'}->write_dlf( { 'time' => time(),
                                         'job_name' => $self->{'_analyser'},
                                         'job_id' => $self->{'_job_id'},
                                         'type' => 'error',
                                         'msg' => $error_msg,
                                       }
                                     );
    $self->{'_error_count'}++;

    return;
}

# keep perl happy
1;

__END__

=pod

=head1 SEE ALSO

Lire::DlfStore(3pm) Lire::DlfAnalyser(3pm)

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=head1 VERSION

$Id: DlfAnalyserProcess.pm,v 1.10 2006/07/23 13:16:28 vanbaal Exp $

=head1 COPYRIGHT

Copyright (C) 2004 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html.

=cut

