package tests::ChartTypesScatTest;

use strict;

use base qw/Lire::Test::PluginTestCase/;

use Lire::ChartTypes::Scat;
use Lire::Config::Build qw/ac_info/;
use Lire::Config::ChartSpec;
use Lire::Config::ExecutableSpec;
use Lire::PluginManager;
use Lire::ReportParser::ReportBuilder;
use Lire::Utils qw/tempdir/;

use File::Basename qw/dirname/;
use IO::Scalar;

sub set_up {
    my $self = $_[0];
    $self->SUPER::set_up();

    $self->{'cfg'}{'ploticus_path'} =
      new Lire::Config::ExecutableSpec( 'name' => 'ploticus_path' )->instance( 'value' => ac_info( 'PLOTICUS' ) );
    $self->{'cfg'}{'lr_chart_font'} = '';
    $self->{'chart_cfg'} = new Lire::Config::ChartSpec( 'name' => 'chart' )->instance();

    return;
}

sub create_plugin {
    return new Lire::ChartTypes::Scat();
}

sub properties_spec_file {
    return dirname( __FILE__ ) . "/../Lire/ChartTypes/ct_scat.xml";
}

sub registration_file {
    return dirname( __FILE__ ) . "/../Lire/ChartTypes/ct_scat_init";
}

sub set_up_subreport {
    my $self = $_[0];

    my $parser = new Lire::ReportParser::ReportBuilder();
    my $report = $parser->parsefile( dirname( __FILE__ ) . "/data/test-sqlite.xml" );
    $self->{'subreport'} = $report->subreport_by_id( 'downloads-by-period.0' );
}

sub test_get_vars {
    my $self = $_[0];

    $self->set_up_subreport();
    $self->test_registration_file();
    my $scat = Lire::PluginManager->get_plugin( 'chart_type', 'scat' );

    my $info = $self->{'subreport'}->table_info();
    $info->set_variables_indices();

    $self->{'chart_cfg'}->type( $scat );
    $self->{'chart_cfg'}->get( 'case_var' )->set( 'timegroup:time_start' );

    my $props = $self->{'chart_cfg'}->type_properties();
    $props->get( 'x' )->set( 'user_count' );
    $props->get( 'y' )->set( 'download_count' );
    $self->assert_deep_equals( [$info->column_info_by_name( 'user_count' ),
                                $info->column_info_by_name( 'download_count' ),
                               ], $scat->get_vars( $self->{'chart_cfg'} ) );
}

sub test_write_parameters {
    my $self = $_[0];

    $self->set_up_subreport();
    my $fh = new IO::Scalar();
    my $buf = $fh->sref();
    $self->test_registration_file();
    my $info = $self->{'subreport'}->table_info();
    $info->set_variables_indices();
    my $scat = Lire::PluginManager->get_plugin( 'chart_type', 'scat' );
    $self->{'chart_cfg'}->type( $scat );
    $self->{'chart_cfg'}->get( 'case_var' )->set( 'timegroup:time_start' );

    my $props = $self->{'chart_cfg'}->type_properties();
    $props->get( 'x' )->set( 'user_count' );
    $props->get( 'y' )->set( 'download_count' );
    $scat->write_parameters( $self->{'chart_cfg'}, $fh );
    $self->assert_str_equals( $$buf, <<EOF );
#set name = Period
#set x = 2
#set xlbl = Unique Users
#set y = 3
#set ylbl = Downloads
#set corr = no
EOF

    $self->{'chart_cfg'}->title( 'My Title' );
    $self->{'chart_cfg'}->xlabel( '# Users' );
    $self->{'chart_cfg'}->ylabel( '# Downloads' );
    $props->get( 'corr' )->set( 1 );
    $props->get( 'label' )->set( 1 );
    $$buf = '';
    $scat->write_parameters( $self->{'chart_cfg'}, $fh );
    $self->assert_str_equals( $$buf, <<EOF );
#set title = My Title
#set name = Period
#set x = 2
#set xlbl = # Users
#set y = 3
#set ylbl = # Downloads
#set corr = yes
#set id = 1
EOF
}

 sub test_write_chart {
    my $self = $_[0];

    $self->set_up_subreport();
    my $keep = $ENV{'KEEP_CHART'} || 0;
    $self->test_registration_file();
    my $scat = Lire::PluginManager->get_plugin( 'chart_type', 'scat' );

    $self->{'chart_cfg'}->basename( 'scat' );
    $self->{'chart_cfg'}->get( 'case_var' )->set( 'timegroup:time_start' );
    $self->{'chart_cfg'}->type( $scat );

    my $props = $self->{'chart_cfg'}->type_properties();
    $props->get( 'x' )->set( 'user_count' );
    $props->get( 'y' )->set( 'download_count' );
    $props->get( 'corr' )->set( 1 );
    $props->get( 'label' )->set( 1 );

    my $dir = tempdir( $self->name() . '_XXXXXX', CLEANUP => !$keep );
    my $chart = $scat->write_chart( $self->{'chart_cfg'},
                                     $self->{'subreport'},
                                     'outputdir' => $dir, 'format' => 'png' );
    $self->assert( -f $chart ? 1 : 0, "file '$chart' not created" );
    warn( "\nChart written in: $chart\n"  )
      if $keep;
}


1;
