#!/usr/bin/env python

"""
LBAgent is a service to be executed on the nodes of the cluster.
It give a xmlrpc access to hardware specifications and to the current status of the node.
A tread is responsible to refresh status information of all variables
"""

import SimpleXMLRPCServer
from threading import Thread
from lbagentfactory import LBAgentFactory
from lbagent import *

def execCommand(command):
    """
    Execute a shell command
    Returns a string or a table of strings
    """
    import os
    fd = os.popen(command)
    consoleOutput = fd.readlines()
    fd.close()
    for i in range(len(consoleOutput)):
        consoleOutput[i]=consoleOutput[i].replace('\n','')
    if len(consoleOutput) == 1:
        consoleOutput = consoleOutput[0]
    return consoleOutput

class LogFile:
    """
    Log file wrapper
    """
    def __init__(self,file):
        self.file=file
    def close(self):
        self.file.close()
    def flush(self):
        self.file.flush()
    def write(self,str):
        self.file.write(str)
        self.flush()
    def writelines(self,sequence):
        self.file.write(sequence)
        self.flush()
    def isatty(self):
        return False

class XMLRPCServer(SimpleXMLRPCServer.SimpleXMLRPCServer):
    """
    XMLRPCServer
    XMLRPC Server overrided to support custom options
    """
    # Override of TIME_WAIT
    allow_reuse_address = True

class LBAgentContainer:
    """
    LBAgentContainer
    Manage LBAgent instance and configuration
    """
    def __init__(self,configFile):
        self.loading=False
        self._configFile = configFile
        self._loadConfigFile(self._configFile)
    def _loadConfigFile(self,fileName):
        # Create new lbAgent with config
        import os
        print "Config change, reload LBAgent"
        self.loading=True
        try:
            factory = LBAgentFactory(fileName)
            lbAgent = factory.newLBAgent()
            lbAgent.reloadAll()
            # Save last modification time
            self._lastm = os.stat(fileName).st_mtime
            self._lbAgent = lbAgent
        finally:
            self.loading=False
    def _checkConfigChange(self):
        # Check for changes in config file
        import os
        if not(self.loading) and self._lastm != os.stat(self._configFile).st_mtime:
            self._loadConfigFile(self._configFile)
    def getLBAgent(self):
        self._checkConfigChange()
        return self._lbAgent
    def getSpecsStruct(self):
        return self._lbAgent.getSpecsStruct()
    def getStatusStruct(self):
        return self._lbAgent.getStatusStruct()
    def getValuesStruct(self):
        return self._lbAgent.getSpecsStruct() + self._lbAgent.getStatusStruct()

class StatusMonitor(Thread):
    """
    StatusMonitor
    Refresh periodicaly status information of the node
    """
    global container
    def __init__(self):
        self._count = 0
        Thread.__init__(self)
    def reloadVariables(self):
        for var in container.getLBAgent().variables:
            if self._count % var.refresh == 0:
                print "Reload variable : %s at %s"%(var.name,self._count)
                var.reload()
    def run(self):
        import time
        while True:
            try:
                self._count+=1
                self.reloadVariables()
                time.sleep(1)
            except:
                traceback.print_exception(sys.exc_info()[0],sys.exc_info()[1],sys.exc_info()[2])

def main():
    import sys
    import os
    global container

    # Parse parameters
    logFile = None
    configFile = 'lbaconfig.xml'
    for arg in sys.argv[1:]:
        if arg.startswith("--logfile="):
            logFile=arg.split('=')[1]
        elif len(arg)>2:
            configFile=arg
    if not os.path.exists(configFile):
        print "Config file not found. Please specify config file as last argument."
        sys.exit(1)

    # Init logs
    if (logFile != None):
        try:
            logObject =  LogFile(open(logFile,'a',0))
            sys.stderr = logObject
            sys.stdout = logObject
        except:
            pass

    # Create LBAgent Container
    container = LBAgentContainer(configFile)

    # Launch status monitor
    statusMonitor = StatusMonitor()
    statusMonitor.setDaemon(True)
    print "Start StatusMonitor..."
    statusMonitor.start()

    # Init xmlrpc server
    print "Start XMLRPC Service..."
    xmlrpcServer = XMLRPCServer(container.getLBAgent().listen)
    xmlrpcServer.register_function(container.getSpecsStruct,'get_specs')
    xmlrpcServer.register_function(container.getStatusStruct,'get_status')
    xmlrpcServer.register_function(container.getValuesStruct,'get_values')

    # Launch xmlrpc server
    try:
        xmlrpcServer.serve_forever()
    finally:
        xmlrpcServer.server_close()

if __name__ == "__main__":

    import utils

    utils.get_pidfile_lock('/var/run/ltsp-cluster-lbagent.pid')
    if "--daemon" in sys.argv[1:]:
        utils.daemonize()

    main()

