#ifndef __INI_DEFS_H
#define __INI_DEFS_H
/*-------------------------------------------------------------------------
 * Copyright (c) 2000-2002 Kenneth W. Sodemann (stufflehead@bigfoot.com)
 *-------------------------------------------------------------------------
 * ini_defs
 *
 * Synopsis:
 *    Defines used with the INI files
 *
 * $Id: ini_defs.h,v 1.3 2002/05/25 16:44:49 stuffle Exp $
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to 
 * Free Software Foundation, Inc. 
 * 59 Temple Place, Suite 330 
 * Boston, MA  02111-1307  USA
 *
 *-------------------------------------------------------------------------
 */
/*! \file ini_defs.h
 
  \brief INI file handling 

  Functions and defines to make handling the INI file even easier than
  it already is.

  \author Kenneth W. Sodemann <stufflehead@bigfoot.com>
  $Revision: 1.3 $
  $Date: 2002/05/25 16:44:49 $
*/


#define  MIN_NUMBER     0
#define  MAX_NUMBER     20
#define  MIN_SECS       5
#define  MAX_SECS       120
#define  MIN_DEL_SECS   0
#define  MAX_DEL_SECS   15
#define  MIN_ROUNDS     1
#define  MAX_ROUNDS     100
#define  MIN_PCT        0
#define  MAX_PCT        100
#define  MIN_DEV_VALUE  0
#define  MAX_DEV_VALUE  100

#define  OPERATORS      "operators"
#define  ADD_PCT        "addition_pct"
#define  SUB_PCT        "subtraction_pct"
#define  MULT_PCT       "multiplication_pct"
#define  DEF_ADD_PCT    40
#define  DEF_SUB_PCT    30
#define  DEF_MULT_PCT   30
#define  ADD_MAX        "addition_max"
#define  SUB_MAX        "subtraction_max"
#define  MULT_MAX       "multiplication_max"
#define  DEF_ADD_MAX    20
#define  DEF_SUB_MAX    20
#define  DEF_MULT_MAX   12

#define  FONTS          "fonts"
#define  TEXT_FONT      "text_font"
#define  TITLE_FONT     "title_font"
#define  DEF_TEXT_FONT  "-*-courier-*-*-*-*-20-*-*-*-*-*-*-*"
#define  DEF_TITLE_FONT "-*-courier-*-*-*-*-34-*-*-*-*-*-*-*"
#define  FONT_COLOR_R   "color_red"
#define  FONT_COLOR_G   "color_green"
#define  FONT_COLOR_B   "color_blue"
#define  DEF_RED        65535
#define  DEF_GREEN      65535
#define  DEF_BLUE       65535
#define  DEF_ALPHA      65535

#define  COMP_PLAYER    "computer_player"
#define  GUESS_PCT      "guess_pct"
#define  DEF_GUESS_PCT  75
#define  RIGHT_PCT      "correct_pct"
#define  DEF_RIGHT_PCT  50
#define  GUESS_DEV      "guess_deviation"
#define  DEF_GUESS_DEV  10
#define  GUESS_TIME     "time_pct"
#define  DEF_TIME       75
#define  MAX_DEVIATION  "time_deviation"
#define  DEF_MAX_DEV    10

#define  BASIC_PARMS    "game_parameters"
#define  NUM_ROUNDS     "num_rounds"
#define  DEF_NUM_ROUNDS 20
#define  SECONDS        "round_duration"
#define  DEF_SECONDS    10
#define  SECONDS_BTWN   "seconds_btwn_rounds"
#define  DEF_SECS_BTWN  2
#define  RESP_DELAY     "message_delay"
#define  DEF_RESP_DELAY 2

/*! \fn gint ini_get_int (const gchar *ini_file,
                          const gchar *section,
                          const gchar *label,
                          gint         def_val)

  \brief Get the given integer value from the ini file

  \param ini_file
  The name of the INI file

  \param section
  The section within the INI file

  \param label
  The data label within the INI file

  \param def_val
  The default value to use if the given value does not exist

  \return
  The value from the INI file if it exists, def_val if it does not.
*/
gint ini_get_int (const gchar *ini_file,
                  const gchar *section,
                  const gchar *label,
                  gint         def_val);


/*! \fn gint ini_set_int (const gchar *ini_file,
                          const gchar *section,
                          const gchar *label,
                          gint         value)

  \brief Set the given integer value in the ini file

  \param ini_file
  The name of the INI file

  \param section
  The section within the INI file

  \param label
  The data label within the INI file

  \param value
  The value to set the section|label to

  \return
  Nothing
*/
void ini_set_int (const gchar *ini_file,
                  const gchar *section,
                  const gchar *label,
                  gint         value);


/*! \fn gint ini_get_string (const gchar *ini_file,
                             const gchar *section,
                             const gchar *label,
                             const gchar *def_val)

  \brief Get the given string value from the ini file.  The value returned
         should be freed with g_free() when it is no longer needed.

  \param ini_file
  The name of the INI file

  \param section
  The section within the INI file

  \param label
  The data label within the INI file

  \param def_val
  The default value to use if the given value does not exist

  \return
  The value from the INI file if it exists, def_val if it does not.
  The string that is returned must be g_free()'ed.
*/
gchar *ini_get_string (const gchar *ini_file,
                       const gchar *section,
                       const gchar *label,
                       const gchar *def_val);


/*! \fn gint ini_set_string (const gchar *ini_file,
                             const gchar *section,
                             const gchar *label,
                             const gchar *value)

  \brief Set the given string value in the ini file

  \param ini_file
  The name of the INI file

  \param section
  The section within the INI file

  \param label
  The data label within the INI file

  \param value
  The value to set the section|label to

  \return
  Nothing
*/
void ini_set_string (const gchar *ini_file,
                     const gchar *section,
                     const gchar *label,
                     const gchar *value);
#endif
