/*

  xmunipack - FITS view


  Copyright © 2009 - 2014, 2017-9 F.Hroch (hroch@physics.muni.cz)

  This file is part of Munipack.

  Munipack is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  Munipack is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Munipack.  If not, see <http://www.gnu.org/licenses/>.

*/

#include "xmunipack.h"
#include "structtree.h"
#include "tune.h"
#include <wx/wx.h>
#include <wx/toolbar.h>
#include <wx/txtstrm.h>
#include <wx/wfstream.h>
#include <wx/utils.h>
#include <wx/wupdlock.h>
#include <wx/combo.h>
#include <wx/treectrl.h>
#include <wx/progdlg.h>
#include <cfloat>
#include <list>

#define TONE_CURVES "Tone curves"
#define COLOR_PALETTES "Colour palettes"
#define MENU_FRAME "Frame"
#define MENU_HEAD "Head"
#define MENU_GRID "Table"
#define MENU_TUNE "Tune"
#define MENU_TOOLS "Tools"
#define MENU_STRUCT "Structure"


using namespace std;


MuniView::MuniView(wxWindow *w, MuniConfig *c):
  wxFrame(w, wxID_ANY,"untitled", wxDefaultPosition, c->view_size),
  config(c), hdusel(0),
  palinv(config->display_palinv), palauto(config->display_palauto),
  loader(0), zoomctrl(0), place(0), coloring(0),
  console(new wxLogWindow(this,"Logs",false))
  /*shell(0),*/
{
  SetIcon(config->munipack_icon);

  // menus
  menuFile = new wxMenu;
  menuFile->Append(wxID_NEW);

  //  menuFile->Append(ID_NEW_VIEW, _("&New View..."));
  //  menuFile->Append(ID_NEW_BROWSER,_("&New Browser..."));
  menuFile->Append(wxID_OPEN);
  wxMenu *menuFileCreate = new wxMenu;
  menuFileCreate->Append(ID_COLORING,"Colour Image...");
  //  menuFileCreate->Append(wxID_ANY,"Image server ...");
  //  menuFileCreate->Append(wxID_ANY,"Synthetic image...");
  menuFile->AppendSubMenu(menuFileCreate,"Create");
  //  menuFile->AppendSubMenu(ID_VOOPEN,"Virtual Observatory");
  menuFile->AppendSeparator();
  menuFile->Append(wxID_SAVE);
  menuFile->Append(ID_EXPORT,"Export As...");
#ifdef __WXMAC__
  menuFile->Append(wxID_CLOSE);
#endif
  menuFile->AppendSeparator();
  menuFile->Append(wxID_PROPERTIES);
  menuFile->AppendSeparator();
  menuFile->Append(ID_PAGE_SETUP,"Page Setup");
  menuFile->Append(wxID_PREVIEW);
  menuFile->Append(wxID_PRINT);
#ifndef __WXMAC__
  menuFile->AppendSeparator();
  menuFile->Append(wxID_CLOSE);
#endif
#ifdef __WXMAC__
  menuFile->AppendSeparator();
  menuFile->Append(wxID_EXIT);
#endif

  wxMenu *menuEdit =  new wxMenu;
  menuEdit->Append(wxID_UNDO);
  menuEdit->Append(wxID_CUT);
  menuEdit->Append(wxID_COPY);
  menuEdit->Append(wxID_PASTE);
#ifndef __WXMAC__
  menuEdit->AppendSeparator();
  menuEdit->Append(wxID_PREFERENCES);
#endif


  menuView = new wxMenu;
  menuView->Append(ID_FULLSCREEN,"&Fullscreen",
		   "Enable fullscreen image display");
  menuView->AppendSeparator();
  menuView->AppendCheckItem(ID_TOOLBAR,"&Show Tool Bar",
    "Change visibility of toolbar (shorthand buttons with icons on top)");
  menuView->Check(ID_TOOLBAR,config->view_tbar);

  wxMenu *menuHelp = new wxMenu;
  menuHelp->Append(wxID_HELP,wxEmptyString,"A web connection to " HOMEPAGE);
  //  menuHelp->AppendCheckItem(ID_LOG,"&Log");
  menuHelp->Append(ID_LOG,"Log");
  /*
  menuHelp->Append(ID_BUG,"Report a Bug ...",
		   "Launch your default browser to Munipack's issues.");
  */
  menuHelp->Append(wxID_ABOUT);

  menuWin = new wxMenu;

  wxMenuBar *menuBar = new wxMenuBar;
  menuBar->Append(menuFile,"&File");
  menuBar->Append(menuEdit,"&Edit");
  menuBar->Append(menuView,"&View");
  menuBar->Append(menuWin,MENU_STRUCT);
  menuBar->Append(menuHelp,"&Help");
  SetMenuBar(menuBar);


  // toolbars
  MuniArtIcons ico(wxART_TOOLBAR,wxSize(22,22));

  wxToolBar *tbar = CreateToolBar(wxTB_HORIZONTAL|wxTB_TEXT);
  tbar->AddTool(ID_INFO,"Header",ico.Icon(wxART_INFORMATION),
		"Show FITS header");
  tbar->AddSeparator();
  tbar->AddStretchableSpace();

  structtree = new wxComboCtrl(tbar,wxID_ANY,wxEmptyString,wxDefaultPosition,
			       wxDefaultSize,wxCB_READONLY|wxCC_STD_BUTTON);
  MuniStructtree* popupCtrl = new MuniStructtree;
  structtree->SetPopupControl(popupCtrl);

  tbar->AddControl(structtree);

  tbar->EnableTool(ID_INFO,false);

  tbar->Realize();
  tbar->Show(config->view_tbar);
  SetToolBar(tbar);


  // workplace
  place = new MuniSplashing(this,config);

  wxBoxSizer *topsizer = new wxBoxSizer(wxVERTICAL);
  topsizer->Add(place,wxSizerFlags(1).Expand());
  SetSizer(topsizer);

  //  menuFile->Enable(wxID_SAVE,false);
  //  menuFile->Enable(ID_EXPORT,false);
  menuFile->Enable(ID_PAGE_SETUP,false);
  menuFile->Enable(wxID_PREVIEW,false);
  menuFile->Enable(wxID_PRINT,false);

  menuEdit->Enable(wxID_UNDO,false);
  menuEdit->Enable(wxID_CUT,false);
  menuEdit->Enable(wxID_COPY,false);
  menuEdit->Enable(wxID_PASTE,false);
  menuEdit->Enable(wxID_PREFERENCES,false);

  menuView->Enable(ID_FULLSCREEN,false);
  menuFile->Enable(wxID_PROPERTIES,false);

  Bind(wxEVT_CLOSE_WINDOW,&MuniView::OnClose,this);
  //  Bind(wxEVT_SIZE,&MuniView::OnSize,this);
  //  Bind(wxEVT_ENTER_WINDOW,&MuniView::OnEnterWin,this);
  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::FileClose,this,wxID_CLOSE);
  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::FileClose,this,wxID_EXIT);
  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::NewView,this,wxID_NEW);
  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::FileOpen,this,wxID_OPEN);
  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::FileSave,this,wxID_SAVE);
  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::FileExport,this,ID_EXPORT);
  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::FileProperties,this,wxID_PROPERTIES);
  //	  wxCommandEventHandler(MuniView::OnPreferences));
  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::ShowToolbar,this,ID_TOOLBAR);
  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnFullScreen,this,ID_FULLSCREEN);
  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::Coloring,this,ID_COLORING);
  //  Bind(EVT_SHELL,&MuniView::OnShell,this);
  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::HelpGuide,this,wxID_HELP);
  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::ViewLog,this,ID_LOG);
  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::HelpAbout,this,wxID_ABOUT);
  //  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::HelpBug,this,ID_BUG);

  Bind(EVT_HDU,&MuniView::OnResumeHdu,this);

}

void MuniView::OnClose(wxCloseEvent& event)
{
  if( event.CanVeto() && IsModified() ) {
    if( Unsaved(fits.GetName()) == wxID_CANCEL ) {
      event.Veto();
      return;
    }
  }


  // if( event.CanVeto() && /*fits.IsModified()*/ IsModified() ) {
  //   wxMessageDialog dialog(this,"Do you really want to leave this window ?",
  // 			   "Confirm Close",wxICON_EXCLAMATION | wxYES_NO | wxNO_DEFAULT);
  //   dialog.SetExtendedMessage("Your changes in \""+fits.GetName()+
  // 			      "\" will be lost if you will continue in closing now.");
  //   if ( dialog.ShowModal() != wxID_YES ) {
  //     event.Veto();
  //     return;
  //   }
  // }

  // probbaly crashes during loading (see close functions in astrometry)
  StopLoading();

  MuniDisplay *display = dynamic_cast<MuniDisplay *>(place);
  if( display )
    display->StopRendering();


  // if( ! backup.IsEmpty() ) {
  //   wxBusyCursor wait;
  //   for(size_t i = 0; i < backup.GetCount(); i++)
  //     wxRemoveFile(backup[i]);
  // }

  if( GetParent() )
    wxQueueEvent(GetParent(),event.Clone());

  config->display_palinv = palinv;
  config->display_palauto = palauto;
  config->view_tbar = GetToolBar()->IsShown();
  config->view_size = GetSize();


  // see above
  Destroy();
}

bool MuniView::IsModified() const
{
  return ! backup.IsEmpty();
}

void MuniView::RemoveBackup()
{
  for(size_t i = 0; i < backup.GetCount(); i++) {
    if( wxFileExists(backup[i]) )
      wxRemoveFile(backup[i]);
  }
  backup.Clear();
}

void MuniView::ViewLog(wxCommandEvent& event)
{
  wxASSERT(console);
  console->Show();
}


void MuniView::OnUpdateInfos(wxUpdateUIEvent& event)
{
  event.Check(GetToolBar()->GetToolState(ID_INFO));
}

void MuniView::StopLoading()
{
  //  wxLogDebug("MuniView::StopLoading()");
  {
    wxCriticalSectionLocker enter(loaderCS);
    if( loader )
      loader->Delete();
  }

  while(true) {
    {
       wxCriticalSectionLocker enter(loaderCS);
       if( ! loader ) break;
    }
    ::wxMilliSleep(1);
  }
}

void MuniView::LoadMeta(const FitsMeta& m)
{
  wxLogDebug("Loading meta .. "+m.GetFullPath());
  SetMeta(m);

  LoadFile(m.GetFullPath());
}

void MuniView::LoadFile(const wxString& filename)
{
  //  wxASSERT(backup.IsEmpty());
  //  RemoveBackup();

  if( IsModified() ) {
    if( Unsaved(fits.GetName()) == wxID_CANCEL )
      return;
  }

  //  backup.Clear();
  meta = FitsMeta();
  LoadStart(filename);
}

void MuniView::LoadFileBackup(const wxString& file, const wxString& b)
{
  wxString a;
  a.Printf(b+"_%d",(int)backup.GetCount());
  backup.Add(a);

  meta = FitsMeta();
  LoadStart(file);

  //  LoadFile(file);
  //  LoadStart(fits.GetFullPath());
}

int MuniView::Unsaved(const wxString& filename)
{
  wxMessageDialog dialog(this,"Do you want to save changes you made in \""+
			 filename+"\" ?",filename,wxICON_EXCLAMATION |
			 wxYES_NO | wxCANCEL | wxNO_DEFAULT);
  dialog.SetExtendedMessage("Your changes will be lost if you don't save them.");
  dialog.SetYesNoLabels("Save","Don't Save");

  int code = dialog.ShowModal();

  if( code == wxID_YES ) {
    wxCommandEvent e;
    FileSave(e);
  }
  else if( code == wxID_NO ) {
    if( IsModified() ) {
      wxBusyCursor wait;
      RemoveBackup();
    }
  }
  else if( code == wxID_CANCEL )
    ;

  return code;
}



void MuniView::LoadStart(const wxString& filename)
{
  StopLoading();

  Bind(EVT_FITS_OPEN,&MuniView::OnLoadFinish,this,ID_LOADER);

  loader = new FitsOpen(this,filename,config->icon_size);
  wxASSERT(loader);
  wxThreadError code = loader->Create();
  wxASSERT(code == wxTHREAD_NO_ERROR);
  loader->Run();

  if( dynamic_cast<MuniSplashing *>(place) )
    dynamic_cast<MuniSplashing *>(place)->Play();

}

void MuniView::OnLoadFinish(FitsOpenEvent& event)
{
  //  wxLogDebug("MuniView::OnLoadFinish");

  // wait for thread finish
  while(true) {
    {
      wxCriticalSectionLocker enter(loaderCS);
      if( ! loader ) break;
    }
    ::wxMilliSleep(1);
  }

  if( dynamic_cast<MuniSplashing *>(place) )
    dynamic_cast<MuniSplashing *>(place)->Stop();

  fits = event.fits;
  vector<wxImage> icons(event.icons);

  if( fits.Status() ) {

    if( ! meta.IsOk() ) {
      MuniIcon micon(fits,config->display_colorspace,config->cdatafile,
		     config->default_icon,config->table_icon,config->head_icon,
		     config->icon_size,icons);
      meta = FitsMeta(fits,micon.GetIcon(),micon.GetList());
      SetMeta(meta);
    }
    wxASSERT(meta.IsOk() && fits.HduCount() > 0);

    hdusel = EstimHduSel(meta);
    HduSelect(hdusel);
    MenuSelect(hdusel);
    TreeSelect(hdusel);

    wxString title = fits.GetName() + (IsModified() ? "*" : "");
    SetTitle(title);

  }
  else {
    wxArrayString es = fits.GetErrorMessage();
    for(size_t i = 0; i < es.size(); i++)
      wxLogError(es[i]);
    wxLogError("Loading failed on `"+fits.GetErrorDescription()+"'");
  }

  Unbind(EVT_FITS_OPEN,&MuniView::OnLoadFinish,this,ID_LOADER);
}

void MuniView::ExportStart(const wxString& savename)
{
  loader = new FitsExport(this,fits.Hdu(hdusel),savename,
			  dynamic_cast<MuniDisplay *>(place)->GetTone(),
			  dynamic_cast<MuniDisplay *>(place)->GetItt(),
			  dynamic_cast<MuniDisplay *>(place)->GetPalette(),
			  dynamic_cast<MuniDisplay *>(place)->GetColor());
  wxASSERT(loader);
  wxThreadError code = loader->Create();
  wxASSERT(code == wxTHREAD_NO_ERROR);
  loader->Run();

}


void MuniView::Clear()
{
  meta.Clear();
  fits.Clear();

  // disable menus ?
  //  hists.clear();
  viewid.clear();
}


void MuniView::SetMeta(const FitsMeta& m)
{
  wxASSERT(m.IsOk() && menuWin);

  meta = m;

  menuFile->Enable(wxID_PROPERTIES,true);

  menuWin = new wxMenu;

  wxMenuBar *menuBar = GetMenuBar();
  wxMenu *oldmenu = menuBar->Replace(menuBar->FindMenu(MENU_STRUCT),
				     menuWin,MENU_STRUCT);
  wxASSERT(oldmenu);

  while( oldmenu->GetMenuItemCount() > 0 ) {
    int i = oldmenu->GetMenuItemCount() - 1;
    wxMenuItem *item = oldmenu->FindItemByPosition(i);
    wxASSERT(item);
    int id = item->GetId();
    Unbind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnMenuStructure,this,id);
    oldmenu->Destroy(id);
  }
  delete oldmenu;
  viewid.clear();


  menuWin->AppendRadioItem(wxID_HOME,"Resume\tAlt+Home");
  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnMenuStructure,this,wxID_HOME);

  // create HDU menu
  for(size_t k = 0; k < meta.HduCount(); k++ ) {

    wxString label = meta.Hdu(k).GetControlLabel();

    if( k < 9 ) {
      wxString l;
      l.Printf("\tAlt+%d",(int) k+1);
      label += l;
    }
    wxMenuItem* item = menuWin->AppendRadioItem(wxID_ANY,label);
    wxASSERT(item);
    int id = item->GetId();
    Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnMenuStructure,this,id);
    viewid.push_back(id);
  }


  menuWin->AppendSeparator();

  menuWin->Append(wxID_BACKWARD,"Previous\tAlt+Left");
  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnMenuStructure,this,wxID_BACKWARD);

  menuWin->Append(wxID_FORWARD,"Next\tAlt+Right");
  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnMenuStructure,this,wxID_FORWARD);

  // set Tree structure
  MuniStructtree *tree = static_cast<MuniStructtree *>(structtree->GetPopupControl());
  wxASSERT(tree);
  tree->SetMeta(meta);

}

int MuniView::EstimHduSel(const FitsMeta& meta)
{
  wxASSERT(meta.HduCount() > 0);

  // we are directly selecting image at these cases:
  // * there is an image and MUNIPACK extension
  // * the first primary extension is an image

  if( meta.Hdu(0).Type() == HDU_IMAGE )
    return 0;
  else
    return -1;
}


// void MuniView::EstimHistos(const FitsFile& fits)
// {
//   double xmin,xmax;
//   bool color = fits.Type() == FITS_COLOR;


//   if( color ) {

//     /*
//     FitsImage fimage(fits,-1);
//     FitsImage fs = fimage.Scale(0.1);
//     FitsArray X = fs.Item(0);
//     FitsArray Y = fs.Item(1);
//     FitsArray Z = fs.Item(2);
//     int npix = X.Width()*X.Height();
//     FitsColor color;
//     for(int i = 0; i < npix; i++) {
//       float L,u,v;
//       color.XYZ_Luv(X.PIxel(i),Y.Pixel(i),Z.Pixel(i),1.0,L,u,v);
//     }
//     */

//     FitsArrayStat array(fits.Hdu(2));
//     /*
//     xmin = max(array.Med() - 2.0*array.Mad(),0.0);
//     xmax = array.Med() + 3.0*array.Mad();
//     xmin = 0.0;
//     xmax = 7.0*500.0;
//     */
//     xmin = array.Med() - 10.0*array.Mad();
//     xmax = array.Med() + 20.0*array.Mad();
//     // convrt to XYZ, Luv
//     //    hits = FitsHisto(
//     hist = FitsHisto(fits.Hdu(2),xmin,xmax);
//   }
//   else if( fits.Type() == FITS_GRAY ) {
//     FitsArrayStat array(fits.Hdu(0));
//     xmin = array.Med() - 5.0*array.Mad();
//     xmax = array.Med() + 7.0*array.Mad();
//     hist = FitsHisto(fits.Hdu(0),xmin,xmax);
//   }
//   else
//     hist = FitsHisto();

//   hists.clear();
//   for(size_t i = 0; i < fits.HduCount(); i++) {
//     if( fits.Hdu(i).Type() == HDU_IMAGE ) {
//       //      if( ! color ) {
// 	FitsArrayStat array(fits.Hdu(i));
// 	xmin = array.Med() - 5.0*array.Mad();
// 	xmax = array.Med() + 7.0*array.Mad();
// 	//      }
//       hists.push_back(FitsHisto(fits.Hdu(i),xmin,xmax));
//     }
//     else {
//       hists.push_back(FitsHisto());
//     }
//   }
// }


void MuniView::ShowToolbar(wxCommandEvent& event)
{
  GetToolBar()->Show(event.IsChecked());
  Layout();
}

void MuniView::FileClose(wxCommandEvent& WXUNUSED(event))
{
  Close();
}

void MuniView::FileOpen(wxCommandEvent& WXUNUSED(event))
{
  if( IsModified() ) {
    if( Unsaved(fits.GetName()) == wxID_CANCEL )
      return;
  }

  wxFileDialog select(this,"Choose a file",wxEmptyString,wxEmptyString,
		      "FITS files "+config->dirmask+")|"+
		      config->dirmask+"| All files (*)|*",
		      wxFD_FILE_MUST_EXIST|wxFD_CHANGE_DIR);
  if (select.ShowModal() == wxID_OK ) {
    meta = FitsMeta();
    LoadFile(select.GetPath());
  }
}

void MuniView::SaveFile(const wxString& savename)
{
  bool s = fits.Save("!"+savename);
  if( s ) {
    SetTitle(fits.GetName());
    RemoveBackup();
  }

}

void MuniView::FileSave(wxCommandEvent& WXUNUSED(event))
{
  wxFileDialog select(this,"Choose a file",fits.GetPath(),fits.GetFullName(),
		      "FITS files "+config->dirmask+")|"+
		      config->dirmask+"| All files (*)|*",
		      wxFD_SAVE|wxFD_OVERWRITE_PROMPT|wxFD_CHANGE_DIR);

  if (select.ShowModal() == wxID_OK ) {
    wxBusyCursor wait;
    SaveFile(select.GetPath());
  }


  // if( false /*HduHead()*/ ) {

  //   wxFileDialog select(this,"Choose text file",wxEmptyString,wxEmptyString,
  // 			"Text files (*.txt)|*.txt| All files (*)|*",
  // 			wxFD_SAVE|wxFD_OVERWRITE_PROMPT|wxFD_CHANGE_DIR);

  //   //    if (select.ShowModal() == wxID_OK )
  //     //      SaveAsText(select.GetPath());
  // }
  // else
  //   wxLogDebug("Not implemented yet.");
}

void MuniView::FileExport(wxCommandEvent& WXUNUSED(event))
{
  // exporting images
  if( dynamic_cast<MuniDisplay *>(place) ) {

    wxFileDialog select(this,"Export As",wxEmptyString,fits.GetName(),
			"PNG files (*.png)|*.png|JPEG files (*.jpg)|*.jpg|TIFF files (*.tif)|*.tif|PNM files (*.pnm)|*.pnm",
			wxFD_SAVE|wxFD_OVERWRITE_PROMPT|wxFD_CHANGE_DIR);

    if (select.ShowModal() == wxID_CANCEL )
      return;

    // save thread
    ExportStart(select.GetPath());

    wxProgressDialog dialog("Export of an image","Exporting "+
			    select.GetFilename()+" ... ",100,this,
			    wxPD_APP_MODAL|wxPD_AUTO_HIDE);
    while(true) {
      dialog.Pulse();
      ::wxMilliSleep(50);
      wxCriticalSectionLocker enter(loaderCS);
      if( ! loader ) break;
    }
  }

  // exporting tables
  else if( dynamic_cast<MuniGrid *>(place) ) {

    wxFileDialog select(this,"Export As",wxEmptyString,fits.GetName(),
			"TEXT files (*.txt)|*.txt",
			wxFD_SAVE|wxFD_OVERWRITE_PROMPT|wxFD_CHANGE_DIR);

    if (select.ShowModal() == wxID_OK ) {

      /*
      wxProgressDialog dialog(_("Export of Table"),_("Exporting ")+
			      select.GetFilename()+_(" ... "),100,this,
			      wxPD_APP_MODAL|wxPD_AUTO_HIDE);
      dialog.Pulse();
      */

      wxBusyCursor wait;

      wxFileOutputStream output(select.GetPath());
      wxTextOutputStream cout(output);

      const FitsTable table(fits.Hdu(/*HduSel()*/hdusel));
      cout << "#";
      for(int i = 0; i < table.Width(); i++) {
	wxString key;
	key.Printf("TTYPE%d",(int) i+1);
	cout << " " << table.GetKey(key);
      }
      cout << endl;

      for(int j = 0; j < table.Height(); j++) {
	cout << j;
	for(int i = 0; i < table.Width(); i++)
	  cout << " " << table.Pixel_str(i,j) ;
	cout << endl;
	//	if( j % 100 == 0) dialog.Pulse();
      }
    }
  }

  // exporting headers
  else if( dynamic_cast<MuniHead *>(place) ) {

    wxFileDialog select(this,"Export As",wxEmptyString,fits.GetName(),
			"TEXT files (*.txt)|*.txt",
			wxFD_SAVE|wxFD_OVERWRITE_PROMPT|wxFD_CHANGE_DIR);

    if (select.ShowModal() == wxID_OK ) {

      /*
      wxProgressDialog dialog(_("Export of Header"),_("Exporting ")+
			      select.GetFilename()+_(" ... "),100,this,
			      wxPD_APP_MODAL|wxPD_AUTO_HIDE);
      dialog.Pulse();
      */

      wxBusyCursor wait;

      wxFileOutputStream output(select.GetPath());
      wxTextOutputStream cout(output);

      const FitsHdu head = fits.Hdu(hdusel);
      for(size_t i = 0; i < head.GetCount(); i++)
	cout << head.Item(i) << endl;
    }
  }

  else
    wxFAIL_MSG("----- WARNING: Unreachable code.");

}

/*
void MuniView::SaveAsText(wxString filename)
{
  wxFileOutputStream output(filename);
  wxTextOutputStream cout(output);

  const FitsHdu head = fits.Hdu(HduSel());
  for(size_t i = 0; i < head.GetCount(); i++)
    cout << head.Item(i) << endl;
}
*/

void MuniView::FileProperties(wxCommandEvent& WXUNUSED(event))
{
  MuniFileProperties *w = new MuniFileProperties(this,meta,config);
  w->Show();
}

void MuniView::OnPreferences(wxCommandEvent& WXUNUSED(event))
{
  MuniPreferences *w = new MuniPreferences(this,config);
  w->Show();
}

void MuniView::OnSelectFrame(wxCommandEvent& event)
{
  if( GetParent() )
    wxQueueEvent(GetParent(),event.Clone());
}

void MuniView::OnCycleZoom(wxCommandEvent& event)
{
  wxASSERT(dynamic_cast<MuniDisplay *>(place));
  wxASSERT(zoomctrl);

  FitsZoom zoom = dynamic_cast<MuniDisplay *>(place)->GetZoom();

  MuniTuneEvent ev(EVT_TUNE,ID_ZOOM_SCALE);

  if( event.GetId() == wxID_ZOOM_FIT ) {
    ev.shrink = true;
    ev.n = dynamic_cast<MuniDisplay *>(place)->GetFitShrink();
  }
  else if ( event.GetId() == wxID_ZOOM_100 ) {
    ev.shrink = true;
    ev.n = 1;
  }
  else {
    //wxLogDebug("MuniView::OnCycleZoom %d %d %d",zoom.GetShrink(),
    //	       zoom.GetZoom(),event.GetId()==wxID_ZOOM_IN);
    FitsZoom z;
    switch(event.GetId()) {
    case wxID_ZOOM_IN:  z = zoom.GetIn(); break;
    case wxID_ZOOM_OUT: z = zoom.GetOut(); break;
    }


    ev.shrink = z.GetKind() == ZOOM_SHRINK;
    if(  ev.shrink )
      ev.n = z.GetShrink();
    else
      ev.n = z.GetZoom();
  }
  wxQueueEvent(place,ev.Clone());

  wxASSERT(zoomctrl);
  wxString a;
  if( ev.shrink )
    a.Printf("1:%d",ev.n);
  else
    a.Printf("%d:1",ev.n);
  zoomctrl->SetValue(a);
}

void MuniView::OnMenuZoom(wxCommandEvent& event)
{
  wxASSERT(place && zoomctrl);

  MuniTuneEvent ev(EVT_TUNE,ID_ZOOM_SCALE);
  ev.shrink = true;

  if( event.GetId() == wxID_ZOOM_100 )
    ev.n = 1;
  else if( event.GetId() == wxID_ZOOM_FIT )
    ev.n = dynamic_cast<MuniDisplay *>(place)->GetFitShrink();

  wxQueueEvent(place,ev.Clone());

  wxString a;
  if( ev.shrink )
    a.Printf("1:%d",ev.n);
  else
    a.Printf("%d:1",ev.n);
  zoomctrl->SetValue(a);
}


void MuniView::OnToolZoom(wxCommandEvent& event)
{
  wxASSERT(place);

  MuniTuneEvent ev(EVT_TUNE,ID_ZOOM_SCALE);

  if( event.GetString() == "4:1" ) {
    ev.shrink = false;
    ev.n = 4;
  }
  else if( event.GetString() == "2:1" ) {
    ev.n = 2;
    ev.shrink = false;
  }
  else if( event.GetString() == "1:1" ) {
    ev.n = 1;
    ev.shrink = true;
  }
  else if( event.GetString() == "1:2" ) {
    ev.n = 2;
    ev.shrink = true;
  }
  else if( event.GetString() == "1:3" ) {
    ev.n = 3;
    ev.shrink = true;
  }
  else if( event.GetString() == "1:4" ) {
    ev.n = 4;
    ev.shrink = true;
  }
  else if( event.GetString() == "1:6" ) {
    ev.n = 6;
    ev.shrink = true;
  }
  else if( event.GetString() == "1:8" ) {
    ev.n = 8;
    ev.shrink = true;
  }
  else
    wxFAIL_MSG("MuniView::OnToolZoom: UNKNOWN ZOOM");

  //  wxLogDebug("MuniView::OnToolZoom: "+event.GetString());
  wxQueueEvent(place,ev.Clone());
}

void MuniView::OnMenuPal(wxCommandEvent& event)
{
  wxMenu *menuTune = GetMenuBar()->GetMenu(GetMenuBar()->FindMenu(MENU_TUNE));
  wxMenuItem *item = menuTune->FindItem(menuTune->FindItem(COLOR_PALETTES));
  wxMenu *menuPalette= item->GetSubMenu();
  wxASSERT(place && menuTune && menuPalette &&
	   menuPalette->FindItem(event.GetId()));

  MuniTuneEvent ev(EVT_TUNE,ID_PALETTE_TYPE);
  wxMenuItem *sitem = menuPalette->FindItem(event.GetId());
  ev.SetString(sitem->GetItemLabelText());
  wxQueueEvent(place,ev.Clone());
}

void MuniView::OnCyclePal(wxCommandEvent& event)
{
  wxMenu *menuTune = GetMenuBar()->GetMenu(GetMenuBar()->FindMenu(MENU_TUNE));
  wxMenuItem *item = menuTune->FindItem(menuTune->FindItem(COLOR_PALETTES));
  wxMenu *menuPalette= item->GetSubMenu();
  wxASSERT(place && menuTune && menuPalette);

  size_t nmenu = menuPalette->GetMenuItemCount();
  wxString label;
  int itemid = -1;
  for(size_t i = 0; i < nmenu; i++) {
    wxMenuItem *item = menuPalette->FindItemByPosition(i);
    if( item && item->IsChecked() ) {
      wxMenuItem *it = menuPalette->FindItemByPosition(i+1 < nmenu ? i+1 : 0);
      if( it ) {
	label = it->GetItemLabelText();
	itemid = it->GetId();
      }
    }
  }
  wxASSERT(!label.IsEmpty() );

  menuPalette->Check(itemid,true);

  MuniTuneEvent ev(EVT_TUNE,ID_PALETTE_TYPE);
  ev.SetString(label);
  wxQueueEvent(place,ev.Clone());
}

void MuniView::OnMenuInverse(wxCommandEvent& event)
{
  wxASSERT(place);

  palinv = event.IsChecked();

  MuniTuneEvent ev(EVT_TUNE,ID_PALETTE_INVERSE);
  ev.SetInt(event.IsChecked());
  wxQueueEvent(place,ev.Clone());
}

void MuniView::OnMenuPalauto(wxCommandEvent& event)
{
  wxASSERT(place);
  palauto = event.IsChecked();
  wxQueueEvent(place,event.Clone());
}


void MuniView::OnTuneFine(MuniTuneEvent& event)
{
  wxMenu *menuTune = GetMenuBar()->GetMenu(GetMenuBar()->FindMenu(MENU_TUNE));
  wxMenuItem *item = menuTune->FindItem(menuTune->FindItem(COLOR_PALETTES));
  wxMenu *menuPalette = 0;
  if( item )
    menuPalette = item->GetSubMenu();

  item = menuTune->FindItem(menuTune->FindItem(TONE_CURVES));

  int itemid;

  switch(event.GetId()) {
  case ID_PALETTE_TYPE:
    if( menuPalette ) {
      itemid = menuPalette->FindItem(event.GetString());
      wxASSERT(itemid != wxNOT_FOUND);
      menuPalette->Check(itemid,true);
    }
    break;
  case ID_PALETTE_INVERSE:
    if( menuTune ) {
      palinv = event.GetInt();
      menuTune->Check(ID_MENU_INVERSE,event.GetInt());
    }
    break;
  case ID_ZOOM:
    if( zoomctrl ) {
     wxString a;
     if( event.shrink )
       a.Printf("1:%d",event.n);
     else
       a.Printf("%d:1",event.n);
     zoomctrl->SetValue(a);
    }
    break;
  } // case

  wxQueueEvent(place,event.Clone());
}


// void MuniView::NewBrowser(wxCommandEvent& WXUNUSED(event))
// {
//   MuniBrowser *b = new MuniBrowser(config);
//   b->Show();
// }

void MuniView::NewView(wxCommandEvent& WXUNUSED(event))
{
  MuniView *w = new MuniView(this,config);
  w->Show();
}

void MuniView::OnFullScreen(wxCommandEvent& WXUNUSED(event))
{
  MuniDisplay *display = dynamic_cast<MuniDisplay*>(place);
  if( ! display ) return;

  if( IsFullScreen() ) {
    GetToolBar()->Show(config->view_tbar);
    if( display )
      display->ShowPanel(true);
    ShowFullScreen(false);
  }
  else {
    GetToolBar()->Show(false);
    if( display )
      display->ShowPanel(false);
    ShowFullScreen(true);
  }
}

void MuniView::OnTune(wxCommandEvent& event)
{
  wxASSERT(dynamic_cast<MuniDisplay *>(place));
  dynamic_cast<MuniDisplay *>(place)->ShowTune(event.IsChecked());
}

void MuniView::OnHeader(wxCommandEvent& event)
{
  MuniHeader *header = new MuniHeader(this,config);
  header->SetHdu(fits.Hdu(hdusel));
  header->Show();
}

void MuniView::OnAstrometry(wxCommandEvent& event)
{
  FitsTable table;

  for(size_t i = 0; i < fits.HduCount(); i++) {
    const FitsHdu h(fits.Hdu(i));

    if( h.IsOk() && h.GetExtname().Find(APEREXTNAME) != wxNOT_FOUND ) {
      table = FitsTable(fits.Hdu(i));
      break;
    }
  }

  wxASSERT(dynamic_cast<MuniDisplay *>(place));
  dynamic_cast<MuniDisplay *>(place)->Astrometry(fits.GetFullPath(),table);
}

void MuniView::OnPhotometry(wxCommandEvent& event)
{
  wxASSERT(dynamic_cast<MuniDisplay *>(place));
  dynamic_cast<MuniDisplay *>(place)->Photometry(fits.GetFullPath());
}

void MuniView::OnCalibrate(wxCommandEvent& event)
{
  wxASSERT(dynamic_cast<MuniDisplay *>(place));
  dynamic_cast<MuniDisplay *>(place)->Calibrate(fits.GetFullPath());
}

void MuniView::OnShowGrid(wxCommandEvent& event)
{
  // check presence of WCS?

  wxASSERT(dynamic_cast<MuniDisplay *>(place));
  dynamic_cast<MuniDisplay *>(place)->ShowGrid(event.IsChecked());
  config->display_grid = event.IsChecked();
}

void MuniView::OnShowSources(wxCommandEvent& event)
{
  wxASSERT(dynamic_cast<MuniDisplay *>(place));
  dynamic_cast<MuniDisplay *>(place)->ShowSources(event.IsChecked(),fits);
  config->display_sources = event.IsChecked();
}

void MuniView::HelpGuide(wxCommandEvent& WXUNUSED(event))
{
  MuniHelp *h = new MuniHelp(config);
  h->Show();
}

void MuniView::HelpAbout(wxCommandEvent& WXUNUSED(event))
{
  MuniAbout(config->munipack_icon);
}

/*
void MuniView::HelpBug(wxCommandEvent& WXUNUSED(event))
{
  wxLaunchDefaultBrowser(BUGPAGE);
}
*/

void MuniView::OnMenuStructure(wxCommandEvent& event)
{
  int hdu = -2;

  if( event.GetId() == wxID_HOME )
    hdu = -1;

  else if( event.GetId() == wxID_BACKWARD )
    hdu = hdusel - 1;

  else if( event.GetId() == wxID_FORWARD )
    hdu = hdusel + 1;

  for(size_t i = 0; i < viewid.size(); i++)
    if( viewid[i] == event.GetId() )
      hdu = i;

  if( hdu == hdusel || hdu < -1 || hdu >= static_cast<int>(meta.HduCount()) )
    return;

  hdusel = hdu;

  HduSelect(hdusel);
  MenuSelect(hdusel);
  TreeSelect(hdusel);

}

void MuniView::OnResumeHdu(MuniHduEvent& event)
{
  wxLogDebug("HDU %d ",event.hdu);

  if( hdusel == event.hdu ) return;

  hdusel = event.hdu;

  MenuSelect(event.hdu);
  TreeSelect(event.hdu);
  HduSelect(event.hdu);
}

// void MuniView::OnPreviousHdu(MuniHduEvent& event)
// {
//   wxLogDebug("HDU %d ",event.hdu);

//   int hdu = event.hdu - 1;

//   if( hdu >= 0 ) {

//   //  if( hdusel == event.hdu ) return;

//   hdusel = hdu;

//   MenuSelect(hdu);
//   TreeSelect(hdu);
//   HduSelect(hdu);
//   }
// }

// void MuniView::OnNextHdu(MuniHduEvent& event)
// {
//   wxLogDebug("HDU %d ",event.hdu);

//   int hdu = event.hdu + 1;

//   if( hdu < meta.HduCount() ) {

//   //  if( hdusel == event.hdu ) return;

//   hdusel = hdu;

//   MenuSelect(hdu);
//   TreeSelect(hdu);
//   HduSelect(hdu);
//   }
// }

void MuniView::MenuSelect(int hdu)
{
  // set menu selector
  wxASSERT(menuWin);
  if( hdu >= 0 )
    menuWin->Check(viewid[hdu],true);
  else
    menuWin->Check(wxID_HOME,true);
}

void MuniView::TreeSelect(int hdu)
{
  MuniStructtree *tree = static_cast<MuniStructtree *>(structtree->GetPopupControl());
  wxASSERT(tree);
  tree->SetHdu(hdu);
  wxTreeItemId item = tree->GetSelection();
  structtree->SetValue(tree->GetItemText(item));
  structtree->Refresh();
}

void MuniView::HduSelect(int h)
{
  if( typeid(*place)==typeid(MuniSplashing) )
    DeleteSplashing();
  else if( typeid(*place)==typeid(MuniHead) )
    DeleteHead();
  else if( typeid(*place)==typeid(MuniResume) )
    DeleteResume();
  else if( typeid(*place)==typeid(MuniDisplay) )
    DeleteDisplay();
  else if( typeid(*place)==typeid(MuniGrid) )
    DeleteGrid();
  else
    wxFAIL_MSG("----- WARNING: Unreachable code.");

  hdusel = h;
  wxLogDebug("MuniView::HduSelect %d",hdusel);


  bool backward = false;
  bool forward = false;

  if( hdusel >= 0 ) {

    const FitsMetaHdu hdu = meta.Hdu(hdusel);

    switch(fits.Hdu(hdusel).Type()) {
    case HDU_IMAGE:  CreateDisplay(); break;
    case HDU_TABLE:  CreateGrid();    break;
    case HDU_HEAD:   CreateHead();    break;
    }

    if( hdusel == 0 ) {
      backward = false;
      forward = true;
    }
    else if( hdusel == static_cast<int>(meta.HduCount() - 1) ) {
      backward = true;
      forward = false;
    }
    else {
      backward = true;
      forward = true;
    }
  }
  else
    CreateResume();


  // set previous/next toolbar buttons
  wxToolBar *tbar = GetToolBar();
  wxASSERT(tbar);
  int idm = GetMenuBar()->FindMenu(MENU_STRUCT);
  wxASSERT(idm != wxNOT_FOUND);
  wxMenu *menuWin = GetMenuBar()->GetMenu(idm);
  wxASSERT(menuWin);
  menuWin->Enable(wxID_BACKWARD,backward);
  menuWin->Enable(wxID_FORWARD,forward);

}

void MuniView::ReplacePlace(wxWindow *newin)
{
  wxWindow *p = place;
  place = newin;
  wxSizer *topsizer = GetSizer();
  topsizer->Replace(p,place);
  p->Destroy();
  Layout();
}

void MuniView::CreateSplashing()
{
  ReplacePlace(new MuniSplashing(this,config));
}

void MuniView::DeleteSplashing()
{
}

void MuniView::CreateResume()
{
  ReplacePlace(new MuniResume(this,config,meta));
}

void MuniView::DeleteResume()
{
}

void MuniView::CreateHead()
{
  ReplacePlace(new MuniHead(this,config));
  static_cast<MuniHead *>(place)->SetHdu(fits.Hdu(hdusel));
  GetToolBar()->EnableTool(ID_INFO,true);

  wxMenu *menuHead = new wxMenu;
  menuHead->Append(ID_INFO,"Header");
  GetMenuBar()->Insert(3,menuHead,MENU_HEAD);

  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnHeader,this,ID_INFO);
}

void MuniView::DeleteHead()
{
  GetToolBar()->EnableTool(ID_INFO,false);
  Unbind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnHeader,this,ID_INFO);
  wxMenuBar *menuBar = GetMenuBar();
  int id = menuBar->FindMenu(MENU_HEAD);
  if( id != wxNOT_FOUND ) {
    wxMenu *menuHead = menuBar->Remove(id);
    delete menuHead;
  }
}

void MuniView::CreateGrid()
{
  ReplacePlace(new MuniGrid(this,config));
  static_cast<MuniGrid *>(place)->SetHdu(fits.Hdu(hdusel));
  GetToolBar()->EnableTool(ID_INFO,true);

  wxMenu *menuGrid = new wxMenu;
  menuGrid->Append(ID_INFO,"Header");
  GetMenuBar()->Insert(3,menuGrid,MENU_GRID);

  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnHeader,this,ID_INFO);
}

void MuniView::DeleteGrid()
{
  GetToolBar()->EnableTool(ID_INFO,false);
  Unbind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnHeader,this,ID_INFO);

  wxMenuBar *menuBar = GetMenuBar();
  int id = menuBar->FindMenu(MENU_GRID);
  if( id != wxNOT_FOUND ) {
    wxMenu *menuGrid = menuBar->Remove(id);
    delete menuGrid;
  }
}

void MuniView::CreateDisplay()
{
  ReplacePlace(new MuniDisplay(this,config));
  static_cast<MuniDisplay *>(place)->SetHdu(fits.Hdu(hdusel),
					    meta.Hdu(hdusel).GetIcon(),
					    palauto);

  // menus
  menuView->AppendCheckItem(ID_DETAIL,"Detail Panel");
  menuView->Check(ID_DETAIL,true);
  menuView->AppendSeparator();
  menuView->AppendCheckItem(ID_GRID,"Coordinate Grid");
  menuView->AppendCheckItem(ID_SOURCES,"Detected Sources\tCtrl+G");

  wxMenu *menuImage = new wxMenu;
  menuImage->Append(ID_INFO,"Header");
  //  menuImage->Append(ID_PHOTOMETRY,"Photometry");
  //  menuImage->Append(ID_ASTROMETRY,"Astrometry");
  //  menuImage->Append(ID_CALIBR,"Calibrate");
  //  menuImage->AppendSeparator();
  //  menuImage->AppendCheckItem(ID_GRID,"Grid");

  wxMenu *menuTune = new wxMenu;
  menuTune->Append(wxID_ZOOM_FIT,"Best fit\tCtrl+*");
  menuTune->Append(wxID_ZOOM_100,"Normal size\tCtrl+0");
  menuTune->Append(wxID_ZOOM_IN,"Zoom\tCtrl++");
  menuTune->Append(wxID_ZOOM_OUT,"Shrink\tCtrl+-");
  menuTune->AppendSeparator();

  FitsArray hdu(fits.Hdu(hdusel));
  if( ! hdu.IsColor() ) {

    wxMenu *menuPalette = new wxMenu;
    wxArrayString pals = FitsPalette::Type_str();
    for(size_t i = 0; i < pals.GetCount(); i++) {
      menuPalette->AppendRadioItem(wxID_ANY,pals.Item(i));
      long id = menuPalette->FindItem(pals.Item(i));
      Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnMenuPal,this,id);
    }

    wxString label(FitsPalette::Type_str(config->display_pal));
    int id = menuPalette->FindItem(label);
    wxASSERT(id != wxNOT_FOUND);
    menuPalette->Check(id,true);

    menuTune->Append(ID_CYCLE_PAL,"&Cycle Palette\tCtrl+P",
		     "Cycle throughout Colour Palettes");
    menuTune->AppendSubMenu(menuPalette,"&" COLOR_PALETTES);
    menuTune->AppendCheckItem(ID_MENU_INVERSE,"Inverse");
    menuTune->AppendCheckItem(ID_MENU_PALAUTO,"Auto Select");
    menuTune->AppendSeparator();
    menuTune->Check(ID_MENU_INVERSE,palinv);
    menuTune->Check(ID_MENU_PALAUTO,palauto);
  }

  menuTune->Append(ID_TUNE,"&Tune...");

  wxMenu *menuTools = new wxMenu;
  menuTools->Append(ID_PHOTOMETRY,"Photometry...");
  menuTools->Append(ID_ASTROMETRY,"Astrometry...");

  GetMenuBar()->Insert(3,menuImage,MENU_FRAME);
  GetMenuBar()->Insert(4,menuTune,MENU_TUNE);
  GetMenuBar()->Insert(5,menuTools,MENU_TOOLS);
  menuView->Enable(ID_FULLSCREEN,true);
  menuView->Enable(ID_GRID,true);
  menuView->Enable(ID_SOURCES,true);
  menuTools->Enable(ID_PHOTOMETRY,false);

  // toolbar
  MuniArtIcons ico(wxART_TOOLBAR,wxSize(22,22));

  wxToolBar *tbar = GetToolBar();
  wxASSERT(tbar);

  int n = tbar->GetToolPos(ID_INFO) + 2;

  tbar->InsertTool(n,ID_TUNE,"Tune",ico.Icon("preferences-desktop"),
		   wxNullBitmap,wxITEM_NORMAL,"Fine tunning controls");

  wxArrayString zoomlist;
  zoomlist.Add("1:8");
  zoomlist.Add("1:6");
  zoomlist.Add("1:4");
  zoomlist.Add("1:3");
  zoomlist.Add("1:2");
  zoomlist.Add("1:1");
  zoomlist.Add("2:1");
  zoomlist.Add("4:1");

  zoomctrl = new wxComboBox(tbar,ID_TOOLZOOM,"",wxDefaultPosition,
			    wxDefaultSize,zoomlist,wxCB_READONLY);
  tbar->InsertControl(n+1,zoomctrl);

  tbar->InsertTool(n+2,wxID_ZOOM_100,"Original",ico.Icon("zoom-original"),
		   wxNullBitmap,wxITEM_NORMAL,"Original Size");
  tbar->InsertTool(n+3,wxID_ZOOM_FIT,"Fit",ico.Icon("zoom-fit-best"),
  		   wxNullBitmap,wxITEM_NORMAL,"Fit to Size");

  tbar->EnableTool(ID_INFO,true);

  tbar->Realize();


  // some stars
  for(size_t i = 0; i < fits.HduCount(); i++) {
    if( fits.Hdu(i).GetKey("EXTNAME").Find("MUNIPACK") != wxNOT_FOUND ) {
      FitsTable t(fits.Hdu(i));
      dynamic_cast<MuniDisplay *>(place)->SetStars(t);
      /*
      wxMemoryOutputStream ostream;
      t.GetStarChart(ostream);

      wxMemoryInputStream istream(ostream);
      dynamic_cast<MuniDisplay *>(place)->SetOverlay(istream);
      */
    }
  }

  Bind(EVT_TUNE,&MuniView::OnTuneFine,this);
  Bind(wxEVT_COMBOBOX,&MuniView::OnToolZoom,this,ID_TOOLZOOM);
  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnMenuZoom,this,wxID_ZOOM_100);
  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnMenuZoom,this,wxID_ZOOM_FIT);
  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnCycleZoom,this,wxID_ZOOM_IN);
  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnCycleZoom,this,wxID_ZOOM_OUT);
  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnCyclePal,this,ID_CYCLE_PAL);
  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnTune,this,ID_TUNE);
  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnDetailPanel,this,ID_DETAIL);
  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnHeader,this,ID_INFO);
  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnCalibrate,this,ID_CALIBR);
  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnShowGrid,this,ID_GRID);
  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnShowSources,this,ID_SOURCES);
  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnPhotometry,this,ID_PHOTOMETRY);
  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnAstrometry,this,ID_ASTROMETRY);
  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnMenuInverse,this,
       ID_MENU_INVERSE);
  Bind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnMenuPalauto,this,
       ID_MENU_PALAUTO);

  double xpix, ypix, alpha, delta, scale, angle, reflex;
  if( fits.Hdu(hdusel).GetWCS(xpix, ypix, alpha, delta, scale, angle, reflex) ){
    menuView->Check(ID_GRID,config->display_grid);
    wxASSERT(dynamic_cast<MuniDisplay *>(place));
    dynamic_cast<MuniDisplay *>(place)->ShowGrid(config->display_grid);
  }

  if( fits.HasPhotometry() || fits.HasPhcal() ) {
    menuView->Check(ID_SOURCES,config->display_sources);
    wxASSERT(dynamic_cast<MuniDisplay *>(place));
    dynamic_cast<MuniDisplay *>(place)->ShowSources(config->display_sources,fits);
  }

  MuniTuneEvent ev(EVT_TUNE,ID_PALETTE_INVERSE);
  ev.SetInt(palinv);
  wxQueueEvent(place,ev.Clone());
}

void MuniView::DeleteDisplay()
{
  wxToolBar *tbar = GetToolBar();
  wxASSERT(tbar);

  tbar->DeleteTool(ID_TOOLZOOM);
  tbar->DeleteTool(ID_TUNE);
  tbar->DeleteTool(wxID_ZOOM_FIT);
  tbar->DeleteTool(wxID_ZOOM_100);

  tbar->EnableTool(ID_INFO,false);

  tbar->Realize();

  Unbind(EVT_TUNE,&MuniView::OnTuneFine,this);
  Unbind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnMenuZoom,this,wxID_ZOOM_100);
  Unbind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnMenuZoom,this,wxID_ZOOM_FIT);
  Unbind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnCycleZoom,this,wxID_ZOOM_IN);
  Unbind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnCycleZoom,this,wxID_ZOOM_OUT);
  Unbind(wxEVT_COMBOBOX,&MuniView::OnToolZoom,this,ID_TOOLZOOM);
  Unbind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnCyclePal,this,ID_CYCLE_PAL);
  Unbind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnTune,this,ID_TUNE);
  Unbind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnDetailPanel,this,ID_DETAIL);
  Unbind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnHeader,this,ID_INFO);
  Unbind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnCalibrate,this,ID_CALIBR);
  Unbind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnShowGrid,this,ID_GRID);
  Unbind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnShowSources,this,ID_SOURCES);
  Unbind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnPhotometry,this,
	 ID_PHOTOMETRY);
  Unbind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnAstrometry,this,
	 ID_ASTROMETRY);
  Unbind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnMenuInverse,this,
	 ID_MENU_INVERSE);
  Unbind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnMenuPalauto,this,
	 ID_MENU_PALAUTO);

  wxMenuBar *menuBar = GetMenuBar();
  int id = menuBar->FindMenu(MENU_FRAME);
  if( id != wxNOT_FOUND ) {
    wxMenu *menuImage = menuBar->Remove(id);
    delete menuImage;
  }
  id = menuBar->FindMenu(MENU_TUNE);
  if( id != wxNOT_FOUND ) {
    wxMenu *menuTune = menuBar->Remove(id);
    delete menuTune;
  }
  id = menuBar->FindMenu(MENU_TOOLS);
  if( id != wxNOT_FOUND ) {
    wxMenu *menuTools = menuBar->Remove(id);
    delete menuTools;
  }

  menuView->Delete(ID_DETAIL);
  menuView->Delete(ID_GRID);
  menuView->Delete(ID_SOURCES);
  //  menuView->Enable(ID_GRID,false);
  menuView->Enable(ID_FULLSCREEN,false);

  // delete pal and profile bindings !!!
  // Unbind(wxEVT_COMMAND_MENU_SELECTED,&MuniView::OnMenuPal,this,id);


  // delete display itself !!!!!!!!!!!!!!
  //  static_cast<MuniDisplay *>(place)->StopRendering();
}

void MuniView::OnDetailPanel(wxCommandEvent& event)
{
  wxASSERT(dynamic_cast<MuniDisplay *>(place));
  dynamic_cast<MuniDisplay *>(place)->ShowPanel(event.IsChecked());
}


void MuniView::OnEnterWin(wxMouseEvent& event)
{
  wxLogDebug("MuniView::OnEnterWin");
  wxASSERT(place);
  place->SetFocus();
}

void MuniView::OnSize(wxSizeEvent& event)
{
  //  wxASSERT(place);
  //Layout();

  // arrange stretch space in toolbar
  //  wxSize size(GetSize());

  /*
  if( dynamic_cast<MuniDisplay*>(place) )
    place->SetSize(1,1);
  */
  // Black Magick. The part of code together with code in display.cpp
  // setups scrolling for images. I had spend lot of time with looking
  // for a way how correctly does it. The principle is set size
  // of place to minumum while deriving virtual size by image.

  //  wxPostEvent(place,event);

  event.Skip();
}

void MuniView::Coloring(wxCommandEvent& WXUNUSED(event))
{
  if( coloring ) return;

  Bind(EVT_FILELOAD,&MuniView::OnColoringFinish,this);

  coloring = new MuniColoring(this,config);
  coloring->Show(true);
}

void MuniView::OnColoringFinish(wxCommandEvent& event)
{
  wxLogDebug("MuniView::OnColoringFinish");

  wxASSERT(coloring);
  coloring->Destroy();
  coloring = 0;

  Unbind(EVT_FILELOAD,&MuniView::OnColoringFinish,this);

  // CHECK!!! (probably unproper handling of backups !!!!

  wxString file(event.GetString());
  if( ! file.IsEmpty() )
    LoadFile(file);
}

// void MuniView::OnPhotometry(MuniDialogEvent& event)
// {
//   wxLogDebug("Running OnPhotometry...");

//   result = event.results[0];

//   Bind(wxEVT_END_PROCESS,&MuniView::OnPhotometryFinish,this);

//   MuniProcess *action = new MuniProcess(&pipe,event.commands[0]);
//   pipe.push(action);

//   for(size_t i = 0; i < event.params.GetCount(); i++){
//     wxLogDebug(event.params[i]);
//     action->Write(event.params[i]);
//   }

//   pipe.Start();

// }

// void MuniView::OnPhotometryFinish(wxProcessEvent& event)
// {
//   wxLogDebug("MuniView::OnPhotometryFinish");

//   Unbind(wxEVT_END_PROCESS,&MuniView::OnPhotometryFinish,this);

//   if( event.GetExitCode() != 0 )
//     wxLogDebug("Failed with exit code %d",event.GetExitCode());
//   else
//     LoadFile(result);
// }

// void MuniView::RunShell(const queue<MuniCommander>& com)
// {
//   //  wxBeginBusyCursor(); // must be called out of exec and write to

//   shell = new MuniShell(GetEventHandler(),com);

//   /*
//   menuView->Enable(wxID_STOP,true);
//   tbot->AddTool(tstop);
//   tbot->Realize();
//   archiveprop->SetLabel(_("Files are being processed."));
//   SetArchiveSize();
//   */
// }

// void MuniView::OnShell(MuniShellEvent& event)
// {
//   wxLogDebug("MuniBrowser::OnShell");

//   if( event.finish ) {
//     /*
//     tstop = tbot->RemoveTool(wxID_STOP);
//     tbot->Realize();
//     menuView->Enable(wxID_STOP,false);
//     */
//     delete shell;
//     shell = 0;

//     //    wxEndBusyCursor();
//   }

//   /*
//   console->AppendOutput(event.out);
//   console->AppendError(event.err);
//   */

//   wxArrayString results = event.res;
//   for(size_t i = 0; i < results.GetCount(); i++) {
//     wxLogDebug(results[i]);
//     LoadFile(results[i]);
//     /*
//     FitsFile fits(results[i]);
//     if( fits.Status() ) {
//       MuniIcon micon(fits,config);
//       FitsMeta meta(fits,micon.GetIcon(),micon.GetList());
//       if( meta.IsOk() ) {
// 	FitsOpenEvent ev(xEVT_FITS_OPEN,ID_MRENDER);
// 	ev.filename = results[i];
// 	ev.fits = fits;
// 	ev.meta = meta;
// 	wxPostEvent(list,ev);
//       }
//     }
//     */
//   }

// }
