/*
 *  Copyright (c) 2008-2009 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "CodeGenerator_p.h"

#include <vector>

#include <llvm/CallingConv.h>
#include <llvm/Constants.h>
#include <llvm/Constant.h>
#include <llvm/DerivedTypes.h>
#include <llvm/Function.h>
#include <llvm/Instructions.h>
#include <llvm/Module.h>
#include <llvm/Value.h>

#include "GTLCore/LLVMBackend/CodeGenerator_p.h"
#include "GTLCore/LLVMBackend/ExpressionResult_p.h"
#include "GTLCore/Function.h"
#include "GTLCore/Function_p.h"
#include "GTLCore/LLVMBackend/GenerationContext_p.h"
#include "GTLCore/ModuleData_p.h"
#include "GTLCore/Parameter.h"
#include "GTLCore/PixelDescription.h"
#include "GTLCore/Type.h"
#include "GTLCore/TypesManager_p.h"
#include "GTLCore/Value.h"
#include "GTLCore/VariableNG_p.h"

#include "Wrapper_p.h"
#include "wrappers/ImageWrap_p.h"
#include "wrappers/PixelWrap_p.h"

#include "Debug.h"
#include "Kernel.h"

using namespace OpenShiva;

int CodeGenerator::s_evaluatePixelsId = 0;

llvm::Function* CodeGenerator::createMemCpyFunction( llvm::Module* _module )
{
  // TODO that stuff was copied from a llvmgcc output, can't we use memcpy intrinsinc ?
  std::vector<llvm::Type*> memcpyTyArgs;
  memcpyTyArgs.push_back(llvm::PointerType::get(llvm::Type::getInt8Ty(_module->getContext()), 0));
  memcpyTyArgs.push_back(llvm::PointerType::get(llvm::Type::getInt8Ty(_module->getContext()), 0));
  memcpyTyArgs.push_back(llvm::Type::getInt32Ty(_module->getContext()));
  memcpyTyArgs.push_back(llvm::Type::getInt32Ty(_module->getContext()));
  llvm::FunctionType* memcpyTy = llvm::FunctionType::get( llvm::Type::getVoidTy(_module->getContext()), memcpyTyArgs, false);
  
  llvm::Function* func_llvm_memcpy_i32 = (llvm::Function*)_module->getOrInsertFunction(
      "llvm.memcpy.i32", memcpyTy); // (external, no body)
  func_llvm_memcpy_i32->setCallingConv(llvm::CallingConv::C);
  
/*  const llvm::ParamAttrsList *func_llvm_memcpy_i32_PAL = 0;
  {
    llvm::ParamAttrsVector Attrs;
    llvm::ParamAttrsWithIndex PAWI;
    PAWI.index = 0; PAWI.attrs = 0  | llvm::ParamAttr::NoUnwind;
    Attrs.push_back(PAWI);
    func_llvm_memcpy_i32_PAL = llvm::ParamAttrsList::get(Attrs);
  }
  func_llvm_memcpy_i32->setParamAttrs(func_llvm_memcpy_i32_PAL);*/
  return func_llvm_memcpy_i32;
}

llvm::Value* CodeGenerator::accessPixelAlphaPos( LLVMBackend::GenerationContext& _gc, llvm::BasicBlock* _currentBlock, llvm::Value* _pixel)
{
  SHIVA_DEBUG( *_pixel );
  std::vector<llvm::Value*> indexes;
  indexes.push_back( _gc.codeGenerator()->integerToConstant(_gc.llvmContext(), INT32_C(0)));
  indexes.push_back( _gc.codeGenerator()->integerToConstant(_gc.llvmContext(), gtl_int32(PixelWrap::POS_ALPHAPOS) ));
  return llvm::GetElementPtrInst::Create( _pixel, indexes, "", _currentBlock );
}

llvm::Value* CodeGenerator::accessPixelDataPtr( LLVMBackend::GenerationContext& _gc, llvm::BasicBlock* _currentBlock, llvm::Value* _pixel)
{
  SHIVA_DEBUG( *_pixel );
  std::vector<llvm::Value*> indexes;
  indexes.push_back( _gc.codeGenerator()->integerToConstant(_gc.llvmContext(), INT32_C(0)));
  indexes.push_back( _gc.codeGenerator()->integerToConstant(_gc.llvmContext(), gtl_int32(PixelWrap::POS_DATA) ));
  return llvm::GetElementPtrInst::Create( _pixel, indexes, "", _currentBlock );
}

llvm::Value* CodeGenerator::accessColorConverterPtr( LLVMBackend::GenerationContext& _gc, llvm::BasicBlock* _currentBlock, llvm::Value* _pixel)
{
  SHIVA_DEBUG( *_pixel );
  std::vector<llvm::Value*> indexes;
  indexes.push_back( _gc.codeGenerator()->integerToConstant(_gc.llvmContext(), INT32_C(0)));
  indexes.push_back( _gc.codeGenerator()->integerToConstant(_gc.llvmContext(), gtl_int32(PixelWrap::POS_CONVERTER) ));
  return llvm::GetElementPtrInst::Create( _pixel, indexes, "", _currentBlock );
}

llvm::Value* CodeGenerator::accessPixelDataAsU8Ptr( LLVMBackend::GenerationContext& _gc, llvm::BasicBlock* _currentBlock, llvm::Value* _pixel)
{
  llvm::Value* _pointerToPixelVector = accessPixelDataPtr( _gc, _currentBlock, _pixel) ;
  // Cast the vector to a "char*"
  return new llvm::BitCastInst(
      _pointerToPixelVector, llvm::PointerType::get(llvm::Type::getInt8Ty(_gc.llvmContext()), 0), "", _currentBlock);
}

llvm::Value* CodeGenerator::accessPixelDataAsF32Ptr( LLVMBackend::GenerationContext& _gc, llvm::BasicBlock* _currentBlock, llvm::Value* _pixel)
{
  llvm::Value* _pointerToPixelVector = accessPixelDataPtr( _gc, _currentBlock, _pixel) ;
  // Cast the vector to a "char*"
  return new llvm::BitCastInst(
      _pointerToPixelVector, llvm::PointerType::get(llvm::Type::getFloatTy(_gc.llvmContext()), 0), "", _currentBlock);
}

llvm::Value* CodeGenerator::callMemcpy( LLVMBackend::GenerationContext& _gc, llvm::BasicBlock* _currentBlock, llvm::Value* _dst, llvm::Value* _src, llvm::Value* _n )
{
  SHIVA_DEBUG( "memcpy( dst = " << *_dst << ", src = " << *_src << ", _n = " << *_n );
  // Initialise llvm_memcpy_i32
  llvm::Function* func_llvm_memcpy_i32 = createMemCpyFunction( _gc.llvmModule() );
  // Set parameters
  std::vector<llvm::Value*> memcpy_params;
  memcpy_params.push_back( _dst );
  memcpy_params.push_back( _src );
  memcpy_params.push_back( _n );
  memcpy_params.push_back( LLVMBackend::CodeGenerator::integerToConstant( _gc.llvmContext(), INT32_C(1) ) );
  llvm::CallInst* callMemCpy = llvm::CallInst::Create(
                  func_llvm_memcpy_i32, memcpy_params, "", _currentBlock);
  callMemCpy->setCallingConv(llvm::CallingConv::C);
  callMemCpy->setTailCall( false );
  return callMemCpy;
}

llvm::BasicBlock* CodeGenerator::memToPixel( LLVMBackend::GenerationContext& _gc, llvm::BasicBlock* _currentBlock, llvm::Value* _dataPointer, llvm::Value* _pixel, const GTLCore::Type* _imageType, llvm::Value* _image )
{
  // Access to the data pointer for the _pixel
  llvm::Value* _pointerToPixelVector = accessPixelDataPtr( _gc, _currentBlock, _pixel );
  
  std::vector<llvm::Value*> arguments;
  arguments.push_back( _pointerToPixelVector );
  arguments.push_back( _dataPointer );
  callVirtualMember( _gc, _currentBlock, _image, ImageWrap::INDEX_MEM_TO_VEC, arguments);

  llvm::Value* _pointerToColorConverter = accessColorConverterPtr( _gc, _currentBlock, _pixel );
  
  new llvm::StoreInst(callImageColorConverter(_gc, _currentBlock, _imageType, _image), _pointerToColorConverter, "CodeGenerator::memToPixel", _currentBlock);
  
  
  return _currentBlock;
}

llvm::BasicBlock* CodeGenerator::pixelToMem( LLVMBackend::GenerationContext& _gc, llvm::BasicBlock* _currentBlock, llvm::Value* _pixel, llvm::Value* _dataPointer, llvm::Value* _image, llvm::Value* channelsFlags )
{
  // Access to the data pointer for the _pixel
  llvm::Value* _pointerToPixelVector = accessPixelDataPtr( _gc, _currentBlock, _pixel );
  
  std::vector<llvm::Value*> arguments;
  arguments.push_back( _dataPointer );
  arguments.push_back( _pointerToPixelVector );
  arguments.push_back( channelsFlags );
  callVirtualMember( _gc, _currentBlock, _image, ImageWrap::INDEX_VEC_TO_MEM, arguments);
  
  return _currentBlock;
}

void CodeGenerator::setPixelCoordinates( LLVMBackend::GenerationContext& _gc, llvm::BasicBlock* _currentBlock, llvm::Value* _pixel, llvm::Value* _iVal, const GTLCore::Type* _iValType, llvm::Value* _jVal, const GTLCore::Type* _jValType )
{
  SHIVA_DEBUG("setPixelCoordinates");
  // Create the coord vector
  llvm::Value* result = LLVMBackend::CodeGenerator::createVector( 2, LLVMBackend::CodeGenerator::floatToConstant( _gc.llvmContext(), 0.0 ) );
  SHIVA_DEBUG(*result << " " << *_iVal << " " << *_jVal);
  result = llvm::InsertElementInst::Create( result, LLVMBackend::CodeGenerator::convertValueTo( _currentBlock, _iVal, _iValType, GTLCore::Type::Float32), LLVMBackend::CodeGenerator::integerToConstant(_gc.llvmContext(), INT32_C(0)), "", _currentBlock );
  result = llvm::InsertElementInst::Create( result, LLVMBackend::CodeGenerator::convertValueTo( _currentBlock, _jVal, _jValType, GTLCore::Type::Float32), LLVMBackend::CodeGenerator::integerToConstant(_gc.llvmContext(), INT32_C(1)), "", _currentBlock );
  
  std::vector<llvm::Value*> indexes;
  indexes.push_back( _gc.codeGenerator()->integerToConstant(_gc.llvmContext(), INT32_C(0)));
  indexes.push_back( _gc.codeGenerator()->integerToConstant(_gc.llvmContext(), gtl_int32(PixelWrap::POS_COORD) ));
  llvm::Value* ptr = llvm::GetElementPtrInst::Create( _pixel, indexes, "", _currentBlock );
  SHIVA_DEBUG( *_pixel << " " << *ptr << " " << *result);
  new llvm::StoreInst( result, ptr, "", _currentBlock );
}

void CodeGenerator::setPixelAlphaPos( LLVMBackend::GenerationContext& _gc, llvm::BasicBlock* _currentBlock, llvm::Value* _pixel, llvm::Value* _image, const GTLCore::Type* _imageType )
{
  // Get the alpha position from the image
  llvm::Value* alphaPos = callImageAlphaPos(_gc, _currentBlock, _imageType, _image);
  
  // Create a pointer to the alphaPos index in the pixel structure
  std::vector<llvm::Value*> indexes;
  indexes.push_back( _gc.codeGenerator()->integerToConstant(_gc.llvmContext(), INT32_C(0)));
  indexes.push_back( _gc.codeGenerator()->integerToConstant(_gc.llvmContext(), gtl_int32(PixelWrap::POS_ALPHAPOS) ));
  llvm::Value* ptr = llvm::GetElementPtrInst::Create( _pixel, indexes, "", _currentBlock );
  
  // Store the alpha position
  new llvm::StoreInst( alphaPos, ptr, "", _currentBlock );
}    

llvm::Function* CodeGenerator::generateEvaluatePixels( Kernel* _kernel, llvm::Module* _module, int _channels)
{
  GTLCore::ModuleData* moduleData = _kernel->d->moduleData();
  LLVMBackend::CodeGenerator codeGenerator( moduleData );
  
  // Pseudo-code explaining step-by-step what an evaluatePixels function does :
  //
  //  AbstractImage* are seen as a void*.
  //
  //  template< int nbSources >
  //  void evaluatePixels( int _x, int _y, int _width, int _height, AbstractImage** _sources,
  //                       AbstractImage* _result, ProgressReport* _report, Transform* transfo, gtl_uint64 _channelFlags)
  //  {
  //    for( int i = _x; i < nbSource; ++i )
  //    {
  //      imageWrap[i] = wrap(sources[i]);
  //    }
  //    for( int j = _y; j < height; ++j )
  //    {
  //      for( int i = 0; i < width; ++i )
  //      {
  //        pixel result;
  //        evaluatePixel( imageWrap[0], imageWrap[1], ..., imageWrap[nbSource-1], result );
  //        for( int k = 0; k < channels; ++k)
  //        {
  //          int channelMask = 1 << k;
  //          if( (_channelFlags & channelMask) == channelMask )
  //          {
  //            _result.data( width, height )[k] = result.data[k];
  //          }
  //        }
  //        _report->nextPixel();
  //      }
  //      _report->nextRow();
  //    }
  //  }
  //
  // Simple, isn't it ?
  
  llvm::LLVMContext& context = moduleData->llvmModule()->getContext();
  
  // Create the function
  std::vector<llvm::Type*> params;
  params.push_back( llvm::Type::getInt32Ty(context) ); // int x
  params.push_back( llvm::Type::getInt32Ty(context) ); // int y
  params.push_back( llvm::Type::getInt32Ty(context) ); // int width
  params.push_back( llvm::Type::getInt32Ty(context) ); // int height
  SHIVA_ASSERT( _kernel->d->outputImageType() );
  params.push_back( llvm::PointerType::get( GTLCore::Type::Pointer->d->type(context), 0 ) ); // AbstractImage** _sources
  params.push_back( llvm::PointerType::get( _kernel->d->outputImageType()->d->type(context), 0 ) ); // AbstractImage* _result
  params.push_back( GTLCore::Type::Pointer->d->type(context) ); // ProgressReport* _report
  params.push_back( GTLCore::Type::Pointer->d->type(context) ); // Transform* _transfo
  params.push_back( llvm::Type::getInt64Ty(context) ); // gtl_uint64 _channelFlags
  llvm::FunctionType* definitionType = llvm::FunctionType::get( llvm::Type::getVoidTy(context), params, false );
  int evaluatePixelsId = ++CodeGenerator::s_evaluatePixelsId;
  llvm::Function* func = codeGenerator.createFunction( _module, definitionType, "evaluatePixels" + GTLCore::String::number(evaluatePixelsId));
  // Initialise a generation context
  LLVMBackend::GenerationContext generationContext( &codeGenerator, &context, func, 0, moduleData , _module);
  
  // {
    llvm::BasicBlock* initialBlock = llvm::BasicBlock::Create(context);
    func->getBasicBlockList().push_back( initialBlock );
  // Get the args.
    llvm::Function::arg_iterator arg_it = func->arg_begin();
    //   int _x = first arg;
    llvm::Value* arg_x = arg_it;
    //   int _y = second arg;
    ++arg_it;
    llvm::Value* arg_y = arg_it;
    //   int _width = third arg;
    ++arg_it;
    llvm::Value* arg_width = arg_it;
    llvm::Value* arg_x_p_width = LLVMBackend::CodeGenerator::createAdditionExpression(initialBlock, arg_width, GTLCore::Type::Integer32, arg_x, GTLCore::Type::Integer32) ;
    //   int _height = fourth arg;
    ++arg_it;
    llvm::Value* arg_height = arg_it;
    llvm::Value* arg_y_p_height = LLVMBackend::CodeGenerator::createAdditionExpression(initialBlock, arg_height, GTLCore::Type::Integer32, arg_y, GTLCore::Type::Integer32) ;
    //   void** _sources = fifth arg;
    ++arg_it;
    llvm::Value* arg_sources = arg_it;
    //   void* _result = sixth arg;
    ++arg_it;
    llvm::Value* arg_result = arg_it;
    //   void* _report = seventh arg;
    ++arg_it;
    llvm::Value* arg_report = arg_it;
    //   gtl_uint64 _transfo = eigth arg;
    ++arg_it;
    llvm::Value* arg_transfo = arg_it;
    //   gtl_uint64 _channelFlags = nineth arg;
    ++arg_it;
    llvm::Value* arg_channelFlags = arg_it;
  // Create the pixel
    GTLCore::VariableNG* resultVar = new GTLCore::VariableNG( _kernel->d->outputPixelType(), false, false );
    resultVar->initialise( generationContext, initialBlock, LLVMBackend::ExpressionResult(), std::list<llvm::Value*>());
    llvm::Value* _pointerToColorConverter = accessColorConverterPtr( generationContext, initialBlock, resultVar->pointer(initialBlock) );
  
    new llvm::StoreInst(callImageColorConverter(generationContext, initialBlock, _kernel->d->outputImageType(), arg_result), _pointerToColorConverter, "CodeGenerator::memToPixel", initialBlock);
  
  
  // Get the evaluatePixel function
    const std::list<GTLCore::Function*>* ePFunctions = moduleData->function( _kernel->name(), "evaluatePixel" );
    SHIVA_ASSERT( ePFunctions );
    GTLCore::Function* ePFunction = ePFunctions->front();
    unsigned int countArguments = ePFunction->parameters().size();
    
  // Buffer for the map function
    llvm::Value* iValBuffer = new llvm::AllocaInst(llvm::Type::getFloatTy(context), "", initialBlock);
    llvm::Value* jValBuffer = new llvm::AllocaInst(llvm::Type::getFloatTy(context), "", initialBlock);
    
  // Create the list of parameters for the evaluatePixel function
    std::vector<llvm::Value*> evaluatePixel_params;
    for(gtl_uint32 i = 0; i < countArguments - 1; ++i )
    {
      llvm::Value* imgPtr = new llvm::LoadInst(
              llvm::GetElementPtrInst::Create( arg_sources, LLVMBackend::CodeGenerator::integerToConstant( context, i ), "", initialBlock ),
              "", initialBlock );
      evaluatePixel_params.push_back( LLVMBackend::CodeGenerator::convertPointerTo( initialBlock, imgPtr, ePFunction->parameters()[i].type()->d->type(context) ) );
    }
    evaluatePixel_params.push_back( resultVar->pointer(initialBlock) );
    SHIVA_ASSERT( evaluatePixel_params.size() == countArguments );
  // Construct the "conditions" of the first loop
    // int j = arg_y;
    GTLCore::VariableNG* incJ = new GTLCore::VariableNG( GTLCore::Type::Integer32, false, false);
    incJ->initialise( generationContext, initialBlock,
                      LLVMBackend::ExpressionResult( arg_y, GTLCore::Type::Integer32),
                      std::list<llvm::Value*>());
    
    // {
      llvm::BasicBlock* firstBlockJLoop = llvm::BasicBlock::Create(context);
      func->getBasicBlockList().push_back( firstBlockJLoop );
      
      // int i = arg_x;
      GTLCore::VariableNG* incI = new GTLCore::VariableNG( GTLCore::Type::Integer32, false, false);
      incI->initialise( generationContext, firstBlockJLoop,
                        LLVMBackend::ExpressionResult( arg_x, GTLCore::Type::Integer32),
                        std::list<llvm::Value*>());
      
      // {
        llvm::BasicBlock* firstBlockILoop = llvm::BasicBlock::Create(context);
        func->getBasicBlockList().push_back( firstBlockILoop );
        llvm::Value* jVal_int = incJ->get( generationContext, firstBlockILoop );
        llvm::Value* iVal_int = incI->get( generationContext, firstBlockILoop );

        // Call the map function of the transform object
        llvm::Value* jVal_transformed = LLVMBackend::CodeGenerator::convertValueTo(firstBlockILoop,
                                                jVal_int, GTLCore::Type::Integer32, GTLCore::Type::Float32 );
        llvm::Value* iVal_transformed = LLVMBackend::CodeGenerator::convertValueTo(firstBlockILoop,
                                                iVal_int, GTLCore::Type::Integer32, GTLCore::Type::Float32 );
        
        std::vector<llvm::Type*> transfoArgs;
        transfoArgs.push_back(llvm::PointerType::get(llvm::Type::getInt8Ty(context), 0));
        transfoArgs.push_back(llvm::Type::getFloatTy(context));
        transfoArgs.push_back(llvm::Type::getFloatTy(context));
        transfoArgs.push_back(llvm::PointerType::get(llvm::Type::getFloatTy(context), 0));
        transfoArgs.push_back(llvm::PointerType::get(llvm::Type::getFloatTy(context), 0));
        llvm::FunctionType* transfoMapTy = llvm::FunctionType::get( llvm::Type::getVoidTy(context), transfoArgs, false);
        
        llvm::Function* func_transform_map = (llvm::Function*)_module->getOrInsertFunction(
            "gtl_transform_map_f", transfoMapTy); // (external, no body)
        func_transform_map->setCallingConv(llvm::CallingConv::C);

        std::vector<llvm::Value*> transfo_params;
        transfo_params.push_back( arg_transfo );
        transfo_params.push_back( iVal_transformed );
        transfo_params.push_back( jVal_transformed );
        transfo_params.push_back( iValBuffer );
        transfo_params.push_back( jValBuffer );
        llvm::CallInst *CallFunc = llvm::CallInst::Create(func_transform_map, transfo_params, "", firstBlockILoop);
        CallFunc->setTailCall(false);
        
        iVal_transformed = new llvm::LoadInst(iValBuffer, "", firstBlockILoop);
        jVal_transformed = new llvm::LoadInst(jValBuffer, "", firstBlockILoop);
        
        // Fill the pixel variable
        llvm::Value* px_var = resultVar->pointer(firstBlockILoop);
        
        // Set the coordinates of the pixel
        setPixelCoordinates( generationContext, firstBlockILoop, px_var, 
                LLVMBackend::CodeGenerator::createAdditionExpression( firstBlockILoop, iVal_transformed,
                                                         GTLCore::Type::Float32,
                          LLVMBackend::CodeGenerator::floatToConstant( generationContext.llvmContext(), 0.5f ), GTLCore::Type::Float32 ),
                GTLCore::Type::Float32,
                LLVMBackend::CodeGenerator::createAdditionExpression( firstBlockILoop, jVal_transformed,
                                                         GTLCore::Type::Float32,
                          LLVMBackend::CodeGenerator::floatToConstant( generationContext.llvmContext(), 0.5f ), GTLCore::Type::Float32 ),
                GTLCore::Type::Float32 );
        
        // Set the position of the alpha channel
        setPixelAlphaPos( generationContext, firstBlockILoop, px_var, arg_result, _kernel->d->outputImageType() );
        
        // Call evaluatePixel
        llvm::Function* llvmEPFunction = ePFunction->d->data->function( );
        SHIVA_ASSERT( llvmEPFunction );
        SHIVA_DEBUG( evaluatePixel_params.size() );
        GTL_COMPARE_FUNCTION_PARAMETERS( llvmEPFunction, evaluatePixel_params );
        llvm::CallInst::Create( llvmEPFunction, evaluatePixel_params, "", firstBlockILoop );
        
        // Synchronize the output pixel with input
        // Call image_wrap_data on the result to get the pointer on destination data
        llvm::Value* pointer = callImageWrapData( generationContext, firstBlockILoop, _kernel->d->outputImageType(), arg_result, iVal_int, jVal_int, false);
        llvm::BasicBlock* lastBlockILoop = pixelToMem( generationContext, firstBlockILoop, resultVar->pointer(firstBlockILoop), pointer, arg_result, arg_channelFlags );
        lastBlockILoop = LLVMBackend::CodeGenerator::callProgressReportNextPixel( generationContext, arg_report, lastBlockILoop );
        
      // }
      
      llvm::BasicBlock* lastBlockJLoop = LLVMBackend::CodeGenerator::createIterationForStatement( generationContext, firstBlockJLoop, incI, arg_x_p_width, GTLCore::Type::Integer32, firstBlockILoop, lastBlockILoop);
      lastBlockJLoop = LLVMBackend::CodeGenerator::callProgressReportNextRow( generationContext, arg_report, lastBlockJLoop );
    // }
    
    llvm::BasicBlock* lastBlock = LLVMBackend::CodeGenerator::createIterationForStatement( generationContext, initialBlock, incJ, arg_y_p_height, GTLCore::Type::Integer32, firstBlockJLoop, lastBlockJLoop);
//     resultVar->cleanUp( generationContext, lastBlock, 0); // FIXME don't leak result pixels
    llvm::ReturnInst::Create(context, lastBlock);
  // Cleanup
  
  delete resultVar;
  delete incJ;
  delete incI;
  return func;
}

int memToVecId = 0;

llvm::Function* CodeGenerator::generateMemToVec( GTLCore::ModuleData* _moduleData, llvm::Module* _module, const GTLCore::PixelDescription& _pixelDescription )
{
  int channelsNb = _pixelDescription.channels();
  // Check if all channels are floats
  bool allFloat = true;
  for( int i = 0; i < channelsNb; ++i)
  {
    if( _pixelDescription.channelTypes()[i]->dataType() != GTLCore::Type::FLOAT32 )
    {
      allFloat = false;
      break;
    }
  }
  //
  // If all channels are float, then do a direct memcpy.
  // Otherwise do a conversion to a float array and memcpy it.
  //
  //  template<bool _TAllFloat_, int _TChannels_>
  //  void memToVec( vec4* _dst, char* _imgData)
  //  {
  //    if( _TAllFloat_ )
  //    {
  //      memcpy( (char*)result, src, _TChannels * 4 );
  //    } else {
  //      for(int i = 0; i < _TChannels; ++i)
  //      {
  //        result[i] = convert(_imgData + pos(i) ) 
  //      }
  //    }
  //  }
  
  llvm::LLVMContext& context = _moduleData->llvmModule()->getContext();
  
  LLVMBackend::CodeGenerator codeGenerator( _moduleData );

  llvm::Function* func = codeGenerator.createFunction( _module,
      Wrapper::image_wrap_mem_to_vec_float_type( context, _moduleData->typesManager(), _pixelDescription.channels()) ,
      "image_wrap_memToVec" + GTLCore::String::number( ++memToVecId) );
  // Initialise a generation context
  LLVMBackend::GenerationContext generationContext( &codeGenerator, &context, func, 0, _moduleData , _module);
  
  // Get the args.
    llvm::Function::arg_iterator arg_it = func->arg_begin();
    //   vec4* _dst = first arg;
    llvm::Value* arg_dst = arg_it;
    //   char* _imgData= second arg;
    ++arg_it;
    llvm::Value* arg_imgData = arg_it;
    
  // {
    llvm::BasicBlock* currentBlock = llvm::BasicBlock::Create(context);
    func->getBasicBlockList().push_back( currentBlock );
    
  //    if( _TAllFloat_ )
    if( allFloat )
  //    {
    {
  //    memcpy( (char*)result, src, _TChannels * 4 );
      callMemcpy( generationContext, currentBlock, 
                  LLVMBackend::CodeGenerator::convertPointerToCharP( currentBlock, arg_dst),
                  arg_imgData, LLVMBackend::CodeGenerator::integerToConstant(context, gtl_uint32(sizeof(float) * 4 ) ) );
  //    } else {
    } else {
  //      for(int i = 0; i < _TChannels; ++i)
      gtl_int32 currentPos = 0;
      llvm::Value* floatVec = new llvm::LoadInst( arg_dst, "", currentBlock );
      for( int i = 0; i < channelsNb; ++i)
  //      {
      {
  //        result[i] = convert(_imgData + pos(i) ) 
        // _imgData + pos(i);
        const GTLCore::Type* channelType = _pixelDescription.channelTypes()[i];
        llvm::Value* posInP = llvm::GetElementPtrInst::Create( arg_imgData, LLVMBackend::CodeGenerator::integerToConstant(context, currentPos ), "", currentBlock );
        llvm::Value* posInPNative = LLVMBackend::CodeGenerator::convertPointerTo( currentBlock, posInP, channelType->d->type(context) );
        llvm::Value* nativeValue = new llvm::LoadInst( posInPNative, "", currentBlock );
        // convert(_imgData + pos(i) )
        llvm::Value* floatValue = LLVMBackend::CodeGenerator::convertValueTo( currentBlock, nativeValue, channelType, GTLCore::Type::Float32 );
        switch( channelType->dataType() )
        {
          case GTLCore::Type::INTEGER8:
            floatValue = LLVMBackend::CodeGenerator::createAdditionExpression( currentBlock, floatValue, GTLCore::Type::Float32, LLVMBackend::CodeGenerator::floatToConstant( context, 127.0 ), GTLCore::Type::Float32);
          case GTLCore::Type::UNSIGNED_INTEGER8:
            floatValue = LLVMBackend::CodeGenerator::createDivisionExpression( currentBlock, floatValue, GTLCore::Type::Float32, LLVMBackend::CodeGenerator::floatToConstant( context, 255.0 ), GTLCore::Type::Float32);
            break;
          case GTLCore::Type::INTEGER16:
            floatValue = LLVMBackend::CodeGenerator::createAdditionExpression( currentBlock, floatValue, GTLCore::Type::Float32, LLVMBackend::CodeGenerator::floatToConstant( context, 32767.0 ), GTLCore::Type::Float32);
          case GTLCore::Type::UNSIGNED_INTEGER16:
            floatValue = LLVMBackend::CodeGenerator::createDivisionExpression( currentBlock, floatValue, GTLCore::Type::Float32, LLVMBackend::CodeGenerator::floatToConstant( context, 65535.0 ), GTLCore::Type::Float32);
            break;
          case GTLCore::Type::INTEGER32:
            floatValue = LLVMBackend::CodeGenerator::createAdditionExpression( currentBlock, floatValue, GTLCore::Type::Float32, LLVMBackend::CodeGenerator::floatToConstant( context, 2147483647.0 ), GTLCore::Type::Float32);
          case GTLCore::Type::UNSIGNED_INTEGER32:
            floatValue = LLVMBackend::CodeGenerator::createDivisionExpression( currentBlock, floatValue, GTLCore::Type::Float32, LLVMBackend::CodeGenerator::floatToConstant( context, 4294967295.0 ), GTLCore::Type::Float32);
            break;
          default:
            GTL_ABORT("unimplemented");
        }
        
        // result[i] = convert(_imgData + pos(i) )
        if( channelsNb == 1 )
        { // Special case, special hack, for a pixel of one channel, the vec isn't a vec
          floatVec = floatValue;
        } else {
          floatVec = llvm::InsertElementInst::Create( floatVec, floatValue, LLVMBackend::CodeGenerator::integerToConstant( context, gtl_uint32(_pixelDescription.channelPositions()[i])), "", currentBlock);
        }
        GTL_ASSERT(channelType->bitsSize() % 8 == 0);
        currentPos += channelType->bitsSize() / 8;
  //      }
      }
      new llvm::StoreInst( floatVec, arg_dst, "", currentBlock);
  //    }
    }
  //  }
    llvm::ReturnInst::Create( context, currentBlock );
  return func;
}

int vecToMemId = 0;

llvm::Function* CodeGenerator::generateVecToMem( GTLCore::ModuleData* _moduleData, llvm::Module* _module, const GTLCore::PixelDescription& _pixelDescription )
{
  int channelsNb = _pixelDescription.channels();
  //
  // If all channels are float, then do a direct memcpy.
  // Otherwise do a memcpy to a float array and a conversion from it.
  //
  //  template<bool _TAllFloat_, int _TChannels_>
  //  void vecToMem( char* _imgData, vec4* _src, gtl_uint64 _channelFlags)
  //  {
  //    char *dst;
  //    for(int i = 0; i < _TChannels; ++i)
  //    {
  //      gtl_uint64 channelMask = 1 << k;
  //      if( (_channelFlags & channelMask) == channelMask )
  //      {
  //        *(_imgData + pos(i) ) = conver( _src[i] );
  //      }
  //    }
  //  }
  
  llvm::LLVMContext& context = _moduleData->llvmModule()->getContext();
  
  LLVMBackend::CodeGenerator codeGenerator( _moduleData );
  
  llvm::Function* func = codeGenerator.createFunction( _module,
      Wrapper::image_wrap_vec_float_to_mem_type( context, _moduleData->typesManager(), _pixelDescription.channels()) ,
      "image_wrap_vecToMem" + GTLCore::String::number( ++vecToMemId) );
  // Initialise a generation context
  LLVMBackend::GenerationContext generationContext( &codeGenerator, &context, func, 0, _moduleData , _module);
  
  // Get the args.
    llvm::Function::arg_iterator arg_it = func->arg_begin();
    //   char* _imgData = first arg;
    llvm::Value* arg_imgData = arg_it;
    //   vec4* _src = second arg;
    ++arg_it;
    llvm::Value* arg_src = arg_it;
    //   gtl_uint64* _channelFlags = third arg;
    ++arg_it;
    llvm::Value* arg_channelFlags = arg_it;
    
  // {
    llvm::BasicBlock* currentBlock = llvm::BasicBlock::Create(context);
    func->getBasicBlockList().push_back( currentBlock );
    llvm::Value* floatValuePtr = new llvm::AllocaInst(GTLCore::Type::Float32->d->type(context), "", currentBlock);
  //      for(int i = 0; i < _TChannels; ++i)
    gtl_int32 currentPos = 0;
    llvm::Value* floatVec = new llvm::LoadInst( arg_src, "", currentBlock );
    for( int i = 0; i < channelsNb; ++i)
  //      {
    {
      
      llvm::BasicBlock* ifBlock = llvm::BasicBlock::Create(context);
      llvm::BasicBlock* sartIfBlock = ifBlock;
      func->getBasicBlockList().push_back( ifBlock );
      
      const GTLCore::Type* channelType = _pixelDescription.channelTypes()[i];
      // _src[i]
      llvm::Value* floatValue;
      if( channelsNb == 1 )
      {
        floatValue = floatVec;
      } else {
        floatValue = llvm::ExtractElementInst::Create( floatVec, LLVMBackend::CodeGenerator::integerToConstant(context, gtl_uint32(_pixelDescription.channelPositions()[i])), "", ifBlock );
      }
  //        convert( _src[i] );
        // Clamp
        if( channelType->isInteger() )
        {
          new llvm::StoreInst(floatValue, floatValuePtr, "", ifBlock);
          ifBlock = LLVMBackend::CodeGenerator::createClampExpression(generationContext, ifBlock, floatValuePtr, GTLCore::Type::Float32, LLVMBackend::CodeGenerator::floatToConstant(context, 0.0), LLVMBackend::CodeGenerator::floatToConstant(context, 1.0));
          floatValue = new llvm::LoadInst(floatValuePtr, "", ifBlock);
        }
        // Scale
        switch( channelType->dataType() )
        {
          case GTLCore::Type::INTEGER8:
            floatValue = LLVMBackend::CodeGenerator::createSubstractionExpression( ifBlock, floatValue, GTLCore::Type::Float32, LLVMBackend::CodeGenerator::floatToConstant(context, 0.5 ), GTLCore::Type::Float32);
          case GTLCore::Type::UNSIGNED_INTEGER8:
            floatValue = LLVMBackend::CodeGenerator::createMultiplicationExpression( ifBlock, floatValue, GTLCore::Type::Float32, LLVMBackend::CodeGenerator::floatToConstant(context, 255.0 ), GTLCore::Type::Float32);
            break;
          case GTLCore::Type::INTEGER16:
            floatValue = LLVMBackend::CodeGenerator::createSubstractionExpression( ifBlock, floatValue, GTLCore::Type::Float32, LLVMBackend::CodeGenerator::floatToConstant(context, 0.5 ), GTLCore::Type::Float32);
          case GTLCore::Type::UNSIGNED_INTEGER16:
            floatValue = LLVMBackend::CodeGenerator::createMultiplicationExpression( ifBlock, floatValue, GTLCore::Type::Float32, LLVMBackend::CodeGenerator::floatToConstant(context, 65535.0 ), GTLCore::Type::Float32);
            break;
          case GTLCore::Type::INTEGER32:
            floatValue = LLVMBackend::CodeGenerator::createSubstractionExpression( ifBlock, floatValue, GTLCore::Type::Float32, LLVMBackend::CodeGenerator::floatToConstant(context, 0.5 ), GTLCore::Type::Float32);
          case GTLCore::Type::UNSIGNED_INTEGER32:
            floatValue = LLVMBackend::CodeGenerator::createMultiplicationExpression( ifBlock, floatValue, GTLCore::Type::Float32, LLVMBackend::CodeGenerator::floatToConstant(context, 4294967295.0 ), GTLCore::Type::Float32);
            break;
          case GTLCore::Type::FLOAT16:
          case GTLCore::Type::FLOAT32:
            break;
          default:
            GTL_ABORT("unimplemented");
        }
        // Convert back to native
        llvm::Value* nativeValue = LLVMBackend::CodeGenerator::convertValueTo( ifBlock, floatValue, GTLCore::Type::Float32, channelType);
  //        *(_imgData + pos(i) ) = convert( _src[i] );
        llvm::Value* posInP = llvm::GetElementPtrInst::Create( arg_imgData, LLVMBackend::CodeGenerator::integerToConstant( context, currentPos ), "", ifBlock );
        llvm::Value* posInPNative = LLVMBackend::CodeGenerator::convertPointerTo( ifBlock, posInP, channelType->d->type(context) );
        new llvm::StoreInst( nativeValue, posInPNative, "", ifBlock );
  //      }
      // result[i] = convert(_imgData + pos(i) )
      GTL_ASSERT(channelType->bitsSize() % 8 == 0);
      currentPos += channelType->bitsSize() / 8;
  //      }
  
  // Test if the channel is locked
  
  //      gtl_uint64 channelMask = 1 << k;
      gtl_uint64 channelMask = gtl_uint64(1) << i;
      llvm::Value* channelMaskValue = LLVMBackend::CodeGenerator::valueToConstant(generationContext, channelMask );
  // (_channelFlags & channelMask)
      llvm::Value* lhs = LLVMBackend::CodeGenerator::createBitAndExpression(currentBlock,
                                arg_channelFlags, GTLCore::Type::UnsignedInteger64,
                                channelMaskValue, GTLCore::Type::UnsignedInteger64 );
      
  //  (_channelFlags & channelMask) == channelMask
      llvm::Value* comp = LLVMBackend::CodeGenerator::createEqualExpression(currentBlock,
                                lhs, GTLCore::Type::UnsignedInteger64,
                                channelMaskValue, GTLCore::Type::UnsignedInteger64);
      
  //      if( (_channelFlags & channelMask) == channelMask )
      llvm::BasicBlock* nextCurrentBlock = llvm::BasicBlock::Create(context);
      func->getBasicBlockList().push_back( nextCurrentBlock );
      
      LLVMBackend::CodeGenerator::createIfStatement(currentBlock, comp, GTLCore::Type::Boolean, sartIfBlock, ifBlock, nextCurrentBlock);
      
      currentBlock = nextCurrentBlock;
    }
  //    }
  //  }
    llvm::ReturnInst::Create( context, currentBlock );
  return func;
}

llvm::Value* CodeGenerator::callVirtualMember( LLVMBackend::GenerationContext& _gc, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, int _member_index , std::vector<llvm::Value*> _arguments)
{
  std::vector<llvm::Value*> indexes;
  indexes.push_back( _gc.codeGenerator()->integerToConstant( _gc.llvmContext(), INT32_C(0)));
  indexes.push_back( _gc.codeGenerator()->integerToConstant( _gc.llvmContext(), gtl_int32(_member_index + STRUCT_FIRST_ELEMENT)));
  GTL_DEBUG( "Call " << _member_index << " on " << *_pointer );
  
  llvm::Value* funcPtr = new llvm::LoadInst( llvm::GetElementPtrInst::Create( _pointer, indexes, "", _currentBlock ), "" , _currentBlock);
  GTL_DEBUG( *funcPtr );
  GTL_COMPARE_FUNCTION_PARAMETERS( funcPtr, _arguments );
  return llvm::CallInst::Create( funcPtr, _arguments, "", _currentBlock );
}

llvm::Value* CodeGenerator::callImageWrapData( LLVMBackend::GenerationContext& _gc, llvm::BasicBlock* _currentBlock, const GTLCore::Type* _imageType, llvm::Value* _imageWrap, llvm::Value* _x, llvm::Value* _y, bool callConst )
{
  GTL_DEBUG( "CodeGenerator::callImageWrapData " << *_imageType << " " << *_imageWrap << " " << *_x << " " << *_y);
  GTL_ASSERT( _imageType );
  std::vector<llvm::Value*> image_wrap_data_params;
  image_wrap_data_params.push_back( _imageWrap );
  image_wrap_data_params.push_back( _x );
  image_wrap_data_params.push_back( _y );
  
  llvm::Function* func = 0;
  
  if(callConst)
  {
    func = Wrapper::image_wrap_const_dataFunction( _gc.llvmModule(), _imageType );
  } else {
    func = Wrapper::image_wrap_dataFunction( _gc.llvmModule(), _imageType );
  }
  GTL_ASSERT(func);
  GTL_COMPARE_FUNCTION_PARAMETERS( func, image_wrap_data_params );
  
  return llvm::CallInst::Create(
      func, image_wrap_data_params, "", _currentBlock );
}

llvm::Value* CodeGenerator::callImageAlphaPos( LLVMBackend::GenerationContext& _gc, llvm::BasicBlock* _currentBlock, const GTLCore::Type* _imageType, llvm::Value* _imageWrap )
{
  GTL_DEBUG( "CodeGenerator::callImageAlphaPos " << *_imageType << " " << *_imageWrap );
  GTL_ASSERT( _imageType );
  std::vector<llvm::Value*> image_wrap_data_params;
  image_wrap_data_params.push_back( _imageWrap );
  
  llvm::Function* func = Wrapper::image_alpha_posFunction( _gc.llvmModule(), _imageType );
  GTL_COMPARE_FUNCTION_PARAMETERS( func, image_wrap_data_params );
  
  return llvm::CallInst::Create(
      func, image_wrap_data_params, "", _currentBlock );
}

llvm::Value* CodeGenerator::callImageColorConverter( LLVMBackend::GenerationContext& _gc, llvm::BasicBlock* _currentBlock, const GTLCore::Type* _imageType, llvm::Value* _imageWrap )
{
  GTL_ASSERT(_imageType);
  std::vector<llvm::Value*> params;
  params.push_back(_imageWrap);
  
  llvm::Function* func = Wrapper::image_color_converter( _gc.llvmModule(), _imageType );
  GTL_COMPARE_FUNCTION_PARAMETERS( func, params );
  
  return llvm::CallInst::Create(
      func, params, "", _currentBlock );  
}

int imageSampleNearestId = 0;

llvm::Function* CodeGenerator::generateImageSampleNearest(GTLCore::ModuleData* _moduleData, llvm::Module* _module, const GTLCore::Type* _imageType, const GTLCore::Type* _pixelType )
{
  //  pixelX* sampleNearest( ImageWrap* self, float2 pt )
  //  {
  //    int x = pt[0];
  //    int y = pt[1];
  //    pixelX* px = new pixelX;
  //    memToPixel( image, px, image_wrap_data(self, x, y ) );
  //    px.x = pt[0];
  //    px.y = pt[1];
  //    return pixelX;
  //  }
  
  llvm::LLVMContext& context = _moduleData->llvmModule()->getContext();
  
  LLVMBackend::CodeGenerator codeGenerator( _moduleData );
    
  llvm::Function* func = codeGenerator.createFunction( _module,
      Wrapper::image_wrap_sample_nearest_type( context, _moduleData->typesManager(), _imageType, _pixelType),
      "image_wrap_sample_nearest" + GTLCore::String::number( ++imageSampleNearestId) );
  LLVMBackend::GenerationContext generationContext( &codeGenerator, &context, func, 0, _moduleData , _module);
  // Get the args.
    llvm::Function::arg_iterator arg_it = func->arg_begin();
    //   ImageWrap* self = first arg;
    llvm::Value* arg_self = arg_it;
    //   float2 _pt = second arg;
    ++arg_it;
    llvm::Value* arg_pt = arg_it;
  //  {
    llvm::BasicBlock* currentBlock = llvm::BasicBlock::Create(context);
    func->getBasicBlockList().push_back( currentBlock );
  //    int x = pt[0];
    llvm::Value* x_f = LLVMBackend::CodeGenerator::vectorValueAt( currentBlock, arg_pt,
                          LLVMBackend::CodeGenerator::integerToConstant( context, INT32_C(0) ) );
    llvm::Value* x_i = LLVMBackend::CodeGenerator::convertValueTo( currentBlock, x_f, GTLCore::Type::Float32, GTLCore::Type::Integer32 );
    llvm::Value* y_f = LLVMBackend::CodeGenerator::vectorValueAt( currentBlock, arg_pt,
                          LLVMBackend::CodeGenerator::integerToConstant( context, INT32_C(1) ) );
    llvm::Value* y_i = LLVMBackend::CodeGenerator::convertValueTo( currentBlock, y_f, GTLCore::Type::Float32, GTLCore::Type::Integer32 );
  //    pixelX* px = new pixelX;
    llvm::Value* px_var = LLVMBackend::CodeGenerator::allocateMemory( generationContext, _pixelType->d->type(context), LLVMBackend::CodeGenerator::integerToConstant(context, INT32_C(1) ), currentBlock );
    LLVMBackend::CodeGenerator::setCountFieldOf( currentBlock, px_var, LLVMBackend::CodeGenerator::integerToConstant( context, INT32_C(0) ) );
  //    memToPixel( image, px, image_wrap_data(self, x, y ) );
    currentBlock = memToPixel( generationContext, currentBlock,
                               callImageWrapData( generationContext, currentBlock, _imageType, arg_self, x_i, y_i, true ),
                               px_var, _imageType, arg_self );
    setPixelCoordinates( generationContext, currentBlock, px_var, x_f, GTLCore::Type::Float32, y_f, GTLCore::Type::Float32 );
    setPixelAlphaPos( generationContext, currentBlock, px_var, arg_self, _imageType );
    llvm::ReturnInst::Create( context, px_var, currentBlock);
  return func;
}

llvm::Function* CodeGenerator::generatePixelSetAlpha( GTLCore::ModuleData* _moduleData, llvm::Module* _module, const GTLCore::Type* _pixelType, int channelsNb)
{
  // void setAlpha(PixelWrap* self, float alpha)
  // {
  //   int alphaPos = self->alphaPos;
  //   if(alphaPos != -1 ) self->data[alphaPos] = alpha;
  //   return ;
  // }
  
  llvm::LLVMContext& context = _moduleData->llvmModule()->getContext();
  
  LLVMBackend::CodeGenerator codeGenerator( _moduleData );
    
  llvm::Function* func = codeGenerator.createFunction( _module,
      Wrapper::pixel_wrap_set_alpha_type( context, _moduleData->typesManager(), _pixelType),
      "pixel_wrap_set_alpha" + GTLCore::String::number( ++imageSampleNearestId) );
  LLVMBackend::GenerationContext generationContext( &codeGenerator, &context, func, 0, _moduleData, _module );
  // Get the args.
    llvm::Function::arg_iterator arg_it = func->arg_begin();
    //   PixelWrap* self = first arg;
    llvm::Value* arg_self = arg_it;
    //   float alpha = second arg;
    ++arg_it;
    llvm::Value* arg_alpha = arg_it;
  //  {
    llvm::BasicBlock* firstBlock = llvm::BasicBlock::Create(context);
    func->getBasicBlockList().push_back( firstBlock );
    
  //   int alphaPos = self->alphaPos;
    llvm::Value* alphaPos = new llvm::LoadInst( accessPixelAlphaPos( generationContext, firstBlock, arg_self), "", firstBlock);

  // alphaPos != -1
    llvm::Value* compar = LLVMBackend::CodeGenerator::createDifferentExpression(firstBlock, alphaPos, GTLCore::Type::Integer32, LLVMBackend::CodeGenerator::integerToConstant(context, INT32_C(-1)), GTLCore::Type::Integer32 );
  
  // self->data[alphaPos] = alpha;
    llvm::BasicBlock* alphaPosBlock = llvm::BasicBlock::Create(context);
    func->getBasicBlockList().push_back( alphaPosBlock );
    // self->data
    llvm::Value* pixelDataPtr = accessPixelDataPtr( generationContext, alphaPosBlock, arg_self );
    // self->data[alphaPos]
    if( channelsNb == 1 ) {
      new llvm::StoreInst( arg_alpha, pixelDataPtr, "", alphaPosBlock );
    } else {
      llvm::Value* floatVec = new llvm::LoadInst( pixelDataPtr, "", alphaPosBlock );
      floatVec = llvm::InsertElementInst::Create( floatVec, arg_alpha, alphaPos, "", alphaPosBlock);
      new llvm::StoreInst( floatVec, pixelDataPtr, "", alphaPosBlock );
    }
  
  // return;
    llvm::BasicBlock* returnBlock = llvm::BasicBlock::Create(context);
    func->getBasicBlockList().push_back( returnBlock );
    llvm::ReturnInst::Create(context, returnBlock);
    
    LLVMBackend::CodeGenerator::createIfStatement(firstBlock, compar, GTLCore::Type::Boolean, alphaPosBlock, alphaPosBlock, returnBlock);
    
  return func;
}
llvm::Function* CodeGenerator::generatePixelAlpha( GTLCore::ModuleData* _moduleData, llvm::Module* _module, const GTLCore::Type* _pixelType, int channelsNb)
{
  // float alpha(PixelWrap* self)
  // {
  //   int alphaPos = self->alphaPos;
  //   if(alphaPos == -1 ) return 1.0;
  //   return self->data[alphaPos];
  // }
  
  llvm::LLVMContext& context = _moduleData->llvmModule()->getContext();
  
  LLVMBackend::CodeGenerator codeGenerator( _moduleData );
    
  llvm::Function* func = codeGenerator.createFunction( _module,
      Wrapper::pixel_wrap_alpha_type( context, _moduleData->typesManager(), _pixelType),
      "pixel_wrap_alpha" + GTLCore::String::number( ++imageSampleNearestId) );
  LLVMBackend::GenerationContext generationContext( &codeGenerator, &context, func, 0, _moduleData, _module );

  // Get the args.
    llvm::Function::arg_iterator arg_it = func->arg_begin();
    //   PixelWrap* self = first arg;
    llvm::Value* arg_self = arg_it;
  
  //  {
    llvm::BasicBlock* firstBlock = llvm::BasicBlock::Create(context);
    func->getBasicBlockList().push_back( firstBlock );
    
  //   int alphaPos = self->alphaPos;
    llvm::Value* alphaPos = new llvm::LoadInst( accessPixelAlphaPos( generationContext, firstBlock, arg_self), "read alpha pos", firstBlock);

  // alphaPos == -1
    llvm::Value* compar = LLVMBackend::CodeGenerator::createEqualExpression(firstBlock, alphaPos, GTLCore::Type::Integer32, LLVMBackend::CodeGenerator::integerToConstant(context, INT32_C(-1)), GTLCore::Type::Integer32 );
  // return 1.0
    llvm::BasicBlock* noAlphaPosBlock = llvm::BasicBlock::Create(context);
    func->getBasicBlockList().push_back( noAlphaPosBlock );
    llvm::ReturnInst::Create( context, LLVMBackend::CodeGenerator::floatToConstant(context, 1.0), noAlphaPosBlock );
  // return self->data[alphaPos];
    llvm::BasicBlock* alphaPosBlock = llvm::BasicBlock::Create(context);
    func->getBasicBlockList().push_back( alphaPosBlock );
    // self->data
    llvm::Value* pixelDataPtr = accessPixelDataPtr( generationContext, alphaPosBlock, arg_self );
    // self->data[alphaPos]
    llvm::Value* floatVec = new llvm::LoadInst( pixelDataPtr, "", alphaPosBlock );
    llvm::Value* floatValue = 0;
    if( channelsNb == 1 ) {
      floatValue = floatVec;
    } else {
      floatValue = llvm::ExtractElementInst::Create( floatVec, alphaPos, "", alphaPosBlock );
    }
    llvm::ReturnInst::Create( context, floatValue, alphaPosBlock );
    
    LLVMBackend::CodeGenerator::createIfStatement(firstBlock, compar, GTLCore::Type::Boolean, noAlphaPosBlock, noAlphaPosBlock, alphaPosBlock);
  return func;
}

llvm::Function* CodeGenerator::createWrapConvertColorToPixelFunction( LLVMBackend::GenerationContext& _gc, llvm::Type* _pixelType )
{
  std::vector<llvm::Type*> tyArgs;
  tyArgs.push_back( GTLCore::Type::Pointer->d->type(_gc.llvmContext()) );
  tyArgs.push_back( GTLCore::Type::Float32->d->pointerType(_gc.llvmContext()) );
  tyArgs.push_back( _pixelType );
  tyArgs.push_back( GTLCore::Type::Integer32->d->type(_gc.llvmContext()) );
    
  llvm::FunctionType* ty = llvm::FunctionType::get( llvm::Type::getVoidTy(_gc.llvmContext()), tyArgs, false);
  
  llvm::Function* func = (llvm::Function*)_gc.llvmModule()->getOrInsertFunction(
      "wrapConvertColorToPixel", ty); // (external, no body)
  func->setCallingConv(llvm::CallingConv::C);
  return func;
}

llvm::Function* CodeGenerator::createWrapConvertPixelToColorFunction( LLVMBackend::GenerationContext& _gc, llvm::Type* _pixelType )
{
  std::vector<llvm::Type*> tyArgs;
  tyArgs.push_back( GTLCore::Type::Pointer->d->type(_gc.llvmContext()) );
  tyArgs.push_back( _pixelType );
  tyArgs.push_back( GTLCore::Type::Float32->d->pointerType(_gc.llvmContext()) );
  tyArgs.push_back( GTLCore::Type::Integer32->d->type(_gc.llvmContext()) );
    
  llvm::FunctionType* ty = llvm::FunctionType::get( llvm::Type::getVoidTy(_gc.llvmContext()), tyArgs, false);
  
  llvm::Function* func = (llvm::Function*)_gc.llvmModule()->getOrInsertFunction(
      "wrapConvertPixelToColor", ty); // (external, no body)
  func->setCallingConv(llvm::CallingConv::C);
  return func;
}
