/*                                                                            */
/* CDDL HEADER START                                                          */
/*                                                                            */
/* The contents of this file are subject to the terms of the Common           */
/* Development and Distribution License Version 1.0 (the "License").          */
/*                                                                            */
/* You can obtain a copy of the license at                                    */
/* http://www.opensource.org/licenses/CDDL-1.0.  See the License for the      */
/* specific language governing permissions and limitations under the License. */
/*                                                                            */
/* When distributing Covered Code, include this CDDL HEADER in each file and  */
/* include the License file in a prominent location with the name             */
/* LICENSE.CDDL.  If applicable, add the following below this CDDL HEADER,    */
/* with the fields enclosed by brackets "[]" replaced with your own           */
/* identifying information:                                                   */
/*                                                                            */
/* Portions Copyright (c) [yyyy] [name of copyright owner].                   */
/* All rights reserved.                                                       */
/*                                                                            */
/* CDDL HEADER END                                                            */
/*                                                                            */

/*                                                                            */
/* <REPLACE: Change copyright information as needed>                          */
/* Copyright (c) 2019, Regents of the University of Minnesota.                */
/* All rights reserved.                                                       */
/* </REPLACE>                                                                 */
/*                                                                            */
/* <REPLACE: Change contributors as needed>                                   */
/* Contributors:                                                              */
/*    Daniel S. Karls                                                         */
/*    Ellad B. Tadmor                                                         */
/* </REPLACE>                                                                 */
/*                                                                            */

#include <math.h>

/* <REPLACE: Change following text as needed.>                                */
/*                                                                            */
/* Model driver for the Stillinger-Weber (SW) potential                       */
/*                                                                            */
/* Source:                                                                    */
/* Frank H. Stillinger and Thomas A. Weber,                                   */
/* "Computer simulations of local order in condensed phases of silicon",      */
/* Physical Review B, Vol. 31, 5262-5271, 1985.                               */
/*                                                                            */
/* </REPLACE>                                                                 */

/* <REPLACE: Remove this line if no auxiliary functions are read in.>         */
#include "cluster_aux.inc"
/* </REPLACE>                                                                 */

/* The following flag indicates whether or not to do an explicit check to
 * determine if the distance between neighbors j and k of atom i in a
 * three-body computations exceeds the (single) cutoff distance.  The check
 * is skipped if the flag is set to 1, and performed if the flag is set to 0.
 * This feature is provided for efficiency reasons. */
/*                                                                            */
/* <REPLACE: Change RJK check flag as needed.>                                */
#define SKIP_CHECK_ON_RJK 0
/* </REPLACE>                                                                 */

/*                                                                            */
/* IMPORTANT: The driver which accompanies this file, ThreeBodyCluster.c,
 *            assumes that your potential uses distance units of Angstroms and
 *            energy units of eV.  If this is not true for your potential, you
 *            need to edit the call to KIM_ModelDriverCreate_SetUnits in the
 *            driver.  */

/* The following enumeration provides a named index to each element in your
 * parameter array.  The last entry *must* be NUM_PARAMS (by construction by
 * being at the end of the list, its value will be the number of parameters).
 * These names will be * used in the functions below to unpack and access
 * parameters.  The order of parameters *is* important.  It must match the
 * ordering in the parameter file read in by the model driver. */
/*                                                                            */
/* <REPLACE: Change the PARAM_* definitions as needed.  Leave NUM_PARAMS.>    */
enum PARAMS {
  PARAM_A,
  PARAM_B,
  PARAM_C,
  PARAM_p,
  PARAM_a,
  PARAM_costheta0,
  PARAM_lambda,
  PARAM_sigma,
  PARAM_epsilon,
  PARAM_gamma,
  NUM_PARAMS
};
/* </REPLACE>                                                                 */

/* The following array of strings contains the names of the parameters that
 * will be registered in the KIM API object. The number of entries must match
 * the number of PARAM_* lines above, and the order must correspond to the
 * ordering above. */
/*                                                                            */
/* <REPLACE: Change the number of entries and strings as needed.>             */
char * PARAM_NAMES[] = {
  "A",
  "B",
  "C",
  "p",
  "a",
  "costheta0",
  "lambda",
  "sigma",
  "epsilon",
  "gamma"
};
/* </REPLACE>                                                                 */

/* The following array of strings contains descriptions of the parameters that
 * will be registered in the KIM API object. The number of entries must match
 * the number of PARAM_* lines above, and the order must correspond to the
 * ordering above. */
/*                                                                            */
/* <REPLACE: Change the number of entries and strings as needed.>             */
char * PARAM_DESCRIPTIONS[] = {
  "Amplitude of the pairwise interaction function (A).",
  "Prefactor of r^-p in pairwise term (B).",
  "Description of parameter C",
  "Exponent in the r^-p term in the pairwise interaction (p)",
  "Cutoff distance appearing in the exponential cutoff function (a)",
  "Cosine of the equilibrium (tetrahedral) angle in the diamond structure (costheta0)",
  "Amplitude of the Stillinger-Weber h-function appearing int the three-body term (lambda)",
  "Length scale (sigma)",
  "Energy scale (epsilon)",
  "Description of parameter gamma"
};
/* </REPLACE>                                                                 */

static double get_influence_distance(double const * params)
{
  const double A = params[PARAM_A];

  return A;
}

/* Define functions used in two-body calculations */
static double fc2(double const * params, double R)
{
  /* Unpack parameters */
  const double a = params[PARAM_a];
  const double sigma = params[PARAM_sigma];

  if (R < a) { return exp(sigma / (R - a) + (sigma / a)); }
  else
  {
    return 0.0;
  }
}

static void
fc2_dfc2(double const * params, double R, double * fc2, double * dfc2_dR)
{
  /* Unpack parameters */
  const double a = params[PARAM_a];
  const double sigma = params[PARAM_sigma];

  if (R < a)
  {
    *fc2 = exp(sigma / (R - a) + (sigma / a));
    *dfc2_dR = -sigma * *fc2 / ((R - a) * (R - a));
  }
  else
  {
    *fc2 = 0.0;
    *dfc2_dR = 0.0;
  }
}

/* Calculate two-body energy phi2(r) and its derivative w.r.t. Rij */
static void calc_phi2_dphi2(double const * params,
                            double const Rij,
                            double * phi2,
                            double * dphi2_dRij)
{
  /* Unpack parameters */
  const double A = params[PARAM_A];
  const double B = params[PARAM_B];
  const double p = params[PARAM_p];
  const double sigma = params[PARAM_sigma];
  const double epsilon = params[PARAM_epsilon];

  double tmp;
  double fc2val;
  double dfc2_dRij;

  fc2_dfc2(params, Rij, &fc2val, &dfc2_dRij);
  tmp = pow(sigma / Rij, p);
  *phi2 = epsilon * A * (B * tmp - 1) * fc2(params, Rij);

  if (dphi2_dRij != NULL)
  {
    *dphi2_dRij
        = epsilon * A * (B * tmp * (-p * fc2val / Rij + dfc2_dRij) - dfc2_dRij);
  }

  return;
}


/* Define functions used in three-body calculations */
static double g(double const * params, double theta)
{
  /* Unpack parameters */
  const double costheta0 = params[PARAM_costheta0];

  double costheta;

  costheta = cos(theta);
  return (costheta - costheta0) * (costheta - costheta0);
}

static double dg_dcostheta(double const * params, double theta)
{
  /* Unpack parameters */
  const double costheta0 = params[PARAM_costheta0];

  double costheta;

  costheta = cos(theta);
  return 2 * (costheta - costheta0);
}

static double fc3(double const * params, double R)
{
  /* Unpack parameters */
  const double a = params[PARAM_a];
  const double gamma = params[PARAM_gamma];

  if (R < a) { return exp(gamma / (R - a) + (gamma / a)); }
  else
  {
    return 0.0;
  }
}

static void
fc3_dfc3(double const * params, double R, double * fc3, double * dfc3_dR)
{
  /* Unpack parameters */
  const double a = params[PARAM_a];
  const double gamma = params[PARAM_gamma];

  if (R < a)
  {
    *fc3 = exp(gamma / (R - a) + (gamma / a));
    *dfc3_dR = -gamma * *fc3 / ((R - a) * (R - a));
  }
  else
  {
    *fc3 = 0.0;
    *dfc3_dR = 0.0;
  }
}

static double h(double const * params, double r, double s, double theta)
{
  return fc3(params, r) * fc3(params, s) * g(params, theta);
}

static void dh_drdsdtheta(double const * params,
                          double r,
                          double s,
                          double theta,
                          double * dh_dr,
                          double * dh_ds,
                          double * dh_dtheta)
{
  double fc3_r;
  double fc3_s;
  double dfc3_dr;
  double dfc3_ds;
  double costheta;
  double t;
  double gval;
  double dgdcos;

  double dcostheta_dr;
  double dcostheta_ds;
  double dcostheta_dt;

  /* Compute third distance using law of cosines */
  costheta = cos(theta);
  t = sqrt(r * r + s * s - 2 * r * s * costheta);

  fc3_dfc3(params, r, &fc3_r, &dfc3_dr);
  fc3_dfc3(params, s, &fc3_s, &dfc3_ds);

  dcostheta_dr = (r * r - s * s + t * t) / (2 * r * r * s);
  dcostheta_ds = (s * s - r * r + t * t) / (2 * r * s * s);
  dcostheta_dt = -t / (r * s);

  gval = g(params, theta);
  dgdcos = dg_dcostheta(params, theta);

  *dh_dr = fc3_s * (fc3_r * dgdcos * dcostheta_dr + dfc3_dr * gval);
  *dh_ds = fc3_r * (fc3_s * dgdcos * dcostheta_ds + dfc3_ds * gval);
  *dh_dtheta = fc3_r * fc3_s * dgdcos * dcostheta_dt;
}

/* Calculate three-body energy phi3(Rij, Rik, Rjk) and its derivatives w.r.t
 * Rij, Rik, and Rjk */
static void calc_phi3_dphi3(double const * params,
                            double const Rij,
                            double const Rik,
                            double const Rjk,
                            double * phi3,
                            double * dphi3_dRij,
                            double * dphi3_dRik,
                            double * dphi3_dRjk)
{
  /* Unpack parameters */
  const double C = params[PARAM_C];
  const double lambda = params[PARAM_lambda];
  const double epsilon = params[PARAM_epsilon];

  double theta_jik;
  double theta_ijk;
  double theta_ikj;

  double dh1_dRij;
  double dh1_dRik;
  double dh1_dRjk;

  double dh2_dRij;
  double dh2_dRik;
  double dh2_dRjk;

  double dh3_dRij;
  double dh3_dRik;
  double dh3_dRjk;

  /* Law of cosines to get angles from distances */
  theta_jik = acos((Rij * Rij + Rik * Rik - Rjk * Rjk) / (2.0 * Rij * Rik));
  theta_ijk = acos((Rij * Rij + Rjk * Rjk - Rik * Rik) / (2.0 * Rij * Rjk));
  theta_ikj = acos((Rik * Rik + Rjk * Rjk - Rij * Rij) / (2.0 * Rik * Rjk));
  *phi3 = h(params, Rij, Rik, theta_jik) + h(params, Rij, Rjk, theta_ijk)
          + h(params, Rik, Rjk, theta_ikj);
  *phi3 *= epsilon * C * lambda;

  if (dphi3_dRij != NULL)
  {
    dh_drdsdtheta(params, Rij, Rik, theta_jik, &dh1_dRij, &dh1_dRik, &dh1_dRjk);
    dh_drdsdtheta(params, Rij, Rjk, theta_ijk, &dh2_dRij, &dh2_dRjk, &dh2_dRik);
    dh_drdsdtheta(params, Rik, Rjk, theta_ikj, &dh3_dRik, &dh3_dRjk, &dh3_dRij);

    *dphi3_dRij = dh1_dRij + dh2_dRij + dh3_dRij;
    *dphi3_dRik = dh1_dRik + dh2_dRik + dh3_dRik;
    *dphi3_dRjk = dh1_dRjk + dh2_dRjk + dh3_dRjk;

    *dphi3_dRij *= epsilon * C * lambda;
    *dphi3_dRik *= epsilon * C * lambda;
    *dphi3_dRjk *= epsilon * C * lambda;
  }

  return;
}
