///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/Viewport.h>
#include <core/gui/properties/FloatControllerUI.h>
#include "BoxObject.h"
#include "BoxCreationMode.h"

namespace StdObjects {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(BoxObject, SimpleGeometryObject)
DEFINE_REFERENCE_FIELD(BoxObject, FloatController, "Width", _width)
DEFINE_REFERENCE_FIELD(BoxObject, FloatController, "Length", _length)
DEFINE_REFERENCE_FIELD(BoxObject, FloatController, "Height", _height)
SET_PROPERTY_FIELD_LABEL(BoxObject, _width, "Width")
SET_PROPERTY_FIELD_LABEL(BoxObject, _length, "Length")
SET_PROPERTY_FIELD_LABEL(BoxObject, _height, "Height")
SET_PROPERTY_FIELD_UNITS(BoxObject, _width, WorldParameterUnit)
SET_PROPERTY_FIELD_UNITS(BoxObject, _length, WorldParameterUnit)
SET_PROPERTY_FIELD_UNITS(BoxObject, _height, WorldParameterUnit)

IMPLEMENT_PLUGIN_CLASS(BoxObjectEditor, PropertiesEditor)
IMPLEMENT_PLUGIN_CLASS(BoxCreationMode, SimpleCreationMode)

/******************************************************************************
* Constructs a new box object.
******************************************************************************/
BoxObject::BoxObject(bool isLoading) : SimpleGeometryObject(isLoading)
{
	INIT_PROPERTY_FIELD(BoxObject, _width);
	INIT_PROPERTY_FIELD(BoxObject, _length);
	INIT_PROPERTY_FIELD(BoxObject, _height);
	if(!isLoading) {
		_width = CONTROLLER_MANAGER.createDefaultController<FloatController>();
		_length = CONTROLLER_MANAGER.createDefaultController<FloatController>();
		_height = CONTROLLER_MANAGER.createDefaultController<FloatController>();
	}
}

/******************************************************************************
* Asks the object for its validity interval at the given time.
******************************************************************************/
TimeInterval BoxObject::objectValidity(TimeTicks time)
{
	TimeInterval interval = TimeForever;
	_width->validityInterval(time, interval);
	_length->validityInterval(time, interval);
	_height->validityInterval(time, interval);
	return interval;
}

/******************************************************************************
* Builds the mesh representation of this geometry object.
******************************************************************************/
void BoxObject::buildMesh(TimeTicks time, TriMesh& mesh, TimeInterval& meshValidity)
{
	// Reset mesh
	meshValidity.setInfinite();

	// Query parameters
	FloatType width=0, length=0, height=0;
	_width->getValue(time, width, meshValidity);
	_length->getValue(time, length, meshValidity);
	_height->getValue(time, height, meshValidity);
	
	width *= 0.5;
	length *= 0.5;

	// Build vertices.
	mesh.setVertexCount(8);
	mesh.setVertex(0, Point3(-width, -length, 0.0));
	mesh.setVertex(1, Point3( width, -length, 0.0));
	mesh.setVertex(2, Point3( width,  length, 0.0));
	mesh.setVertex(3, Point3(-width,  length, 0.0));
	mesh.setVertex(4, Point3(-width, -length, height));
	mesh.setVertex(5, Point3( width, -length, height));
	mesh.setVertex(6, Point3( width,  length, height));
	mesh.setVertex(7, Point3(-width,  length, height));
    
	// Build faces.
	mesh.setFaceCount(12);
	if(height >= 0.0) {
		mesh.face(0).setVertices(0, 1, 5);
		mesh.face(0).setEdgeVisibility(true, true, false);
		mesh.face(1).setVertices(0, 5, 4);
		mesh.face(1).setEdgeVisibility(false, true, true);
		mesh.face(2).setVertices(1, 2, 6);
		mesh.face(2).setEdgeVisibility(true, true, false);
		mesh.face(3).setVertices(1, 6, 5);
		mesh.face(3).setEdgeVisibility(false, true, true);
		mesh.face(4).setVertices(2, 3, 7);
		mesh.face(4).setEdgeVisibility(true, true, false);
		mesh.face(5).setVertices(2, 7, 6);
		mesh.face(5).setEdgeVisibility(false, true, true);
		mesh.face(6).setVertices(3, 0, 4);
		mesh.face(6).setEdgeVisibility(true, true, false);
		mesh.face(7).setVertices(3, 4, 7);
		mesh.face(7).setEdgeVisibility(false, true, true);
		mesh.face(8).setVertices(4, 5, 6);
		mesh.face(8).setEdgeVisibility(true, true, false);
		mesh.face(9).setVertices(4, 6, 7);
		mesh.face(9).setEdgeVisibility(false, true, true);
		mesh.face(10).setVertices(0, 3, 2);
		mesh.face(10).setEdgeVisibility(true, true, false);
		mesh.face(11).setVertices(0, 2, 1);
		mesh.face(11).setEdgeVisibility(false, true, true);
	}
	else {
		// Flip faces on negative height.
		mesh.face(0).setVertices(0, 5, 1);
		mesh.face(0).setEdgeVisibility(false, true, true);
		mesh.face(1).setVertices(0, 4, 5);
		mesh.face(1).setEdgeVisibility(true, true, false);
		mesh.face(2).setVertices(1, 6, 2);
		mesh.face(2).setEdgeVisibility(false, true, true);
		mesh.face(3).setVertices(1, 5, 6);
		mesh.face(3).setEdgeVisibility(true, true, false);
		mesh.face(4).setVertices(2, 7, 3);
		mesh.face(4).setEdgeVisibility(false, true, true);
		mesh.face(5).setVertices(2, 6, 7);
		mesh.face(5).setEdgeVisibility(true, true, false);
		mesh.face(6).setVertices(3, 4, 0);
		mesh.face(6).setEdgeVisibility(false, true, true);
		mesh.face(7).setVertices(3, 7, 4);
		mesh.face(7).setEdgeVisibility(true, true, false);
		mesh.face(8).setVertices(4, 6, 5);
		mesh.face(8).setEdgeVisibility(false, true, true);
		mesh.face(9).setVertices(4, 7, 6);
		mesh.face(9).setEdgeVisibility(true, true, false);
		mesh.face(10).setVertices(0, 2, 3);
		mesh.face(10).setEdgeVisibility(false, true, true);
		mesh.face(11).setVertices(0, 1, 2);
		mesh.face(11).setEdgeVisibility(true, true, false);
	}

	mesh.invalidateVertices();
	mesh.invalidateFaces();
}

/******************************************************************************
* Creates the UI controls for the editor.
******************************************************************************/
void BoxObjectEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	// Create the rollout.
	QWidget* rollout = createRollout(tr("Box"), rolloutParams);
	
	QGridLayout* layout = new QGridLayout(rollout);
	layout->setContentsMargins(4,4,4,4);
	layout->setSpacing(0);
	layout->setColumnStretch(1, 1);

	// Width parameter.
	FloatControllerUI* widthPUI = new FloatControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(BoxObject, _width));
	layout->addWidget(widthPUI->label(), 0, 0);
	layout->addWidget(widthPUI->textBox(), 0, 1);
	layout->addWidget(widthPUI->spinner(), 0, 2);
	widthPUI->setMinValue(0);

	// Length parameter.
	FloatControllerUI* lengthPUI = new FloatControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(BoxObject, _length));
	layout->addWidget(lengthPUI->label(), 1, 0);
	layout->addWidget(lengthPUI->textBox(), 1, 1);
	layout->addWidget(lengthPUI->spinner(), 1, 2);
	lengthPUI->setMinValue(0);

	// Height parameter.
	FloatControllerUI* heightPUI = new FloatControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(BoxObject, _height));
	layout->addWidget(heightPUI->label(), 2, 0);
	layout->addWidget(heightPUI->textBox(), 2, 1);
	layout->addWidget(heightPUI->spinner(), 2, 2);
}

};
