/*
    Tucnak - VHF contest log
    Copyright (C) 2002-2006  Ladislav Vaiz <ok1zia@nagano.cz>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

*/

#include "header.h"
    

#define STARTB 4
#define STOPB 1

struct sconn sconns;
MUTEX_DEFINE(sconns);

void init_sconns(void){
    MUTEX_INIT(sconns);
    init_list(sconns);
}



struct sdev *sd_open_ttys(char saddr, char *filename, int timeout_ms){
    struct sconn *sc;
	struct sdev *sd;
    
/*    dbg("sd_open_ttys(%s)\n", filename);*/
    
    MUTEX_LOCK(sconns);
    foreach(sc, sconns){
        if (sc->type!=CT_TTYS) continue;
        if (strcmp(sc->ttys_filename, filename)!=0) continue;
        /*dbg("found sc=%p\n", sc);*/
    	sc->refcnt++;
        MUTEX_UNLOCK(sconns);
        goto found;
    }
    MUTEX_UNLOCK(sconns);
    /* sconn not found */
    sc=g_new0(struct sconn,1);
    dbg("new sc %p\n",sc);
    sc->refcnt++;
    MUTEX_LOCK(sconns);
    add_to_list(sconns, sc);
    MUTEX_UNLOCK(sconns);
    sc->type=CT_TTYS;
    sc->ttys_filename=g_strdup(filename);
    sc->sprotocol = SPROT_NONE;
    sc->open  = sconn_file_open;
    sc->close2 = sconn_file_close;
    sc->read  = sconn_file_read;
    sc->write = sconn_file_write;

    sc_open_common(sc);
    
found:    
    sd=g_new0(struct sdev,1);
    sc->give_me_chance = 1;
    MUTEX_LOCK(sc->sdevs);
    sc->give_me_chance = 0;
    add_to_list(sc->sdevs, sd);
    MUTEX_UNLOCK(sc->sdevs);

    sd->sconn=sc;
    sd->saddr=saddr;
    sd->timeout_ms = timeout_ms;
    /*dbg("sd=%p refcnt=%d\n",sd,sc->refcnt);*/
    return sd;
}

    
#ifdef HAVE_LIBFTDI
struct sdev *sd_open_ftdi(char saddr, int vid, int pid, char *serial, int timeout_ms){
    struct sconn *sc;
	struct sdev *sd;
    
/*    dbg("sd_open_ttys(%s)\n", filename);*/
    
    MUTEX_LOCK(sconns);
    foreach(sc, sconns){
        if (sc->type!=CT_FTDI) continue;
        if (serial && *serial)
            if (strcmp(sc->ftdi_serial, serial)!=0) continue;
        /*dbg("found sc=%p\n", sc);*/
    	sc->refcnt++;
        MUTEX_UNLOCK(sconns);
        goto found;
    }
    MUTEX_UNLOCK(sconns);
    /* sconn not found */
    sc=g_new0(struct sconn,1);
    dbg("new sc %p\n",sc);
    sc->refcnt++;
    MUTEX_LOCK(sconns);
    add_to_list(sconns, sc);
    MUTEX_UNLOCK(sconns);
    sc->type=CT_FTDI;
    sc->ftdi_vid = vid;
    sc->ftdi_pid = pid;
    sc->ftdi_serial = g_strdup(serial?serial:"");
    sc->sprotocol = SPROT_ROTAR;
    sc->open  = sconn_ftdi_open;
    sc->close2 = sconn_ftdi_close;
    sc->read  = sconn_ftdi_read;
    sc->write = sconn_ftdi_write;

    sc_open_common(sc);
    
found:    
    sd=g_new0(struct sdev,1);
    sc->give_me_chance = 1;
    MUTEX_LOCK(sc->sdevs);
    sc->give_me_chance = 0;
    add_to_list(sc->sdevs, sd);
    MUTEX_UNLOCK(sc->sdevs);

    sd->sconn=sc;
    sd->saddr=saddr;
    sd->timeout_ms = timeout_ms;
    /*dbg("sd=%p refcnt=%d\n",sd,sc->refcnt);*/
    return sd;
}
#endif // HAVE_LIBFTDI
    
    
void sc_open_common(struct sconn *sc){

    MUTEX_INIT(sc->sdevs);
    init_list(sc->sdevs);

    MUTEX_INIT(sc->jobs);
    init_list(sc->jobs);

	sc->thread = g_thread_create(sc_main, (gpointer)sc, TRUE, NULL);
    dbg("sc created thread %p\n", sc->thread);
}

struct sdev *sd_open_udp(char *hostname, int udpport){
    return NULL;    
}

struct sdev *sd_open_tcp(char *hostname, int tcpport){
    
    return NULL;
}

int free_sd(struct sdev *sd){
    struct sconn *sc;
    
/*    dbg(" ------\n");*/
    dbg("free_sd(%p)\n", sd);

    if (!sd) return -1;

    sc=sd->sconn;
   /* dbg("  refcnt=%d\n", sc->refcnt);*/
    sc->give_me_chance = 1;
    MUTEX_LOCK(sc->sdevs);
    sc->give_me_chance = 0;
    del_from_list(sd);
    MUTEX_UNLOCK(sc->sdevs);
    
    sc->refcnt--;

    if (!sc->refcnt){
        
        /*dbg("  freeing sc %p\n", sc);*/
        sc->freeing = 1;
        if (sc->thread){
            sc->thread_break = 1;
            dbg("join sconn...\n");
            g_thread_join(sc->thread);
            dbg("done\n");
            sc->thread = NULL;
        }
        
        sc->give_me_chance = 1;
        MUTEX_LOCK(sc->sdevs);
        sc->give_me_chance = 0;
        del_from_list(sd);
        MUTEX_UNLOCK(sc->sdevs);

        CONDGFREE(sc->ttys_filename);
        CONDGFREE(sc->ip_hostname);
#ifdef HAVE_LIBFTDI
        CONDGFREE(sc->ftdi_serial);
#endif
        MUTEX_LOCK(sconns);
        del_from_list(sc);
        MUTEX_UNLOCK(sconns);
        g_free(sc);
    }else{
        /*dbg("  sc %p already active\n", sc);*/
    }
    g_free(sd);
    return (0);
}

static char sd_chk(unsigned char *s, int len){
	unsigned char chk, *c;
    chk=0;
	for (c=s;len;c++,len--){
		chk^=*c;
	}
	return chk;
}


int sd_prot(struct sconn *sconn, char saddr, char fce, char *data, int *len, int timeout){
	unsigned char rawdata[550];
	int rawlen,written,rawi;
    int ret, i;

    ret = sconn->open(sconn, 0);
    if (ret) return ret;
    
    /* clearing queue, filedescriptor is non-blocking */
/*    for (i=0; i<10; i++){
        if (sconn->read(sconn, rawdata, sizeof(rawdata)-1, 100)<=0) break;}*/
	rawlen=0;
	memset(rawdata, 0xff, STARTB); rawlen+=STARTB;
	rawdata[rawlen++]=0xc5;
	rawdata[rawlen++]=fce&0x7f;
	rawdata[rawlen++]=saddr;
	rawdata[rawlen++]=(unsigned char)*len;
    memcpy(rawdata+rawlen, data, *len); rawlen+=*len;
	rawdata[rawlen]=sd_chk(rawdata+STARTB, rawlen-STARTB);
    rawlen++;
	memset(rawdata+rawlen, 0xff, STOPB); rawlen+=STOPB;
	if (cfg->trace_sdev){
		dbg("\nsconn->write(");
		for (i=0; i<rawlen; i++) dbg("%02x ", (unsigned char)rawdata[i]);
		dbg("\n");
	}
	written=sconn->write(sconn, rawdata, rawlen);
	trace(cfg->trace_sdev, "sd_send: written=%d\n", written);
    if (written<0) return written;

    rawi=0;
    while(1){
        if (rawi>=sizeof(rawdata)-1) return 20;

        
        ret = sconn->read(sconn, rawdata+rawi, sizeof(rawdata)-rawi, timeout);
//        dbg("read=%d\n", ret);
        if (ret<0) return -3;
        if (ret==0) return -4;
        rawi += ret;
		if (cfg->trace_sdev)        
        {
            int j;
            dbg("read=");
            for (j=0; j<rawi; j++) dbg("%02x ", (unsigned char)rawdata[j]);
            dbg("\n");
        }
        
        for (i=0; i<rawi; i++){
            if (rawdata[i]!=0xc5) continue;
#if 0            
            {
                int j;
                dbg("c5 at %d\n", i);
                for (j=0; j<rawi; j++) dbg("%02x ", (unsigned char)rawdata[j]);
                dbg("\n");
            }
#endif            
           // dbg("i+5>rawi %d+5>%d=%d\n", i, rawi, i+5>rawi);
            if (i+5>rawi) goto nextloop;
           // dbg("i+5+rawdata[i+3]>rawi %d+5+%d>%d\n", i, (unsigned char)rawdata[i+3], rawi);
            if (i+5+(unsigned char)rawdata[i+3]>rawi) goto nextloop;
           // dbg("b\n");
            if (sd_chk(rawdata+i, 5+rawdata[i+3]) != 0) return 11;
            if (rawdata[i+1]==0) return 17;
            if (rawdata[i+1]==0x80) return 14;
            if ((rawdata[i+1] & 0x80)==0) continue;
//            dbg("c\n");
            if (rawdata[i+2]!=(unsigned char)saddr) return 16;
//            dbg("d\n");
            if (rawdata[i+1]!=(fce|0x80)) return 16;

            *len = rawdata[i+3];
            memcpy(data, rawdata+i+4, *len);
/*            {
                int j;
                dbg("OK data=\n");
                for (j=0; j<*len; j++) dbg("%02x ", (unsigned char)data[j]);
                dbg("\n");
            }*/
            return 0;
        }
        //dbg("neni C5 0..%d\n", rawi);
nextloop:;        
    }
    

    
    return 0;
}



char *sd_err(int err){
    switch(err){
        case 0:
            return "OK";
        case 11:
            return "Bad checksum";
        case 13:
            return "Timeout";
        case 14:
            return "Error";
        case 16:
            return "Bad response";
        case 17:
            return "Unknown function";
        default:
            return "Unknown error code";
    }

}

int sconn_file_open(struct sconn *sconn, int verbose){
	struct termios tio;
    char s[256], errbuf[256];
    int i, ret;
	
    if (sconn->freeing) return -1;
    if (sconn->opened) return 0;

    ret = fhs_lock(sconn->ttys_filename, 1);
    if (ret){
        fhs_error(s, sizeof(s), ret, sconn->ttys_filename);
        zwrite(tpipe->threadpipe_write, zconcatesc("SC", "!", s, NULL));
        return ret;
    }

	sconn->fd = open(sconn->ttys_filename, O_RDWR | O_SYNC | O_NONBLOCK | O_NOCTTY);
    if (sconn->fd<0){
        g_snprintf(s, sizeof(s), "Can't open device %s", sconn->ttys_filename);
        zwrite(tpipe->threadpipe_write, zconcatesc("SC", "!", s, NULL));
        fhs_unlock(sconn->ttys_filename);

		return -3;
    }
    
    if (fcntl(sconn->fd,F_SETFL,O_NONBLOCK)) {
		g_snprintf(s, sizeof(s), "Can't set O_NONBLOCK on %s: %s",sconn->ttys_filename, strerror_r(errno, errbuf, sizeof(errbuf)) );
		zwrite(tpipe->threadpipe_write, zconcatesc("SC", "!", s, NULL));
        fhs_unlock(sconn->ttys_filename);
        sconn->close2(sconn);
        return -4;
    }

#if 0
    // old buggy code
	tcgetattr(sconn->fd,&tio);
	tio.c_cflag=B9600|CS8|CLOCAL|CREAD|PARENB;/*|PARODD;*/
	tio.c_iflag=INPCK;
	tio.c_lflag=0;
	tio.c_oflag=0;
	tio.c_cc[VMIN]=1;
	tio.c_cc[VTIME]=5;
	tcsetattr(sconn->fd,TCSANOW,&tio);
#else
	tcgetattr(sconn->fd,&tio);
    cfmakeraw(&tio);
    cfsetispeed(&tio, B9600);
    cfsetospeed(&tio, B9600);
    tio.c_cflag |= (CLOCAL | CREAD);
    tio.c_cflag &= ~CSIZE;
    tio.c_cflag |= CS8;
    tio.c_cflag &= ~CSTOPB;
    tio.c_cflag |= PARENB;
    tio.c_cflag &= ~PARODD;
    tio.c_cflag &= ~CRTSCTS;
    tio.c_iflag &= ~IXON;
    tio.c_lflag &= ~(ICANON | ECHO | ECHOE | ISIG);
    tio.c_oflag &= ~OPOST;
    tcflush(sconn->fd, TCIFLUSH);
    tcsetattr(sconn->fd, TCSANOW, &tio);
#endif

    i = TIOCM_RTS; 
    ioctl (sconn->fd, TIOCMBIS, &i); /* set */
    i = TIOCM_DTR; 
    ioctl (sconn->fd, TIOCMBIS, &i); /* set */

/* 
 * normally 0
 * 1 for debugging with atisp
 *
 */

#if 1    
    i = TIOCM_RTS; /* clear PSEN*/
    ioctl (sconn->fd, TIOCMBIC, &i); /* clear = log.1*/
    //usleep(100000);
    i = TIOCM_DTR; /* clear reset*/
    ioctl (sconn->fd, TIOCMBIC, &i); /* clear */
#endif

    sconn->opened = 1;
    return 0;
}

int sconn_file_close(struct sconn *sconn){
    
    //dbg("sconn_file_close %s\n",sconn->ttys_filename); 
    close(sconn->fd);
    fhs_unlock(sconn->ttys_filename);
    sconn->opened = 0;
    return -1;
}

int sconn_file_read(struct sconn *sconn, void *data, size_t len, int timeout_ms){
    int ret;
    fd_set fdr;
    struct timeval tv;
        
    FD_ZERO(&fdr);
    FD_SET(sconn->fd, &fdr);
    tv.tv_usec = timeout_ms * 1000;
    tv.tv_sec = 0;
    
    ret = select(sconn->fd+1, &fdr, NULL, NULL, &tv);
//        dbg("select=%d\n", ret);
    if (ret<0) return -1;
    if (ret==0) return -13;
    if (!FD_ISSET(sconn->fd, &fdr)) return -2;

    ret = read(sconn->fd, data, len);
    if (ret < 0) sconn->close2(sconn);
    return ret;
}

int sconn_file_write(struct sconn *sconn, void *data, size_t len){
    int ret;
    ret = write(sconn->fd, data, len);
    if (ret < 0) sconn->close2(sconn);
    return ret;
}


#ifdef HAVE_LIBFTDI

int sconn_ftdi_open(struct sconn *sconn, int verbose){
    int ret;
    
    if (sconn->freeing) return -1;
    if (sconn->opened) return 0;

    sconn->ftdi = ftdi_new();
    if (!sconn->ftdi){
        if (verbose) log_addf("Can't init ftdi library for sdev");
        return -1;
    }
    if (sconn->ftdi_serial && *sconn->ftdi_serial){
        ret = ftdi_usb_open_desc(sconn->ftdi, sconn->ftdi_vid, sconn->ftdi_pid, NULL, sconn->ftdi_serial);
        if (verbose) dbg("ftdi_usb_open(%04x:%04x, '%s')=%d\n", sconn->ftdi_vid, sconn->ftdi_pid, sconn->ftdi_serial, ret);
    }else{
        ret = ftdi_usb_open(sconn->ftdi, sconn->ftdi_vid, sconn->ftdi_pid);
        if (verbose) dbg("ftdi_usb_open(%04x:%04x)=%d\n", sconn->ftdi_vid, sconn->ftdi_pid, ret);
    }
    if (ret){
        if (verbose) {
            log_addf("Can't open sdev device %04x:%04x, error=%d %s", sconn->ftdi_vid, sconn->ftdi_pid, ret, ftdi_get_error_string(sconn->ftdi));
            if (ret==-8) log_addf("Maybe try to run as root: \"adduser %s dialout\" and relogin", getenv("USER"));
        }
        return -2;

    }
    ret = ftdi_set_baudrate(sconn->ftdi, 9600);
    if (ret){
        if (verbose) log_addf("Can't set baudrate for sconn, error=%d %s", ret, ftdi_get_error_string(sconn->ftdi));
        sconn->close2(sconn);
        return -3;
    }
    ret = ftdi_set_line_property(sconn->ftdi, BITS_8, STOP_BIT_1, EVEN);
    if (ret){
        if (verbose) log_addf("Can't set line properties for sconn, error=%d %s", ret, ftdi_get_error_string(sconn->ftdi));
        sconn->close2(sconn);
        return -4;
    }
    sconn->opened = 1;
    return 0;
}

int sconn_ftdi_close(struct sconn *sconn){
    if (sconn->ftdi!=NULL){
        ftdi_free(sconn->ftdi);
        sconn->ftdi=NULL;
    }
    sconn->opened = 0;
    return 0;
}

int sconn_ftdi_read(struct sconn *sconn, void *data, size_t len, int timeout_ms){
    int ret;
    ttime start, stop;
    
    
    if (!sconn->opened) return -1;
    start = get_time();
    while(1){
        ret = ftdi_read_data(sconn->ftdi, data, len);
        if (ret < 0) {
            sconn->close2(sconn);
            break;
        }
        if (ret) break;
        stop = get_time();
        if (stop < start) break; // time goes back, maybe ntp change
        if (stop - start > timeout_ms) break;
        usleep(1000);
    }
    
//    dbg("sconn_ftdi_read(len=%d)=%d\n", len, ret);
    return ret;
}

int sconn_ftdi_write(struct sconn *sconn, void *data, size_t len){
    int ret;
    
    if (!sconn->opened) return -1;
    ret = ftdi_write_data(sconn->ftdi, data, len);
    if (ret < 0) sconn->close2(sconn);
//    dbg("sconn_ftdi_write(len=%d)=%d\n", len, ret);
    return ret;
}

#endif // HAVE_LIBFTDI


gpointer sc_main(gpointer xxx){
	char data[256];
	int len;
//    struct sslave *sptr; 
    struct sconn *sconn = (struct sconn *)xxx;
         
    dbg("sc_main\n");
        
    sconn->open(sconn, 1);
    
    while(!sconn->thread_break){
        struct sconn_job *job;
        struct sdev *sdev;
        int i, chrapej = 0;

        if (sconn->give_me_chance) {
            usleep(10000);
            continue;
        }
        MUTEX_LOCK(sconn->sdevs);
        job = sconn_job_get(sconn);
        //dbg("job=%p\n", job);
        if (!job){
            i = 0;
            foreach (sdev, sconn->sdevs) i++; // number of sdevs
            if (i==0) {
                error("sc_main: sconn->sdevs is empty");
                sleep(1);
            }
            if (sconn->sdevi >= i) sconn->sdevi = 0;
            i = sconn->sdevi;
            sdev = NULL;
            foreach (sdev, sconn->sdevs) {
                if (!i--) break;
            }
           //  dbg("sdevi=%d sdev=%p main=%p\n", sconn->sdevi, sdev, sdev->sdev_main);
            if (sdev->sdev_main) {
                if (sdev->sdev_main(sdev)) chrapej++;
            }
            sconn->sdevi++;     
            MUTEX_UNLOCK(sconn->sdevs);
            if (chrapej) goto zzzzz;
            continue;
        }
        switch(job->cmd){
            case SCONN_ROT_AZIM:
                data[0] = job->azim & 0xff;
                data[1] = (job->azim >> 8) & 0xff;
                len=2;
				dbg("sd_prot(%d, %d)\n", job->sdev->saddr, job->azim);
                sd_prot(sconn, job->sdev->saddr, 65, data, &len, job->sdev->timeout_ms);
                break;
        }
        MUTEX_UNLOCK(sconn->sdevs);
        continue;
zzzzz:;
        for (i=0; i<20;i++){
            if (sconn->thread_break) break;
            usleep(100000);
        }
    }
    dbg("sc_main exiting\n");
    sconn->close2(sconn);
    return NULL;
}

void sconn_job_add(struct sconn *sconn, struct sconn_job *job){
    MUTEX_LOCK(sconn->jobs);
    add_to_list(sconn->jobs, job);
    MUTEX_UNLOCK(sconn->jobs);
}

struct sconn_job *sconn_job_get(struct sconn *sconn){
    struct sconn_job *job;

    MUTEX_LOCK(sconn->jobs);
    if (list_empty(sconn->jobs)){
        MUTEX_UNLOCK(sconn->jobs);
        return NULL;
    }
    job = sconn->jobs.prev; 
    del_from_list(job);
    MUTEX_UNLOCK(sconn->jobs);
    return job;
}
