/* dircompareop.cc
 * This file belongs to Worker, a file manager for UN*X/X11.
 * Copyright (C) 2017 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "dircompareop.hh"
#include "listermode.h"
#include "worker.h"
#include <aguix/awindow.h>
#include <aguix/text.h>
#include <aguix/button.h>
#include <aguix/cyclebutton.h>
#include <aguix/acontainerbb.h>
#include <aguix/textview.h>
#include "datei.h"
#include "worker_locale.h"
#include "virtualdirmode.hh"
#include "stringcomparator.hh"
#include "dircomparewin.hh"
#include "nwc_path.hh"
#include "commonprefix.hh"

int DirCompareOp::s_vdir_number = 1;

const char *DirCompareOp::name = "DirCompareOp";

DirCompareOp::DirCompareOp() : FunctionProto()
{
    m_category = FunctionProto::CAT_OTHER;

    m_stop = false;
}

DirCompareOp::~DirCompareOp()
{
}

DirCompareOp*
DirCompareOp::duplicate() const
{
    DirCompareOp *ta = new DirCompareOp();
    return ta;
}

bool
DirCompareOp::isName( const char *str )
{
    if ( strcmp( str, name ) == 0 ) return true; else return false;
}

const char *
DirCompareOp::getName()
{
    return name;
}

int DirCompareOp::compareFile( NWC::FSEntry *my_entry,
                               NWC::FSEntry *other_entry )
{
    Datei my_fh;
    Datei other_fh;

    if ( my_fh.open( my_entry->getFullname().c_str(),
                     "r" ) != 0 ) return 1;
    if ( other_fh.open( other_entry->getFullname().c_str(),
                        "r" ) != 0 ) return 1;

    char *my_buf = (char*)_allocsafe( 128 * 1024 );
    char *other_buf = (char*)_allocsafe( 128 * 1024 );

    int res = 0;

    size_t my_read;
    size_t other_read;

    size_t total_bytes = my_entry->stat_size();
    size_t total_bytes_read = 0;

    for (;;) {
        my_read = my_fh.read( my_buf, 128 * 1024 );
        other_read = other_fh.read( other_buf, 128 * 1024 );

        if ( my_read != other_read ) {
            res = 1;
            break;
        }

        if ( my_read < 0 ) {
            res = 1;
            break;
        }

        if ( my_read == 0 ) {
            break;
        }

        if ( memcmp( my_buf, other_buf, my_read ) != 0 ) {
            res = 1;
            break;
        }

        total_bytes_read += my_read;

        if ( total_bytes > 0 ) {
            m_progress_win->setEntryPercent( total_bytes_read * 100 / total_bytes );
        }

        if ( m_progress_win->check() ) {
            m_stop = true;
            res = 1;
            break;
        }
    }
    
    _freesafe( my_buf );
    _freesafe( other_buf );

    return res;
}

int DirCompareOp::compareEntry( NWC::FSEntry *my_entry,
                                NWC::FSEntry *other_entry,
                                std::unique_ptr< NWC::Dir > &my_res_vdir,
                                std::unique_ptr< NWC::Dir > &other_res_vdir )
{
    NWC::Dir *my_dir = dynamic_cast< NWC::Dir * >( my_entry );
    NWC::Dir *other_dir = dynamic_cast< NWC::Dir * >( other_entry );

    m_progress_win->setCurrentEntry( my_entry->getFullname() );
        
    if ( my_dir && other_dir ) {
        auto my_sdir = std::make_shared<NWC::Dir>( *my_dir );
        auto other_sdir = std::make_shared<NWC::Dir>( *other_dir );

        my_sdir->readDir( false );
        other_sdir->readDir( false );

        m_progress_win->setCurrentDir( my_entry->getBasename() );
        
        int res = compareDirs( my_sdir,
                               other_sdir,
                               NULL,
                               NULL,
                               my_res_vdir,
                               other_res_vdir );

        m_progress_win->setCurrentDir( "" );

        if ( m_compare_mode != COMPARE_SIZE_AND_MDATE ) return res;

        if ( res != 0 ) return res;

        // directories seem to be equal but check the modification time
        if ( my_entry->stat_lastmod() != other_entry->stat_lastmod() ) {
            return 1;
        }
        return 0;
    }

    if ( my_dir && ! other_dir ) return 1;
    if ( ! my_dir && other_dir ) return 1;

    // both are not dirs so try to compare

    if ( ( my_entry->stat_mode() & S_IFMT ) != ( other_entry->stat_mode() & S_IFMT ) ) {
        return 1;
    }

    if ( my_entry->isLink() ) {
        std::string my_dest;
        std::string other_dest;

        if ( my_entry->getDestination( my_dest ) == false ) return 1;
        if ( other_entry->getDestination( other_dest ) == false ) return 1;

        if ( my_dest != other_dest ) return 1;

        return 0;
    } else if ( ! my_entry->isReg() ) {
        if ( my_entry->stat_rdev() != other_entry->stat_rdev() ) {
            return 1;
        }

        return 0;
    }

    // regular file

    if ( my_entry->stat_size() != other_entry->stat_size() ) {
        return 1;
    }

    if ( m_compare_mode == COMPARE_CONTENT ) {
        return compareFile( my_entry, other_entry );
    } else if ( m_compare_mode == COMPARE_SIZE_AND_MDATE ) {
        if ( my_entry->stat_lastmod() != other_entry->stat_lastmod() ) {
            return 1;
        }
    }
    
    return 0;
}

int DirCompareOp::compareDirs( std::shared_ptr< NWC::Dir > my_dir,
                               std::shared_ptr< NWC::Dir > other_dir,
                               std::vector< bool > *my_equal,
                               std::vector< bool > *other_equal,
                               std::unique_ptr< NWC::Dir > &my_res_vdir,
                               std::unique_ptr< NWC::Dir > &other_res_vdir )
{
    my_dir->sort( []( const NWC::FSEntry &lhs,
                      const NWC::FSEntry &rhs ) {
                      return StringComparator::compare( lhs.getFullname(),
                                                        rhs.getFullname() ) < 0;
                  } );
    other_dir->sort( []( const NWC::FSEntry &lhs,
                         const NWC::FSEntry &rhs ) {
                         return StringComparator::compare( lhs.getFullname(),
                                                           rhs.getFullname() ) < 0;
                     } );

    int my_pos = 0;
    int other_pos = 0;
    int unequal_count = 0;

    if ( my_equal ) {
        my_equal->resize( my_dir->size() );
    }

    if ( other_equal ) {
        other_equal->resize( other_dir->size() );
    }

    m_progress_win->addEntriesToDo( my_dir->size() );
    m_progress_win->addEntriesToDo( other_dir->size() );

    bool my_is_virtual = false;
    bool other_is_virtual = false;
    std::string my_common_prefix;
    std::string other_common_prefix;

    if ( my_dir->isVirtual() ) {
        my_is_virtual = true;

        CommonPrefix cp;

        for ( my_pos = 0; my_pos < my_dir->size() && m_stop == false; my_pos++ ) {
            cp.updateCommonPrefix( my_dir->getEntryAtPos( my_pos )->getFullname() );
        }

        my_common_prefix = cp.getCommonPrefix();
    }

    if ( other_dir->isVirtual() ) {
        other_is_virtual = true;

        CommonPrefix cp;

        for ( other_pos = 0; other_pos < other_dir->size() && m_stop == false; other_pos++ ) {
            cp.updateCommonPrefix( other_dir->getEntryAtPos( other_pos )->getFullname() );
        }

        other_common_prefix = cp.getCommonPrefix();
    }

    other_pos = 0;

    for ( my_pos = 0; my_pos < my_dir->size() && m_stop == false; my_pos++ ) {
        std::string my_basename;
        std::string other_basename;

        if ( my_is_virtual ) {
            my_basename = NWC::Path::get_extended_basename( my_common_prefix,
                                                            my_dir->getEntryAtPos( my_pos )->getFullname() );
        } else {
            my_basename = my_dir->getEntryAtPos( my_pos )->getBasename();
        }

        if ( other_pos < other_dir->size() ) {
            if ( other_is_virtual ) {
                other_basename = NWC::Path::get_extended_basename( other_common_prefix,
                                                                   other_dir->getEntryAtPos( other_pos )->getFullname() );
            } else {
                other_basename = other_dir->getEntryAtPos( other_pos )->getBasename();
            }
        }
        
        while ( other_pos < other_dir->size() &&
                StringComparator::compare( my_basename,
                                           other_basename ) > 0 ) {
            if ( other_equal ) {
                other_equal->at( other_pos ) = false;
            }
            if ( other_res_vdir ) {
                other_res_vdir->add( *other_dir->getEntryAtPos( other_pos ) );
            }
            other_pos++;
            unequal_count++;

            m_progress_win->finishedEntry();

            if ( other_pos < other_dir->size() ) {
                if ( other_is_virtual ) {
                    other_basename = NWC::Path::get_extended_basename( other_common_prefix,
                                                                        other_dir->getEntryAtPos( other_pos )->getFullname() );
                } else {
                    other_basename = other_dir->getEntryAtPos( other_pos )->getBasename();
                }
            }
        };

        if ( other_pos >= other_dir->size() ) {
            // unequal since there is no corresponding element in the other dir
            if ( my_equal ) {
                my_equal->at( my_pos ) = false;
            }
            unequal_count++;
        } else {
            if ( my_basename == other_basename ) {
                // it's the same entry so check for compare mode
                if ( compareEntry( my_dir->getEntryAtPos( my_pos ),
                                   other_dir->getEntryAtPos( other_pos ),
                                   my_res_vdir,
                                   other_res_vdir ) == 0 ) {
                    if ( my_equal ) {
                        my_equal->at( my_pos ) = true;
                    }
                    if ( other_equal ) {
                        other_equal->at( other_pos ) = true;
                    }
                    if ( my_res_vdir ) {
                        my_res_vdir->add( *my_dir->getEntryAtPos( my_pos ) );
                    }
                } else {
                    if ( my_equal ) {
                        my_equal->at( my_pos ) = false;
                    }
                    if ( other_equal ) {
                        other_equal->at( other_pos ) = false;
                    }
                    if ( other_res_vdir ) {
                        other_res_vdir->add( *other_dir->getEntryAtPos( other_pos ) );
                    }
                }

                other_pos++;

                m_progress_win->finishedEntry();
            } else {
                // the next element in other dir is of a larger name so it's unequal
                if ( my_equal ) {
                    my_equal->at( my_pos ) = false;
                }
                unequal_count++;
            }
        }

        m_progress_win->finishedEntry();

        if ( m_progress_win->check() ) {
            m_stop = true;
        }
    }

    for ( ; other_pos < other_dir->size() && m_stop == false; other_pos++ ) {
        if ( other_equal ) {
            other_equal->at( other_pos ) = false;
        }
        if ( other_res_vdir ) {
            other_res_vdir->add( *other_dir->getEntryAtPos( other_pos ) );
        }
        unequal_count++;

        m_progress_win->finishedEntry();

        if ( m_progress_win->check() ) {
            m_stop = true;
        }
    }

    return unequal_count == 0 ? 0 : 1;
}

int
DirCompareOp::run( WPUContext *wpu, ActionMessage *msg )
{
    if ( msg->mode == msg->AM_MODE_DNDACTION ) {
        return 0;
    }

    if ( doConfigure() != 0 ) return 1;

    VirtualDirMode *this_vdm = nullptr;
    VirtualDirMode *other_vdm = nullptr;

    Lister *l1 = msg->getWorker()->getActiveLister();
    if ( l1 ) {
        ListerMode *lm1 = l1->getActiveMode();
        if ( lm1 ) {
            if ( auto vdm = dynamic_cast< VirtualDirMode *>( lm1 ) ) {
                this_vdm = vdm;
            }
        }

        l1 = msg->getWorker()->getOtherLister( l1 );
        if ( l1 ) {
            ListerMode *lm1 = l1->getActiveMode();
            if ( lm1 ) {
                if ( auto vdm = dynamic_cast< VirtualDirMode *>( lm1 ) ) {
                    other_vdm = vdm;
                }
            }
        }
    }

    m_progress_win = new DirCompareWin( Worker::getAGUIX(),
                                        m_compare_mode == COMPARE_CONTENT );
    m_progress_win->open();

    m_stop = false;
    
    if ( this_vdm && other_vdm ) {
        auto this_dir = this_vdm->getCurrentDir();
        auto other_dir = other_vdm->getCurrentDir();

        if ( this_dir && other_dir ) {
            std::shared_ptr< NWC::Dir > this_sdir( this_dir.release() );
            std::shared_ptr< NWC::Dir > other_sdir( other_dir.release() );

            std::vector< bool > this_equal;
            std::vector< bool > other_equal;

            std::unique_ptr< NWC::Dir > this_res_dir;
            std::unique_ptr< NWC::Dir > other_res_dir;

            if ( m_result_mode == SHOW_AS_VIRTUAL_DIR ) {
                std::string name = AGUIXUtils::formatStringToString( "dircompare%d", s_vdir_number++ );

                if ( s_vdir_number < 0 ) {
                    // avoid negative and zero number
                    s_vdir_number = 1;
                }

                this_res_dir = std::unique_ptr< NWC::Dir >( new NWC::VirtualDir( name ) );

                name = AGUIXUtils::formatStringToString( "dircompare%d", s_vdir_number++ );

                if ( s_vdir_number < 0 ) {
                    // avoid negative and zero number
                    s_vdir_number = 1;
                }

                other_res_dir = std::unique_ptr< NWC::Dir >( new NWC::VirtualDir( name ) );
            }

            compareDirs( this_sdir,
                         other_sdir,
                         &this_equal,
                         &other_equal,
                         this_res_dir,
                         other_res_dir );

            if ( ! m_stop ) {
                if ( m_result_mode == SHOW_AS_VIRTUAL_DIR ) {
                    this_vdm->showDir( this_res_dir );
                    other_vdm->showDir( other_res_dir );
                } else {
                    for ( int this_pos = 0; this_pos < this_sdir->size(); this_pos++ ) {
                        this_vdm->setEntrySelectionState( this_sdir->getEntryAtPos( this_pos )->getFullname(),
                                                          this_equal.at( this_pos ) );
                    }

                    for ( int other_pos = 0; other_pos < other_sdir->size(); other_pos++ ) {
                        other_vdm->setEntrySelectionState( other_sdir->getEntryAtPos( other_pos )->getFullname(),
                                                           ! other_equal.at( other_pos ) );
                    }
                }
            }
        }
    }

    delete m_progress_win;

    m_progress_win = NULL;

    return 0;
}

const char *
DirCompareOp::getDescription()
{
    return catalog.getLocaleCom( 66 );
}

int
DirCompareOp::doConfigure()
{
    AGUIX *aguix = Worker::getAGUIX();
    AGMessage *msg;
    int endmode = -1;

    auto title = AGUIXUtils::formatStringToString( catalog.getLocale( 293 ),
                                                   getDescription() );
    AWindow *win = new AWindow( aguix, 10, 10, 10, 10, title.c_str(), AWindow::AWINDOW_DIALOG );
    win->create();
    
    AContainer *ac0 = win->setContainer( new AContainer( win, 1, 4 ), true );
    ac0->setMinSpace( 5 );
    ac0->setMaxSpace( 5 );
    ac0->setBorderWidth( 5 );

    RefCount<AFontWidth> lencalc( new AFontWidth( aguix, NULL ) );
    TextStorageString help_ts( catalog.getLocale( 1180 ), lencalc );
    TextView *help_tv = (TextView*)ac0->add( new TextView( aguix,
                                                           0, 0, 400, 50, "", help_ts ),
                                             0, 0, AContainer::CO_INCW );
    help_tv->setLineWrap( true );
    help_tv->showFrame( false );
    help_tv->maximizeYLines( 10, help_tv->getWidth() );
    ac0->readLimits();
    help_tv->show();
    help_tv->setAcceptFocus( false );

    TextView::ColorDef tv_cd = help_tv->getColors();
    tv_cd.setBackground( 0 );
    tv_cd.setTextColor( 1 );
    help_tv->setColors( tv_cd );

    AContainer *ac1_1 = ac0->add( new AContainer( win, 2, 1 ), 0, 1 );
    ac1_1->setMinSpace( 5 );
    ac1_1->setMaxSpace( 5 );
    ac1_1->setBorderWidth( 0 );
    ac1_1->addWidget( new Text( aguix, 0, 0, catalog.getLocale( 1181 ), 1, 0 ),
                      0, 0, AContainer::CO_FIX );
    CycleButton *comparemode_cb = ac1_1->addWidget( new CycleButton( aguix, 0, 0, 10, 0 ),
                                                    1, 0, AContainer::CO_INCW );
    comparemode_cb->addOption( catalog.getLocale( 1182 ) );
    comparemode_cb->addOption( catalog.getLocale( 1183 ) );
    comparemode_cb->addOption( catalog.getLocale( 1184 ) );
    comparemode_cb->resize( comparemode_cb->getMaxSize(),
                           comparemode_cb->getHeight() );

    AContainer *ac1_2 = ac0->add( new AContainer( win, 2, 1 ), 0, 2 );
    ac1_2->setMinSpace( 5 );
    ac1_2->setMaxSpace( 5 );
    ac1_2->setBorderWidth( 0 );
    ac1_2->addWidget( new Text( aguix, 0, 0, catalog.getLocale( 1185 ), 1, 0 ),
                      0, 0, AContainer::CO_FIX );
    CycleButton *resultmode_cb = ac1_2->addWidget( new CycleButton( aguix, 0, 0, 10, 0 ),
                                                   1, 0, AContainer::CO_INCW );
    resultmode_cb->addOption( catalog.getLocale( 1186 ) );
    resultmode_cb->addOption( catalog.getLocale( 1187 ) );
    resultmode_cb->resize( resultmode_cb->getMaxSize(),
                           resultmode_cb->getHeight() );
    ac0->readLimits();
    
    AContainer *ac1_3 = ac0->add( new AContainer( win, 2, 1 ), 0, 3 );
    ac1_3->setMinSpace( 5 );
    ac1_3->setMaxSpace( -1 );
    ac1_3->setBorderWidth( 0 );
    Button *okb = ac1_3->addWidget( new Button( aguix,
                                                0,
                                                0,
                                                catalog.getLocale( 11 ),
                                                0 ), 0, 0, AContainer::CO_FIX );
    Button *cb = ac1_3->addWidget( new Button( aguix,
                                               0,
                                               0,
                                               catalog.getLocale( 8 ),
                                               0 ), 1, 0, AContainer::CO_FIX );

    win->setDoTabCycling( true );
    win->contMaximize( true );
    win->show();

    for ( ; endmode == -1; ) {
        msg = aguix->WaitMessage( win );
        if ( msg != NULL ) {
            switch ( msg->type ) {
                case AG_CLOSEWINDOW:
                    if ( msg->closewindow.window == win->getWindow() ) endmode = 1;
                    break;
                case AG_BUTTONCLICKED:
                    if ( msg->button.button == okb ) endmode = 0;
                    else if ( msg->button.button == cb ) endmode = 1;
                default:
                    break;
            }
            aguix->ReplyMessage( msg );
        }
    }

    if ( endmode == 0 ) {
        // ok

        switch ( comparemode_cb->getSelectedOption() ) {
            case 1:
                m_compare_mode = COMPARE_SIZE_AND_MDATE;
                break;
            case 2:
                m_compare_mode = COMPARE_CONTENT;
                break;
            default:
                m_compare_mode = COMPARE_SIZE;
                break;
        }

        switch ( resultmode_cb->getSelectedOption() ) {
            case 1:
                m_result_mode = SHOW_AS_VIRTUAL_DIR;
                break;
            default:
                m_result_mode = CHANGE_SELECTION_STATE;
                break;
        }
    }

    delete win;

    return endmode;
}
